/* -*- tab-width: 4; indent-tabs-mode: nil -*- */
/* vim: set ts=4 sts=4 sw=4 expandtab number : */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <math.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <ctype.h>
#include <libgen.h>

#include <netdb.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include <sys/ioctl.h>
#include <pt1_ioctl.h>

#include "config.h"
#include "decoder.h"
#include "recpt1.h"
#include "version.h"
#include "mkpath.h"

#include <sys/ipc.h>
#include <sys/msg.h>
#include "pt1_dev.h"
#include "tssplitter_lite.h"
#include "ushare.h"
#include "trace.h"
#include "pt1_common.h"

/* maximum write length at once */
#define SIZE_CHANK 1316

/* globals */
boolean f_exit = FALSE;
struct channel_info_list *channel_list = NULL;
extern struct ushare_t *ut;
thread_data *gp_tdata;

static struct channel_info_list *open_list_file(
       char *type,
       struct channel_info_list *info_list)
{
    char   *buf                                 = NULL;
    char   filename[PATH_MAX];
    char   *home                                = NULL;
    char   *p                                   = NULL;
    struct channel_info      *channel_info      = NULL;
    struct channel_info_list *channel_info_list = NULL;
    FILE   *f                                   = NULL;

    home = getenv("HOME");
    snprintf(filename, PATH_MAX, "%s/.recpt1_%s", home, type);
    f = fopen(filename, "r");
    if(!f) {
        return channel_info_list;
    }
    if(!info_list) {
        channel_info_list = malloc(sizeof(*channel_info_list));
        if(!channel_info_list)
            return NULL;
        channel_info_list->nr_channel = 0;
    } else
        channel_info_list = info_list;
    while(1) {
        buf = NULL;
        buf = malloc(256);
        if(!buf)
            break;
        if(!fgets(buf, 255, f)) {
            free(buf);
            buf = NULL;
            break;
        }
        channel_info = NULL;
        channel_info = malloc(sizeof(*channel_info));
        if(!channel_info)
            break;

        channel_info->sid  = NULL;
        channel_info->tp   = NULL;
        channel_info->name = NULL;

        channel_info->id  = channel_info_list->nr_channel;
        channel_info->sid = buf;

        p = strchr(buf, C_CHAR_COMMA);
        if (p == NULL) {
            /* FILE ERROR */
            free(channel_info);
	        channel_info = NULL;
                continue;
        }
        *p = '\0';
        channel_info->tp = ++p;

        p = strchr(p, C_CHAR_COMMA);
        if (p == NULL) {
            /* FILE ERROR */
            free(channel_info);
            channel_info = NULL;
            continue;
        }
        *p = '\0';
        channel_info->name = ++p;
        p = strchr(p, '\n');
        if (p)
            *p = '\0';

        if(channel_info_list->nr_channel < CHANNEL_MAX) {
            channel_info_list->channel_info[channel_info_list->nr_channel] = channel_info;
            channel_info_list->nr_channel += 1;
        }
    }
    fclose(f);
    return channel_info_list;
}

static void close_list_file(struct channel_info_list *channel_info_list)
{
    int i;
    if (!channel_info_list)
        return;
    for (i=0; i < channel_info_list->nr_channel; i++) {
        free(channel_info_list->channel_info[i]->sid);
        channel_info_list->channel_info[i]->sid = NULL;
        free(channel_info_list->channel_info[i]);
        channel_info_list->channel_info[i] = NULL;
    }
    channel_info_list->nr_channel = 0;
    free(channel_info_list);
    channel_info_list = NULL;
    return;
}

/* ipc message receive */
void *
mq_recv(void *t)
{
    thread_data *tdata = (thread_data *)t;
    pt1_message_buf rbuf;
    static char channel[16];
    static char sid_list[16];
    int  recsec = 0, time_to_add = 0;
    int  current_type;
    splitter *splitter   = tdata->splitter;
    boolean use_splitter = splitter ? TRUE : FALSE;
    boolean use_dlna     = tdata->streamer ? TRUE : FALSE;
    boolean stop_rec     = FALSE;
    ISDB_T_FREQ_CONV_TABLE *table = NULL;

    while(1) {
        if(msgrcv(tdata->msqid, &rbuf, MSGSZ, 1, 0) < 0) {
            return NULL;
        }

        sscanf(rbuf.mtext, "ch=%s t=%d e=%d i=%s", channel, &recsec, &time_to_add, sid_list);

        if ((strcmp(channel, "")) && strcmp(tdata->ch, channel) ||
            (strcmp(sid_list, "")) && strcmp(tdata->sid_list, sid_list)) {
#if 0
            /* re-initialize decoder */
            if(tdata->decoder) {
//                b25_finish(tdata->decoder);
                b25_shutdown(tdata->decoder);
                tdata->decoder = b25_startup(tdata->dopt);
                if(!tdata->decoder) {
                    fprintf(stderr, "Cannot start b25 decoder\n");
                    fprintf(stderr, "Fall back to encrypted recording\n");
                }
            }
#endif
            current_type = tdata->table->type;
            table = searchrecoff(channel);
            if (table == NULL) {
                fprintf(stderr, "Invalid Channel: %s\n", channel);
                goto CHECK_TIME_TO_ADD;
            }
            tdata->table = table;

            /* stop stream */
            ioctl(tdata->tfd, STOP_REC, 0);
            stop_rec = TRUE;

            /* wait for remainder */
            while(tdata->queue->num_used > 0) {
                usleep(10000);
            }
        }
        if (use_splitter &&
            (strcmp(tdata->sid_list, sid_list) ||
             strcmp(tdata->ch, channel))) {
            // $BJ*M}%A%c%s%M%kJQ99;~$K$O(B splitter $B$O6/@)E*$K:F5/F0$5$;$k(B
            pthread_mutex_lock(&tdata->splitter_mutex);
            split_shutdown(splitter);
            splitter = split_startup(sid_list, NULL, NULL);
            if (splitter->sid_list == NULL) {
                fprintf (stderr, "reader_func() splitter RESTART FAILED.\n");
                tdata->splitter = NULL;
                pthread_mutex_unlock(&tdata->splitter_mutex);
                continue;
            }
            if (tdata->streamer) {
                pthread_mutex_lock(&tdata->stream_queue->mutex);
                while(1) {
                     STREAM_QUEUE_T *p_queue = tdata->stream_queue;
                     if (p_queue->num_used == 0)
                         break;
                     free(p_queue->buffer[p_queue->out]->data);
                     p_queue->buffer[p_queue->out]->data = NULL;
                     free(p_queue->buffer[p_queue->out]);
                     p_queue->buffer[p_queue->out] = NULL;
                     p_queue->out++;
                     p_queue->out %= p_queue->size;

                     /* update counters */
                     p_queue->num_avail++;
                     p_queue->num_used--;
                }
                pthread_mutex_unlock(&tdata->stream_queue->mutex);
            }
            tdata->table    = table;
            tdata->splitter = splitter;
            time(&splitter->split_start_time);
            strncpy(tdata->sid_list, sid_list, sizeof(tdata->sid_list));
            pthread_mutex_unlock(&tdata->splitter_mutex);
        }

        if (stop_rec) {
            if (tdata->table->type != current_type) {
            /* re-open device */
                if(close_tuner(tdata) != 0)
                    return NULL;
          
                tune(channel, tdata, NULL);
            } else {
            /* SET_CHANNEL only */
                const FREQUENCY freq = {
                    .frequencyno = tdata->table->set_freq,
                    .slot = tdata->table->add_freq,
                };
                if(ioctl(tdata->tfd, SET_CHANNEL, &freq) < 0) {
                    fprintf(stderr, "Cannot tune to the specified channel\n");
                    tdata->ch[0] = '\0';
                    goto CHECK_TIME_TO_ADD;
                }
                calc_cn(tdata->tfd, tdata->table->type, false);
            }
            /* restart recording */
            if(ioctl(tdata->tfd, START_REC, 0) < 0) {
                fprintf(stderr, "Tuner cannot start recording\n");
                return NULL;
            }
            strncpy(tdata->ch, channel, sizeof(tdata->ch));
        }

CHECK_TIME_TO_ADD:
        if(time_to_add) {
            tdata->recsec += time_to_add;
            fprintf(stderr, "Extended %d sec\n", time_to_add);
        }

        if(recsec) {
            time_t cur_time;
            time(&cur_time);
            if(cur_time - tdata->start_time > recsec) {
                f_exit = TRUE;
            }
            else {
                tdata->recsec = recsec;
                fprintf(stderr, "Total recording time = %d sec\n", recsec);
            }
        }
        if(f_exit)
            return NULL;
    }
}

QUEUE_T *
create_queue(size_t size)
{
    QUEUE_T *p_queue;
    int memsize = sizeof(QUEUE_T) + size * sizeof(BUFSZ*);

    p_queue = (QUEUE_T*)calloc(memsize, sizeof(char));

    if(p_queue != NULL) {
        p_queue->size = size;
        p_queue->num_avail = size;
        p_queue->num_used = 0;
        p_queue->in = 0;
        p_queue->out = 0;
        pthread_mutex_init(&p_queue->mutex, NULL);
        pthread_cond_init(&p_queue->cond_avail, NULL);
        pthread_cond_init(&p_queue->cond_used, NULL);
    }

    return p_queue;
}

STREAM_QUEUE_T *
create_stream_queue(size_t size)
{
    STREAM_QUEUE_T *p_queue;
    int memsize = sizeof(STREAM_QUEUE_T) + size * sizeof(ARIB_STD_B25_BUFFER*);

    p_queue = (STREAM_QUEUE_T*)calloc(memsize, sizeof(char));

    if(p_queue != NULL) {
        p_queue->size = size;
        p_queue->num_avail = size;
        p_queue->num_used = 0;
        p_queue->in = 0;
        p_queue->out = 0;
        pthread_mutex_init(&p_queue->mutex, NULL);
        pthread_cond_init(&p_queue->cond_avail, NULL);
        pthread_cond_init(&p_queue->cond_used, NULL);
    }

    return p_queue;
}

void
destroy_queue(QUEUE_T *p_queue)
{
    if(!p_queue)
        return;

    pthread_mutex_destroy(&p_queue->mutex);
    pthread_cond_destroy(&p_queue->cond_avail);
    pthread_cond_destroy(&p_queue->cond_used);
    free(p_queue);
}

void
destroy_stream_queue(STREAM_QUEUE_T *p_queue)
{
    if(!p_queue)
        return;

    pthread_mutex_destroy(&p_queue->mutex);
    pthread_cond_destroy(&p_queue->cond_avail);
    pthread_cond_destroy(&p_queue->cond_used);
    free(p_queue);
}

/* enqueue data. this function will block if queue is full. */
void
enqueue(QUEUE_T *p_queue, BUFSZ *data)
{
    struct timeval now;
    struct timespec spec;

    gettimeofday(&now, NULL);
    spec.tv_sec = now.tv_sec + 1;
    spec.tv_nsec = now.tv_usec * 1000;

    pthread_mutex_lock(&p_queue->mutex);
    /* entered critical section */

    /* wait while queue is full */
    while(p_queue->num_avail == 0) {
        pthread_cond_timedwait(&p_queue->cond_avail,
                               &p_queue->mutex, &spec);
        if(f_exit) {
            pthread_mutex_unlock(&p_queue->mutex);
            return;
        }
    }

    p_queue->buffer[p_queue->in] = data;

    /* move position marker for input to next position */
    p_queue->in++;
    p_queue->in %= p_queue->size;

    /* update counters */
    p_queue->num_avail--;
    p_queue->num_used++;

    /* leaving critical section */
    pthread_mutex_unlock(&p_queue->mutex);
    pthread_cond_signal(&p_queue->cond_used);
}

 /*
 * stream_func()$B$N;HMQ$9$k(B enqueue() $B$O6u$-$,$J$$>l9g$K$O!"L58BBT$A$O$7$F$O$J$i$J$$!#(B
 * $B6u$-$,$J$$>l9g$K$O!"(Bqueue$B$r=i4|2=$7$F$7$^$$!"C<Kv$X$NAw?.%G!<%?$r;E@Z$jD>$7$7$F$7$^$&J}$,%^%7!#(B
 * $B$3$l$K$h$j!"C<Kv$X$NAw?.%G!<%?$O?tICJ,%9%-%C%W$9$k;v$K$J$k$,!"(B
 * $B%P%C%U%!$OM-8B$G$"$j!"%9%H%j!<%`$G$"$k$N$G:F@8$5$lB3$1$k$3$H$rM%@h$9$k!#(B
 */
void
stream_enqueue(STREAM_QUEUE_T *p_queue, ARIB_STD_B25_BUFFER *data)
{
    struct timeval now;
    struct timespec spec;
    int i;

    gettimeofday(&now, NULL);
    spec.tv_sec = now.tv_sec + 1;
    spec.tv_nsec = now.tv_usec * 1000;

    pthread_mutex_lock(&p_queue->mutex);
    /* entered critical section */

    if (p_queue->num_avail == 0) {
        /* stream queue $B$O0lGU$K$J$C$?$i>C5n$7$F$7$^$&(B */
        for ( i=0; i < p_queue->size; i++ ) {
            if ( p_queue->buffer[i] != NULL ) {
                free(p_queue->buffer[i]->data);
                p_queue->buffer[i]->data = NULL;
                free(p_queue->buffer[i]);
                p_queue->buffer[i] = NULL;
            }
        }
        p_queue->in = 0;
        p_queue->out = 0;
        p_queue->num_used = 0;
        p_queue->num_avail = p_queue->size;
    }

    p_queue->buffer[p_queue->in] = data;

    /* move position marker for input to next position */
    p_queue->in++;
    p_queue->in %= p_queue->size;

    /* update counters */
    p_queue->num_avail--;
    p_queue->num_used++;

    /* leaving critical section */
    pthread_mutex_unlock(&p_queue->mutex);
    pthread_cond_signal(&p_queue->cond_used);
}

/* dequeue data. this function will block if queue is empty. */
BUFSZ *
dequeue(QUEUE_T *p_queue)
{
    struct timeval now;
    struct timespec spec;
    BUFSZ *buffer;

    gettimeofday(&now, NULL);
    spec.tv_sec = now.tv_sec + 1;
    spec.tv_nsec = now.tv_usec * 1000;

    pthread_mutex_lock(&p_queue->mutex);
    /* entered the critical section*/

    /* wait while queue is empty */
    while(p_queue->num_used == 0) {
        pthread_cond_timedwait(&p_queue->cond_used,
                               &p_queue->mutex, &spec);
        if(f_exit) {
            pthread_mutex_unlock(&p_queue->mutex);
            return NULL;
        }
    }

    /* take buffer address */
    buffer = p_queue->buffer[p_queue->out];

    /* move position marker for output to next position */
    p_queue->out++;
    p_queue->out %= p_queue->size;

    /* update counters */
    p_queue->num_avail++;
    p_queue->num_used--;

    /* leaving the critical section */
    pthread_mutex_unlock(&p_queue->mutex);
    pthread_cond_signal(&p_queue->cond_avail);

    return buffer;
}

ARIB_STD_B25_BUFFER *
stream_dequeue(STREAM_QUEUE_T *p_queue)
{
    struct timeval now;
    struct timespec spec;
    ARIB_STD_B25_BUFFER *buffer;

    gettimeofday(&now, NULL);
    spec.tv_sec = now.tv_sec + 1;
    spec.tv_nsec = now.tv_usec * 1000;

    pthread_mutex_lock(&p_queue->mutex);
    /* entered the critical section*/

    /* wait while queue is empty */
    while(p_queue->num_used == 0) {
        pthread_cond_timedwait(&p_queue->cond_used,
                               &p_queue->mutex, &spec);
        if(f_exit) {
            pthread_mutex_unlock(&p_queue->mutex);
            return NULL;
        }
    }

    /* take buffer address */
    buffer = p_queue->buffer[p_queue->out];

    /* move position marker for output to next position */
    p_queue->out++;
    p_queue->out %= p_queue->size;

    /* update counters */
    p_queue->num_avail++;
    p_queue->num_used--;

    /* leaving the critical section */
    pthread_mutex_unlock(&p_queue->mutex);
    pthread_cond_signal(&p_queue->cond_avail);

    return buffer;
}

/* this function will be reader thread */
void *
reader_func(void *p)
{
    thread_data *data = (thread_data *)p;
    QUEUE_T *p_queue = data->queue;
    decoder *dec = data->decoder;
    int wfd = data->wfd;
    boolean use_b25 = dec ? TRUE : FALSE;
    boolean use_udp = data->sock_data ? TRUE : FALSE;
    boolean fileless = FALSE;
    boolean use_splitter = data->splitter ? TRUE : FALSE;
    boolean use_dlna = data->streamer ? TRUE : FALSE;
    int sfd = -1;
    pthread_t signal_thread = data->signal_thread;
    struct sockaddr_in *addr = NULL;
    BUFSZ *qbuf;
    ARIB_STD_B25_BUFFER *eqbuf;
    splitbuf_t splitbuf;
    ARIB_STD_B25_BUFFER sbuf, dbuf, buf;
    int code;

    if (use_splitter)
        time(&data->splitter->split_start_time);

    buf.size = 0;
    buf.data = NULL;
    splitbuf.size = 0;
    splitbuf.buffer_length = 0;
    splitbuf.buffer = NULL;

    if(wfd == -1)
        fileless = TRUE;

    if(use_udp) {
        sfd = data->sock_data->sfd;
        addr = &data->sock_data->addr;
    }

    while(1) {
        ssize_t wc = 0;
        int file_err = 0;
        qbuf = dequeue(p_queue);
        /* no entry in the queue */
        if(qbuf == NULL) {
            break;
        }

        sbuf.data = qbuf->buffer;
        sbuf.size = qbuf->size;

        buf = sbuf; /* default */

        if(use_b25) {
            code = b25_decode(dec, &sbuf, &dbuf);
            if(code < 0) {
                fprintf(stderr, "b25_decode failed (code=%d). fall back to encrypted recording.\n", code);
                use_b25 = FALSE;
            }
            else
                buf = dbuf;
        }

        if(use_splitter) {
            pthread_mutex_lock(&data->splitter_mutex);
            splitbuf.size = 0;
            if(splitbuf.buffer_length < buf.size && buf.size > 0) {
                splitbuf.buffer = realloc(splitbuf.buffer, buf.size);
                if(NULL == splitbuf.buffer) {
                    fprintf(stderr, "splitbuf.buffer realloc failed\n");
                    use_splitter = FALSE;
                    goto fin;
                }
                splitbuf.buffer_length = buf.size;
            }

            while(buf.size) {
                /* $BJ,N%BP>](BPID$B$NCj=P(B */
                if(data->splitter->split_select_finish != TSS_SUCCESS) {
                    data->splitter->split_select_finish = split_select(data->splitter, &buf);
                    if(data->splitter->split_select_finish == TSS_NULL) {
                        /* malloc$B%(%i!<H/@8(B */
                        fprintf(stderr, "split_select malloc failed\n");
                        use_splitter = FALSE;
                        goto fin;
                    }
                    else if(data->splitter->split_select_finish != TSS_SUCCESS) {
                        /* $BJ,N%BP>](BPID$B$,40A4$KCj=P$G$-$k$^$G=PNO$7$J$$(B
                         * 1$BICDxEYM>M5$r8+$k$H$$$$$+$b(B
                         */
                        time_t cur_time;
                        time(&cur_time);
                        if(cur_time - data->splitter->split_start_time > 4) {
                            fprintf(stderr, "split_select cur_time out.\n");
                            use_splitter = FALSE;
                            goto fin;
                        }
                        break;
                    }
                }
                /* $BJ,N%BP>]0J30$r$U$k$$Mn$H$9(B */
                code = split_ts(data->splitter, &buf, &splitbuf);
                if(code != TSS_SUCCESS) {
                    fprintf(stderr, "split_ts failed\n");
                    break;
                }

                break;
            } /* while */

            buf.size = splitbuf.size;
            buf.data = splitbuf.buffer;
        fin:
            pthread_mutex_unlock(&data->splitter_mutex);
        } /* if */

        /*
         * 2. reader_func$B2~B$E@(B
         *   2.1 tdata->p_queue $B$+$i(B dequeue() $B$7$F%9%H%j!<%`$J%G!<%?$r<hF@$9$k(B
         *      2.1.1 dequeue()$B$O(B tdata->p_queue->mutex $B$r(B lock/unlock $B$7$FFI$_9~$_;~$NF1;~99?7$rKI;_$7$F$$$k(B
         *   2.2 tdata->stream_queue $B$K(B enqueue() $B$r<B9T$7$F(B http_stream $B$NBg85$H$9$k%G!<%?%P%C%U%!$N%3%T!<$r<B;\(B
         *      2.2.1 http_stream$B$N%3%T!<85$H$9$k$?$a$N%P%C%U%!$r(Balloc$B$9$k(B
         *        2.2.2 2.2.1$B$G(Balloc$B$7$?NN0h$K(B rader_func $B$,(B dequeue $B$7$?%9%H%j!<%`$J%G!<%?$r(Bmemcpy()$B$9$k(B
         *      2.2.3 tdata->stream_queue $B$K(B 2.2.1 $B$N%]%$%s%?$r(B enqueue() $B$9$k(B
         *        2.2.3.1 enqueue() $B$O(B tdata->stream_queue->mutex $B$r(B lock/unlock $B$7$F=q$-9~$_;~$NF1;~99?7$rKI;_$7$F$$$k(B
         */
        /*
         * DLNA $B$G$NJ*M}%A%c%s%M%k!&(BSID$BJQ99$N<BAu<B83(B
         */
        if ( use_dlna && buf.size > 0 ) {
            do {
                eqbuf = malloc(sizeof(ARIB_STD_B25_BUFFER));
                if ( eqbuf == NULL ) {
                    fprintf (stderr, "Cannot malloc eqbuf memory. streaming abort.\n");
                    use_dlna = FALSE;
                    break;
                }
                eqbuf->data = malloc(buf.size);
                if ( eqbuf->data == NULL ) {
                    fprintf (stderr, "Cannot malloc eqbuf memory. streaming abort.\n");
                    use_dlna = FALSE;
                    break;
                }
                eqbuf->size = buf.size;
                memcpy(eqbuf->data, buf.data, buf.size);
                // $B$3$A$i$b0n$l$?$i>C5n$7$F$7$^$&(B stream_enqueue() $B$r;HMQ(B
                stream_enqueue(data->stream_queue, eqbuf); 
            } while(0);
        }

        if(!fileless) {
            /* write data to output file */
            int size_remain = buf.size;
            int offset = 0;

            while(size_remain > 0) {
                int ws = size_remain < SIZE_CHANK ? size_remain : SIZE_CHANK;

                wc = write(wfd, buf.data + offset, ws);
                if(wc < 0) {
                    perror("write");
                    file_err = 1;
                    pthread_kill(signal_thread,
                                 errno == EPIPE ? SIGPIPE : SIGUSR2);
                    break;
                }
                size_remain -= wc;
                offset += wc;
            }
        }

        if(use_udp && sfd != -1) {
            /* write data to socket */
            int size_remain = buf.size;
            int offset = 0;
            while(size_remain > 0) {
                int ws = size_remain < SIZE_CHANK ? size_remain : SIZE_CHANK;
                wc = write(sfd, buf.data + offset, ws);
                if(wc < 0) {
                    if(errno == EPIPE)
                        pthread_kill(signal_thread, SIGPIPE);
                    break;
                }
                size_remain -= wc;
                offset += wc;
            }
        }

        free(qbuf);
        qbuf = NULL;

        /* normal exit */
        if((f_exit && !p_queue->num_used) || file_err) {

            buf = sbuf; /* default */

            if(use_b25) {
                code = b25_finish(dec, &sbuf, &dbuf);
                if(code < 0)
                    fprintf(stderr, "b25_finish failed\n");
                else
                    buf = dbuf;
            }

            if(use_splitter) {
                /* $BJ,N%BP>]0J30$r$U$k$$Mn$H$9(B */
                code = split_ts(data->splitter, &buf, &splitbuf);
                if(code != TSS_SUCCESS) {
                    break;
                }

                buf.data = splitbuf.buffer;
                buf.size = splitbuf.size;
            }

            if(!fileless && !file_err) {
                wc = write(wfd, buf.data, buf.size);
                if(wc < 0) {
                    perror("write");
                    file_err = 1;
                    pthread_kill(signal_thread,
                                 errno == EPIPE ? SIGPIPE : SIGUSR2);
                }
            }

            if(use_udp && sfd != -1) {
                wc = write(sfd, buf.data, buf.size);
                if(wc < 0) {
                    if(errno == EPIPE)
                        pthread_kill(signal_thread, SIGPIPE);
                }
            }
            if(use_splitter) {
                free(splitbuf.buffer);
                splitbuf.buffer = NULL;
                splitbuf.buffer_length = 0;
            }

            break;
        }
    }

    time_t cur_time;
    time(&cur_time);
    fprintf(stderr, "Recorded %dsec\n",
            (int)(cur_time - data->start_time));

    return NULL;
}

/*
 * 3. stream_func() $B$O(B reader_func() $B$+$i%9%H%j!<%`8~$1$K%3%T!<$5$l$?%G!<%?$r!"%9%H%j!<%`%;%C%7%g%sKh$N(Bqueue$B$K%3%T!<$9$k(B
 *   3.1 tdata->stream_queue $B$+$i(B dequeue $B$9$k(B
 *     3.1.1 tdata->stream_queue->mutex $B$NFI$_9~$_;~$N(B lock/unlokc $B$,H/@8$9$k(B
 *   3.2 tdata->streamer->mutex $B$r(B lock
 *   3.3 $B0J2<$r(B tdata->streamer->stream_nr $B$N?t$@$1%k!<%W(B
 *     3.3.1 tdata->streamer->stream_session[N]->is_valid $B$,M-8z$+3NG'(B
 *     3.3.2 tdata->streamer->stream_session[N]->p_queue $B$X$N%3%T!<MQ%P%C%U%!$N(Balloc
 *       3.3.3 3.1$B$G(B dequeue $B$7$?%P%C%U%!$r(B3.3.2$B$G(B alloc $B$7$?%P%C%U%!$X(B memcpy()
 *     3.3.4 tdata->streamer->stream_session[N]->p_queue $B$X(B enqueue()
 *       3.3.4.1 tdata->streamer->stream_session[N]->p_queue->mutex $B$N(B lock/unlock $B$,H/@8(B
 *   3.4 tdata->streamer->mutex $B$r(B unlock
 * stream_func()$B$N(B lock $B$9$k$b$N$H=g=x(B
 * #1. tdata->stream_queue->mutex $B$N(Block/unlock
 * #2. tdata->streamer->mutex $B$r(B lock
 * #2.1 tdata->streamer->stream_session[N]->p_queue->mutex $B$N(B lock/unlock
 * #3. tdata->streamer->mutex $B$N(B unlock
 * $B>e5-$K4X$7$F!"(Block/unlock$B$,I,MW$JItJ,$H!"NN0h3NJ]$H%3%T!<$NCY$a$N=hM}$H$G(B
 * $B@Z$jJ,$1$i$l$kItJ,$K$D$$$F$O@Z$jJ,$1$F$7$^$C$?$[$&$,$$$$$+$b!#(B
 * $B%/%j%F%#%+%k%;%/%7%g%s$O!"%]%$%s%?A`:n$@$1$H$9$k$Y$-!#(B
 */
void *
stream_func(void *p)
{
    thread_data *data = (thread_data *)p;
    STREAM_QUEUE_T *p_queue = data->stream_queue;
    ARIB_STD_B25_BUFFER *qbuf = NULL;
    ARIB_STD_B25_BUFFER *buf;
    int i;
    clock_gettime(CLOCK_REALTIME, &data->streamer->start);
    data->streamer->total_byte = 0;
    //fprintf (stderr, "stream_func(): start.\n");

    while(1) {
        if(f_exit)
            break;
        // 3.1 tdata->stream_queue $B$+$i(B dequeue $B$9$k(B
        // dequeue $B$7$?%G!<%?$O(B ARIB_STD_B25_BUFFER
        qbuf = stream_dequeue(p_queue);
        /* no entry in the queue */
        if(qbuf == NULL) {
            continue;
        }
        data->streamer->total_byte += qbuf->size;
        // $B%/%j%F%#%+%k%;%/%7%g%sD9$$$N$J$s$H$+$7$?$$$J$!!D(B
        // ToDo: memcpy $B$H$+%/%j%F%#%+%k%;%/%7%g%s$N30$K=P$9(B
        // 3.2 tdata->streamer->mutex $B$r(B lock
        pthread_mutex_lock(&data->streamer->mutex);
        // 3.3 $B0J2<$r(B tdata->streamer->stream_nr $B$N?t$@$1%k!<%W(B
        for ( i=0; i < data->streamer->stream_nr; i++ ) {
            // 3.3.1 tdata->streamer->stream_session[N]->is_valid $B$,M-8z$+3NG'(B
            if ( data->streamer->stream_session[i] != NULL ) {
                if ( data->streamer->stream_session[i]->is_valid ) {
                    // 3.3.2 tdata->streamer->stream_session[N]->p_queue $B$X$N%3%T!<MQ%P%C%U%!$N(Balloc
                    buf = malloc(sizeof(ARIB_STD_B25_BUFFER));
                    if ( buf == NULL ) {
                        pthread_mutex_unlock(&data->streamer->mutex);
                        log_error ("stream_func(): alloc NULL pointer. streaming abort.\n");
                        return NULL;
                    }
                    buf->data = NULL;
                    buf->data = malloc(qbuf->size);
                    if ( buf->data == NULL ) {
                        log_error ("Cannot malloc buf memory. streaming session_id[%d] abort.\n", i);
                        pthread_mutex_unlock(&data->streamer->mutex);
                        return NULL;
                    }
                    // 3.3.3 3.1$B$G(B dequeue $B$7$?%P%C%U%!$r(B3.3.2$B$G(B alloc $B$7$?%P%C%U%!$X(B memcpy()
                    memcpy(buf->data, qbuf->data, qbuf->size);
                    buf->size = qbuf->size;
                    // 3.3.4 tdata->streamer->stream_session[N]->p_queue $B$X(B enqueue()
                    stream_enqueue(data->streamer->stream_session[i]->p_queue, buf);
                }
            }
        }
        // 3.4 tdata->streamer->mutex $B$r(B unlock
        pthread_mutex_unlock(&data->streamer->mutex);
        free(qbuf->data);
        free(qbuf);
    }
    return NULL;
}

void
show_usage(char *cmd)
{
#ifdef HAVE_LIBARIB25
    fprintf(stderr, "Usage: \n%s [--b25 [--round N] [--strip] [--EMM]] [--udp [--addr hostname --port portnumber]] [--device devicefile] [--lnb voltage] [--sid SID1,SID2] [--es filename_suffix] [--start_time YYYYMMDDHHMISS] [--dlna] channel rectime destfile\n", cmd);
#else
    fprintf(stderr, "Usage: \n%s [--strip] [--EMM]] [--udp [--addr hostname --port portnumber]] [--device devicefile] [--lnb voltage] [--sid SID1,SID2] [--es filename_suffix] [--start_time YYYYMMDDHHMISS] [--dlna] channel rectime destfile\n", cmd);
#endif
    fprintf(stderr, "\n");
    fprintf(stderr, "Remarks:\n");
    fprintf(stderr, "if rectime  is '-', records indefinitely.\n");
    fprintf(stderr, "if destfile is '-', stdout is used for output.\n");
}

void
show_options(void)
{
    fprintf(stderr, "Options:\n");
#ifdef HAVE_LIBARIB25
    fprintf(stderr, "--b25:               Decrypt using BCAS card\n");
    fprintf(stderr, "  --round N:         Specify round number\n");
    fprintf(stderr, "  --strip:           Strip null stream\n");
    fprintf(stderr, "  --EMM:             Instruct EMM operation\n");
#endif
    fprintf(stderr, "--udp:               Turn on udp broadcasting\n");
    fprintf(stderr, "  --addr hostname:   Hostname or address to connect\n");
    fprintf(stderr, "  --port portnumber: Port number to connect\n");
    fprintf(stderr, "--device devicefile: Specify devicefile to use\n");
    fprintf(stderr, "--lnb voltage:       Specify LNB voltage (0, 11, 15)\n");
    fprintf(stderr, "--sid SID1,SID2,...: Specify SID number in CSV format (101,102,...)\n");
    fprintf(stderr, "  --es filename:     Specify ES out filename prefix\n");
    fprintf(stderr, "  --start_time YYYYMMDDHHMISS: Specify record start datetime\n");
    fprintf(stderr, "--dlna:              Turn on DLNA DMS(Digital Media Server)\n");
    fprintf(stderr, "--help:              Show this help\n");
    fprintf(stderr, "--version:           Show version\n");
    fprintf(stderr, "--list:              Show channel list\n");
}

void
cleanup(thread_data *tdata)
{
    boolean use_dlna = tdata->streamer ? TRUE : FALSE;
    /* stop recording */
    ioctl(tdata->tfd, STOP_REC, 0);

    /* xxx need mutex? */
    f_exit = TRUE;
    if ( use_dlna ) {
        UPnPBreak(0);
    }

    pthread_cond_signal(&tdata->queue->cond_avail);
    pthread_cond_signal(&tdata->queue->cond_used);
}

/* will be signal handler thread */
void *
process_signals(void *data)
{
    sigset_t waitset;
    int sig;
    thread_data *tdata = (thread_data *)data;

    sigemptyset(&waitset);
    sigaddset(&waitset, SIGPIPE);
    sigaddset(&waitset, SIGINT);
    sigaddset(&waitset, SIGTERM);
    sigaddset(&waitset, SIGUSR1);
    sigaddset(&waitset, SIGUSR2);

    sigwait(&waitset, &sig);

    switch(sig) {
    case SIGPIPE:
        fprintf(stderr, "\nSIGPIPE received. cleaning up...\n");
        cleanup(tdata);
        break;
    case SIGINT:
        fprintf(stderr, "\nSIGINT received. cleaning up...\n");
        cleanup(tdata);
        break;
    case SIGTERM:
        fprintf(stderr, "\nSIGTERM received. cleaning up...\n");
        cleanup(tdata);
        break;
    case SIGUSR1: /* normal exit*/
        cleanup(tdata);
        break;
    case SIGUSR2: /* error */
        fprintf(stderr, "Detected an error. cleaning up...\n");
        cleanup(tdata);
        break;
    }

    return NULL; /* dummy */
}

void
init_signal_handlers(pthread_t *signal_thread, thread_data *tdata)
{
    sigset_t blockset;

    sigemptyset(&blockset);
    sigaddset(&blockset, SIGPIPE);
    sigaddset(&blockset, SIGINT);
    sigaddset(&blockset, SIGTERM);
    sigaddset(&blockset, SIGUSR1);
    sigaddset(&blockset, SIGUSR2);

    if(pthread_sigmask(SIG_BLOCK, &blockset, NULL))
        fprintf(stderr, "pthread_sigmask() failed.\n");

    pthread_create(signal_thread, NULL, process_signals, tdata);
}

int
parse_time(char *rectimestr, thread_data *tdata)
{
    /* indefinite */
    if(!strcmp("-", rectimestr)) {
        tdata->indefinite = TRUE;
        tdata->recsec = -1;
    }
    /* colon */
    else if(strchr(rectimestr, ':')) {
        int n1, n2, n3;
        if(sscanf(rectimestr, "%d:%d:%d", &n1, &n2, &n3) == 3)
            tdata->recsec = n1 * 3600 + n2 * 60 + n3;
        else if(sscanf(rectimestr, "%d:%d", &n1, &n2) == 2)
            tdata->recsec = n1 * 3600 + n2 * 60;
    }
    /* HMS */
    else {
        char *tmpstr;
        char *p1, *p2;

        tmpstr = strdup(rectimestr);
        p1 = tmpstr;
        while(*p1 && !isdigit(*p1))
            p1++;

        /* hour */
        if((p2 = strchr(p1, 'H')) || (p2 = strchr(p1, 'h'))) {
            *p2 = '\0';
            tdata->recsec += atoi(p1) * 3600;
            p1 = p2 + 1;
            while(*p1 && !isdigit(*p1))
                p1++;
        }

        /* minute */
        if((p2 = strchr(p1, 'M')) || (p2 = strchr(p1, 'm'))) {
            *p2 = '\0';
            tdata->recsec += atoi(p1) * 60;
            p1 = p2 + 1;
            while(*p1 && !isdigit(*p1))
                p1++;
        }

        /* second */
        tdata->recsec += atoi(p1);

        free(tmpstr);
    }

    return 0; /* success */
}


int
main(int argc, char **argv)
{
    time_t cur_time;
    pthread_t signal_thread;
    pthread_t reader_thread;
    pthread_t stream_thread;
    pthread_t ipc_thread;
    pthread_t dlna_thread;
    QUEUE_T *p_queue = create_queue(MAX_QUEUE);
    STREAM_QUEUE_T *stream_queue = create_stream_queue(MAX_QUEUE);
    BUFSZ   *bufptr;
    decoder *dec = NULL;
    splitter *splitter = NULL;
    static thread_data tdata;
    gp_tdata = &tdata;
    decoder_options dopt = {
        4,  /* round */
        0,  /* strip */
        0   /* emm */
    };
    tdata.dopt = &dopt;
    tdata.lnb = 0;

    int result;
    int option_index;
    struct option long_options[] = {
#ifdef HAVE_LIBARIB25
        { "b25",       0, NULL, 'b'},
        { "B25",       0, NULL, 'b'},
        { "round",     1, NULL, 'r'},
        { "strip",     0, NULL, 's'},
        { "emm",       0, NULL, 'm'},
        { "EMM",       0, NULL, 'm'},
#endif
        { "LNB",       1, NULL, 'n'},
        { "lnb",       1, NULL, 'n'},
        { "udp",       0, NULL, 'u'},
        { "addr",      1, NULL, 'a'},
        { "port",      1, NULL, 'p'},
        { "device",    1, NULL, 'd'},
        { "help",      0, NULL, 'h'},
        { "version",   0, NULL, 'v'},
        { "list",      0, NULL, 'l'},
        { "sid",       1, NULL, 'i'},
        { "SID",       1, NULL, 'i'},
        { "es",       1, NULL, 'e'},
        { "ES",       1, NULL, 'e'},
        { "start_time",       1, NULL, 'y'},
        { "dlna",       0, NULL, 'c'},
        { "DLNA",       0, NULL, 'c'},
        {0, 0, NULL, 0} /* terminate */
    };

    boolean use_b25 = FALSE;
    boolean use_udp = FALSE;
    boolean fileless = FALSE;
    boolean use_stdout = FALSE;
    boolean use_splitter = FALSE;
    boolean use_esout = FALSE;
    boolean use_dlna = FALSE;
    char *host_to = NULL;
    int port_to = 1234;
    sock_data *sockdata = NULL;
    char *device = NULL;
    int val;
    char *voltage[] = {"0V", "11V", "15V"};
    char *sid_list = NULL;
    char *es_name_prefix = NULL;
    char *start_time = NULL;

    while((result = getopt_long(argc, argv, "br:smn:ua:p:d:hvli:y:c",
                                long_options, &option_index)) != -1) {
        switch(result) {
        case 'b':
            use_b25 = TRUE;
            fprintf(stderr, "using B25...\n");
            break;
        case 's':
            dopt.strip = TRUE;
            fprintf(stderr, "enable B25 strip\n");
            break;
        case 'm':
            dopt.emm = TRUE;
            fprintf(stderr, "enable B25 emm processing\n");
            break;
        case 'u':
            use_udp = TRUE;
            host_to = "localhost";
            fprintf(stderr, "enable UDP broadcasting\n");
            break;
        case 'h':
            fprintf(stderr, "\n");
            show_usage(argv[0]);
            fprintf(stderr, "\n");
            show_options();
            fprintf(stderr, "\n");
            show_channels();
            fprintf(stderr, "\n");
            exit(0);
            break;
        case 'v':
            fprintf(stderr, "%s %s\n", argv[0], version);
            fprintf(stderr, "recorder command for PT1/2/3 digital tuner.\n");
            exit(0);
            break;
        case 'l':
            show_channels();
            exit(0);
            break;
        /* following options require argument */
        case 'n':
            val = atoi(optarg);
            switch(val) {
            case 11:
                tdata.lnb = 1;
                break;
            case 15:
                tdata.lnb = 2;
                break;
            default:
                tdata.lnb = 0;
                break;
            }
            fprintf(stderr, "LNB = %s\n", voltage[tdata.lnb]);
            break;
        case 'r':
            dopt.round = atoi(optarg);
            fprintf(stderr, "set round %d\n", dopt.round);
            break;
        case 'a':
            use_udp = TRUE;
            host_to = optarg;
            fprintf(stderr, "UDP destination address: %s\n", host_to);
            break;
        case 'p':
            port_to = atoi(optarg);
            fprintf(stderr, "UDP port: %d\n", port_to);
            break;
        case 'd':
            device = optarg;
            fprintf(stderr, "using device: %s\n", device);
            break;
        case 'i':
            use_splitter = TRUE;
            sid_list = optarg;
            break;
        case 'e':
            use_esout = TRUE;
            es_name_prefix = optarg;
            break;
        case 'y':
            start_time = optarg;
            break;
        case 'c':
            use_dlna = TRUE;
            fprintf(stderr, "enable DLNA DMS(Digital Media Server)\n");
            break;
        }
    }

    if(argc - optind < 3) {
        if(argc - optind == 2 &&
              (use_udp|use_dlna|use_esout) ) {
            if ( use_udp ) {
                fprintf(stderr, "Fileless UDP broadcasting\n");
            }
            else if ( use_dlna ) {
                fprintf(stderr, "Fileless DLNA broadcasting\n");
            }
            else if ( use_esout ) {
                fprintf(stderr, "Fileless ES out\n");
            }
            else {
                fprintf(stderr, "Fileless...not found...\n");
            }
            fileless = TRUE;
            tdata.wfd = -1;
        }
        else {
            fprintf(stderr, "Arguments are necessary!\n");
            fprintf(stderr, "Try '%s --help' for more information.\n", argv[0]);
            return 1;
        }
    }

    fprintf(stderr, "pid = %d\n", getpid());

    /* tune */
    if(tune(argv[optind], &tdata, device) != 0)
        return 1;

    /* set recsec */
    if(parse_time(argv[optind + 1], &tdata) != 0)
        return 1;

    /* open output file */
    char *destfile = argv[optind + 2];
    if(destfile && !strcmp("-", destfile)) {
        use_stdout = TRUE;
        tdata.wfd = 1; /* stdout */
    }
    else {
        if(!fileless) {
            int status;
            char *path = strdup(argv[optind + 2]);
            char *dir = dirname(path);
            status = mkpath(dir, 0777);
            if(status == -1)
                perror("mkpath");
            free(path);

            tdata.wfd = open(argv[optind + 2], (O_RDWR | O_CREAT | O_TRUNC), 0666);
            if(tdata.wfd < 0) {
                fprintf(stderr, "Cannot open output file: %s\n",
                        argv[optind + 2]);
                return 1;
            }
        }
    }

    /* initialize decoder */
    if(use_b25) {
        dec = b25_startup(&dopt);
        if(!dec) {
            fprintf(stderr, "Cannot start b25 decoder\n");
            fprintf(stderr, "Fall back to encrypted recording\n");
            use_b25 = FALSE;
        }
    }
    /* initialize splitter */
    if(use_splitter) {
        splitter = split_startup(sid_list, es_name_prefix, start_time);
        if(splitter->sid_list == NULL) {
            fprintf(stderr, "Cannot start TS splitter\n");
            return 1;
        }
        strncpy(tdata.sid_list, sid_list, sizeof(tdata.sid_list));
    }

    /* initialize DLNA */
    if(use_dlna) {
        do {
            if(use_splitter && tdata.table->type == CHTYPE_GROUND)
                channel_list = open_list_file("ISDB", NULL);
            else {
                channel_list = open_list_file("BS", NULL);
                channel_list = open_list_file("CS", channel_list);
            }
            if(!channel_list) {
                fprintf(stderr, "channel list file does not found.\n"
                                "please create channel list file first.\n"
                                "usage:\n"
                                "$ channelscan_pt1.sh\n");
                use_dlna = FALSE;
                break;
            }
            tdata.stream_queue = stream_queue;
            tdata.streamer = malloc(sizeof(streamer));
            if ( tdata.streamer == NULL ) {
                use_dlna = FALSE;
                break;
            }
            tdata.streamer->stream_nr = 0;
            tdata.streamer->stream_session[0] = NULL;
            pthread_mutex_init(&tdata.streamer->mutex, NULL);
            pthread_mutex_init(&tdata.splitter_mutex, NULL);

            pthread_create(&stream_thread, NULL, stream_func, &tdata);
            pthread_create(&dlna_thread, NULL, dlna_startup, NULL);
        } while(0);
    }
    else {
        tdata.streamer = NULL;
    }

    /* initialize udp connection */
    if(use_udp) {
      sockdata = calloc(1, sizeof(sock_data));
      struct in_addr ia;
      ia.s_addr = inet_addr(host_to);
      if(ia.s_addr == INADDR_NONE) {
            struct hostent *hoste = gethostbyname(host_to);
            if(!hoste) {
                perror("gethostbyname");
                return 1;
            }
            ia.s_addr = *(in_addr_t*) (hoste->h_addr_list[0]);
        }
        if((sockdata->sfd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
            perror("socket");
            return 1;
        }

        sockdata->addr.sin_family = AF_INET;
        sockdata->addr.sin_port = htons (port_to);
        sockdata->addr.sin_addr.s_addr = ia.s_addr;

        if(connect(sockdata->sfd, (struct sockaddr *)&sockdata->addr,
                   sizeof(sockdata->addr)) < 0) {
            perror("connect");
            return 1;
        }
    }

    /* prepare thread data */
    tdata.queue = p_queue;
    tdata.decoder = dec;
    tdata.splitter = splitter;
    tdata.sock_data = sockdata;

    /* spawn signal handler thread */
    init_signal_handlers(&signal_thread, &tdata);

    /* spawn reader thread */
    tdata.signal_thread = signal_thread;
    pthread_create(&reader_thread, NULL, reader_func, &tdata);

    /* spawn ipc thread */
    key_t key;
    key = (key_t)getpid();

    if ((tdata.msqid = msgget(key, IPC_CREAT | 0666)) < 0) {
        perror("msgget");
    }
    pthread_create(&ipc_thread, NULL, mq_recv, &tdata);

    /* start recording */
    if(ioctl(tdata.tfd, START_REC, 0) < 0) {
        fprintf(stderr, "Tuner cannot start recording\n");
        return 1;
    }

    fprintf(stderr, "Recording...\n");

    time(&tdata.start_time);

    /* read from tuner */
    while(1) {
        if(f_exit)
            break;

        time(&cur_time);
        bufptr = malloc(sizeof(BUFSZ));
        if(!bufptr) {
            f_exit = TRUE;
            break;
        }
        bufptr->size = read(tdata.tfd, bufptr->buffer, MAX_READ_SIZE);
        if(bufptr->size <= 0) {
            if((cur_time - tdata.start_time) >= tdata.recsec && !tdata.indefinite) {
                f_exit = TRUE;
                enqueue(p_queue, NULL);
                break;
            }
            else {
                free(bufptr);
                bufptr = NULL;
                continue;
            }
        }
        enqueue(p_queue, bufptr);

        /* stop recording */
        time(&cur_time);
        if((cur_time - tdata.start_time) >= tdata.recsec && !tdata.indefinite) {
            ioctl(tdata.tfd, STOP_REC, 0);
            /* read remaining data */
            while(1) {
                bufptr = malloc(sizeof(BUFSZ));
                if(!bufptr) {
                    f_exit = TRUE;
                    break;
                }
                bufptr->size = read(tdata.tfd, bufptr->buffer, MAX_READ_SIZE);
                if(bufptr->size <= 0) {
                    f_exit = TRUE;
                    enqueue(p_queue, NULL);
                    break;
                } else {
                    free(bufptr);
                    bufptr = NULL;
                    break;
                }
                enqueue(p_queue, bufptr);
            }
            break;
        }
    }
    //fprintf (stderr, "main() break?.\n");

    /* delete message queue*/
    msgctl(tdata.msqid, IPC_RMID, NULL);

    pthread_kill(signal_thread, SIGUSR1);

    /* wait for threads */
    pthread_join(reader_thread, NULL);
    pthread_join(signal_thread, NULL);
    pthread_join(ipc_thread, NULL);
    if ( use_dlna ) {
        pthread_join(stream_thread, NULL);
        pthread_join(dlna_thread, NULL);
        if ( channel_list )
            close_list_file(channel_list);
    }

    /* close tuner */
    if(close_tuner(&tdata) != 0)
        return 1;

    /* release queue */
    destroy_queue(p_queue);

    /* close output file */
    if(!use_stdout)
        close(tdata.wfd);

    /* free socket data */
    if(use_udp) {
        close(sockdata->sfd);
        free(sockdata);
    }

    /* release decoder */
    if(use_b25) {
        b25_shutdown(dec);
    }
    if(use_splitter) {
        split_shutdown(splitter);
    }

    return 0;
}
