/* Pidgin-Audacious plugin.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#define PURPLE_PLUGINS 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "gtkplugin.h"
#include "util.h"
#include "debug.h"
#include "connection.h"
#include "version.h"
#include "cmds.h"
#include "savedstatuses.h"

#define PIDGINAUD_PLUGIN_ID     "pidgin_audacious"
#define PLUGIN_NAME             "Pidgin-Audacious"
#define OPT_PIDGINAUD           "/plugins/pidgin_audacious"
#define OPT_PROCESS_STATUS      OPT_PIDGINAUD "/process_status"
#define OPT_PROCESS_USERINFO    OPT_PIDGINAUD "/process_userinfo"
#define OPT_SONG_TEMPLATE       OPT_PIDGINAUD "/song_template"
#define OPT_PASTE_TEMPLATE      OPT_PIDGINAUD "/paste_template"
#define OPT_LOG_OUTPUT          OPT_PIDGINAUD "/log_output"
#define OPT_PLAYER              OPT_PIDGINAUD "/player"

#define DEFAULT_SONG_TEMPLATE   "%title - %artist"
#define DEFAULT_PLAYER          "audacious"
#define SONG_TOKEN              "%song"
#define NO_SONG_MESSAGE         "No song being played."

#define DBUS_TYPE_G_STRING_VALUE_HASHTABLE (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE))

#define aud_debug(fmt, ...)	do { if(purple_prefs_get_bool(OPT_LOG_OUTPUT)) purple_debug(PURPLE_DEBUG_INFO, PLUGIN_NAME, "%s():%4d:  " fmt, __FUNCTION__, (int)__LINE__, ## __VA_ARGS__); } while(0);

#define aud_error(fmt, ...)	do { if(purple_prefs_get_bool(OPT_LOG_OUTPUT)) purple_debug(PURPLE_DEBUG_ERROR, PLUGIN_NAME, "%s():%4d:  " fmt, __FUNCTION__, (int)__LINE__, ## __VA_ARGS__); } while(0);

/* xxx move up */
#define TITLE   "%title"
#define ARTIST  "%artist"
#define ALBUM   "%album"
#define GENRE   "%genre"

typedef struct song_tuple {
    gchar *title;
    gchar *artist;
    gchar *album;
    gchar *genre;
} song_tuple;


/* globals */
static GHashTable *seed_status;
static GHashTable *seed_userinfo;
static GHashTable *pushed_status;
static GHashTable *pushed_userinfo;
static DBusGConnection *connection = NULL;
static DBusGProxy *session = NULL;
static PurpleCmdId cmdid_paste_current_song;

/* prototypes */
extern gchar *botch_utf(const gchar *msg, gsize len, gsize *newlen) __attribute__ ((weak));
static void aud_process(gchar *aud_info);
static void track_signal_cb(DBusGProxy *player_proxy, GHashTable *table, gpointer data);
static void status_signal_cb(DBusGProxy *player_proxy, gint status, gpointer data);
static gboolean is_app_playing(void);
static GHashTable *get_song_table(void);
static song_tuple *get_song_tuple(GHashTable *table);
static void signed_on_cb(PurpleConnection *gc, void *data);
static void prefs_cb(const char *name, PurplePrefType type, gconstpointer value, gpointer data);


/* implementation */

static DBusGProxy *
get_dbus_session(void)
{
    GError *error = NULL;
    DBusGProxy *proxy;
    gchar *player = g_strconcat("org.mpris.",
                                purple_prefs_get_string(OPT_PLAYER),
                                NULL);

    connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
    g_clear_error(&error);


	proxy = dbus_g_proxy_new_for_name(connection,
                                      player,
                                      "/Player",
                                      "org.freedesktop.MediaPlayer");

    g_clear_error(&error);
    g_free(player);
    return proxy;
}


static void
connect_dbus_signals()
{
	dbus_g_proxy_add_signal(session,
                            "TrackChange",
                            DBUS_TYPE_G_STRING_VALUE_HASHTABLE,
                            G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(session,
                                "TrackChange",
                                G_CALLBACK(track_signal_cb),
                                NULL, NULL);

	dbus_g_proxy_add_signal(session,
                            "StatusChange",
                            G_TYPE_INT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(session,
                                "StatusChange",
                                G_CALLBACK(status_signal_cb),
                                NULL, NULL);
}

static void
connect_purple_signals(PurplePlugin *plugin)
{
	purple_signal_connect(purple_connections_get_handle(),
                          "signed-on",
                          plugin,
                          PURPLE_CALLBACK(signed_on_cb),
                          NULL);

	purple_signal_connect(purple_savedstatuses_get_handle(),
                          "savedstatus-changed",
                          plugin,
                          PURPLE_CALLBACK(signed_on_cb),
                          NULL);

	purple_prefs_connect_callback(purple_prefs_get_handle(),
                                  OPT_PIDGINAUD,
                                  prefs_cb,
                                  NULL);
}

static GHashTable *
get_song_table(void)
{
    GHashTable *table = NULL;

	if(is_app_playing()) {
		dbus_g_proxy_call(session,
                          "GetMetadata",
                          NULL,
                          G_TYPE_INVALID,
                          DBUS_TYPE_G_STRING_VALUE_HASHTABLE,
                          &table,
                          G_TYPE_INVALID);
	}

    return table;
}

static song_tuple *
get_song_tuple(GHashTable *table)
{
    song_tuple *t = NULL;
	GValue *value;

    if(!table)
        return NULL;

    t = g_new0(song_tuple, 1);

	value = (GValue *) g_hash_table_lookup(table, "title");
	if (value && G_VALUE_HOLDS_STRING(value)) {
		t->title = g_value_dup_string(value);
	}
	value = (GValue *) g_hash_table_lookup(table, "artist");
	if (value != NULL && G_VALUE_HOLDS_STRING(value)) {
		t->artist = g_value_dup_string(value);
	}
	value = (GValue *) g_hash_table_lookup(table, "album");
	if (value != NULL && G_VALUE_HOLDS_STRING(value)) {
		t->album = g_value_dup_string(value);
	}
	value = (GValue *) g_hash_table_lookup(table, "genre");
	if (value != NULL && G_VALUE_HOLDS_STRING(value)) {
		t->genre = g_value_dup_string(value);
	}

    return t;
}

void
free_song_tuple(song_tuple *t)
{
    g_free(t->title);
    g_free(t->artist);
    g_free(t->album);
    g_free(t->genre);

    g_free(t);
}


static gchar *
format_song_info(song_tuple *tuple)
{
    gchar *song_info = NULL, *tmp = NULL;

    if(!tuple)
        return NULL;

	song_info = g_strdup(purple_prefs_get_string(OPT_SONG_TEMPLATE));
    if(!song_info)
        return NULL;

    if(tuple->title && strstr(song_info, TITLE)) {
        tmp = purple_strreplace(song_info, TITLE, tuple->title);
        g_free(song_info);
        song_info = tmp;
    }
    if(tuple->artist && strstr(song_info, ARTIST)) {
        tmp = purple_strreplace(song_info, ARTIST, tuple->artist);
        g_free(song_info);
        song_info = tmp;
    }
    if(tuple->album && strstr(song_info, ALBUM)) {
        tmp = purple_strreplace(song_info, ALBUM, tuple->album);
        g_free(song_info);
        song_info = tmp;
    }
    if(tuple->genre && strstr(song_info, GENRE)) {
        tmp = purple_strreplace(song_info, GENRE, tuple->genre);
        g_free(song_info);
        song_info = tmp;
    }

    if(song_info && !strcmp(song_info, "")) {
        g_free(song_info);
        song_info = NULL;
    }

    return song_info;
}



static void
track_signal_cb(DBusGProxy *player_proxy, GHashTable *table, gpointer data)
{
    gchar *song_info = NULL;
    song_tuple *tuple = get_song_tuple(table);

    /* set current song */
    purple_util_set_current_song(tuple->title ? tuple->title : "",
                                 tuple->artist ? tuple->artist : "",
                                 tuple->album ? tuple->album : "");

    song_info = format_song_info(tuple);

	aud_process(song_info);
    free_song_tuple(tuple);
	g_free(song_info);
}


static void
status_signal_cb(DBusGProxy *player_proxy, gint status, gpointer data)
{
	aud_debug("StatusChange %d\n", status);

	if (status != 0) {
		aud_process(NULL);

        /* clear current song */
        purple_util_set_current_song(NULL, NULL, NULL);

	}
}


static void
aud_process_status(PurpleConnection *gc, gchar *aud_info)
{
    gchar *new = NULL, *key = NULL;
    const gchar *current, *seed, *pushed, *proto;
    PurpleAccount *account;
    PurplePresence *presence;
    PurplePlugin *prpl;
    PurplePluginProtocolInfo *prpl_info;
    PurpleStatus *status;

    account = purple_connection_get_account(gc);
    presence = purple_account_get_presence(account);

    proto = purple_account_get_protocol_id(account);
    prpl = purple_find_prpl(proto);
    g_return_if_fail(prpl != NULL);

    prpl_info = PURPLE_PLUGIN_PROTOCOL_INFO(prpl);
    g_return_if_fail(prpl_info != NULL && prpl_info->set_status != NULL);

    status = purple_presence_get_active_status(presence);
    g_return_if_fail(status != NULL);

    /* generate key for hash table */
    key = g_strdup_printf("%s %s", account->username, account->protocol_id);

    /* retrieve current user status */
    current = purple_status_get_attr_string(status, "message");
    aud_debug("status current = %s\n", current);

    /* invalidate pushded status on auto away etc. */
    if(current == NULL || strlen(current) == 0) {
        g_hash_table_replace(pushed_status, g_strdup(key), g_strdup(""));
        g_free(key);
        return;
    }

    /* pop pushed_status */
    pushed = (gchar *)g_hash_table_lookup(pushed_status, key);

    /* if current status differs from pushed_status or
       contains token, replace hashes with current. */
    if((pushed && g_ascii_strcasecmp(current, pushed)) ||
       strstr(current, SONG_TOKEN)) {
        g_hash_table_replace(seed_status, g_strdup(key), g_strdup(current));
        g_hash_table_replace(pushed_status, g_strdup(key), g_strdup(current));
    }

    /* construct new status message */
    seed = (gchar *)g_hash_table_lookup(seed_status, key);
    if(!seed) {
        g_free(key);
        return;
    }
    aud_debug("status seed = %s\n", seed);

    if(strstr(seed, SONG_TOKEN)) {
        if(aud_info){
            new = purple_strreplace(seed, SONG_TOKEN, aud_info);
        }
        else {
            new = g_strdup(NO_SONG_MESSAGE);
        }
    }

    if(!new) {
        g_free(key);
        return;
    }

    /* set status message only if text has been changed */
    pushed = (gchar *)g_hash_table_lookup(pushed_status, key);
    aud_debug("status pushed = %s\n", pushed);

    if (!pushed || g_ascii_strcasecmp(pushed, new)) {
        g_hash_table_replace(pushed_status, g_strdup(key), g_strdup(new));
        purple_status_set_attr_string(status, "message", new);
        prpl_info->set_status(account, status);
    }
    g_free(key);
    g_free(new);
}


static void
aud_process_userinfo(PurpleConnection *gc, gchar *aud_info)
{
    gchar *new = NULL, *key = NULL;
    const gchar *current, *seed, *pushed, *proto;
    PurpleAccount *account;
    PurplePlugin *prpl;
    PurplePluginProtocolInfo *prpl_info;

    account = purple_connection_get_account(gc);

    proto = purple_account_get_protocol_id(account);
    prpl = purple_find_prpl(proto);
    g_return_if_fail(prpl != NULL);

    prpl_info = PURPLE_PLUGIN_PROTOCOL_INFO(prpl);
    g_return_if_fail(prpl_info != NULL && prpl_info->set_info != NULL);

    /* generate key for hash table*/
    key = g_strdup_printf("%s %s", account->username, account->protocol_id);

    /* retrieve current user info */
    current = purple_account_get_user_info(account); /* from account.xml */
    aud_debug("userinfo current = %s\n", current);

    /* invalidate pushded status on auto away etc. */
    if(current == NULL || strlen(current) == 0) {
        g_hash_table_replace(pushed_userinfo, g_strdup(key), g_strdup(""));
        g_free(key);
        return;
    }

    /* pop pushed_userinfo */
    pushed = g_hash_table_lookup(pushed_userinfo, key);

    /* if current userinfo differs from pushed_userinfo or contains
       token, replace seed with this. */
    if((pushed && g_ascii_strcasecmp(current, pushed)) ||
       strstr(current, SONG_TOKEN)) {
        g_hash_table_replace(seed_userinfo, g_strdup(key),
                             g_strdup(current));
        g_hash_table_replace(pushed_userinfo, g_strdup(key),
                             g_strdup(current));
    }

    /* construct new status message */
    seed = (gchar *)g_hash_table_lookup(seed_userinfo, key);
    if(!seed) {
        g_free(key);
        return;
    }
    aud_debug("userinfo seed = %s\n", seed);

    if(strstr(seed, SONG_TOKEN)) {
        if(aud_info){
            new = purple_strreplace(seed, SONG_TOKEN, aud_info);
        }
        else {
            new = g_strdup(NO_SONG_MESSAGE);
        }
    }

    if(!new) {
        g_free(key);
        return;
    }

    /* set user info only if text has been changed */
    pushed = (gchar *)g_hash_table_lookup(pushed_userinfo, key);
    aud_debug("userinfo pushed = %s\n", pushed);

    if (!pushed || g_ascii_strcasecmp(pushed, new) != 0) {
        g_hash_table_replace(pushed_userinfo, g_strdup(key),
                             g_strdup(new));
        prpl_info->set_info(gc, new);
    }
    g_free(key);
    g_free(new);
}


static void
aud_process(gchar *aud_info)
{
    GList *l;
    PurpleConnection *gc;

    for (l = purple_connections_get_all(); l != NULL; l = l->next) {
        gc = (PurpleConnection *) l->data;

        /* make sure we're connected */
        if (purple_connection_get_state(gc) != PURPLE_CONNECTED) {
            continue;
        }

        if (purple_prefs_get_bool(OPT_PROCESS_USERINFO)) {
            aud_process_userinfo(gc, aud_info);
        }

        if (purple_prefs_get_bool(OPT_PROCESS_STATUS)) {
            aud_process_status(gc, aud_info);
        }
    }

}

static void
removekey(gpointer data)
{
    g_free(data);
}

static void
removeval(gpointer data)
{
    g_free(data);
}

static PurpleCmdRet
paste_current_song(PurpleConversation *conv, const gchar *cmd,
                   gchar **args, gchar **error, void *data)
{
    gchar *song = NULL, *tmp = NULL, *tmp2 = NULL;
    PurpleConversationType type = purple_conversation_get_type(conv);
    size_t dummy;
    const gchar *template = NULL;

    /* audacious isn't playing */
    if(!is_app_playing()) {
        return PURPLE_CMD_RET_OK;
    }

    /* dbus lookup */
    GHashTable *table = get_song_table();
    song_tuple *tuple = get_song_tuple(table);
    tmp = format_song_info(tuple);
    free_song_tuple(tuple);
    g_hash_table_destroy(table);

    template = purple_prefs_get_string(OPT_PASTE_TEMPLATE);

    if(template && strstr(template, SONG_TOKEN)) {
        tmp2 = purple_strreplace(template, SONG_TOKEN, tmp);
        g_free(tmp);
        tmp = NULL;
    }
    else {
        tmp2 = tmp;
    }

    if(tmp2) {
        if(botch_utf) {
            song = (gchar *) botch_utf(tmp2, strlen(tmp2), &dummy);
            g_free(tmp2);
            tmp2 = NULL;
        }
        else
            song = tmp2;
    }

    if(type == PURPLE_CONV_TYPE_CHAT) {
        PurpleConvChat *chat = purple_conversation_get_chat_data(conv);
        if (chat && song)
            purple_conv_chat_send(chat, song);
    }
    else if(type == PURPLE_CONV_TYPE_IM) {
        PurpleConvIm *im = purple_conversation_get_im_data(conv);
        if(im && song)
            purple_conv_im_send(im, song);
    }

    g_free(song);
    return PURPLE_CMD_RET_OK;
}

#define DBUS_TIMEOUT 1000
#define DBUS_TYPE_MPRIS_STATUS (dbus_g_type_get_struct ("GValueArray", G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_INVALID))

#define PLAYING 0
#define PAUSED  1
#define STOPPED 2

static gboolean
is_app_playing(void)
{
	gchar *player_name = g_strconcat("org.mpris.",
                                     purple_prefs_get_string(OPT_PLAYER),
                                     NULL);


    aud_debug("player_name = %s\n", player_name);

    if(!session) {
        session = get_dbus_session();
        connect_dbus_signals();
        aud_debug("session == NULL. new session = %p\n", session);
    }

    /* audacious only supports this form */
    GError *error = NULL;
    int status = STOPPED;

    gboolean result =
        dbus_g_proxy_call_with_timeout(
            session,
            "GetStatus",
            DBUS_TIMEOUT,
            &error,
            G_TYPE_INVALID,
            G_TYPE_INT,
            &status,
            G_TYPE_INVALID);

    aud_debug("GetStatus result = %d\n", result);

    if(status == PLAYING || status == PAUSED)
        return TRUE;
    else
        return FALSE;

    aud_debug("GetStatus status = %d\n", status);


    /* mpris specification 1.0 insists this is the way. no player
     * supports this at this time. --yaz */
#if 0
    GError *error = NULL;
    GValueArray *s;
    gboolean result =
        dbus_g_proxy_call_with_timeout(
            session,
            "GetStatus",
            DBUS_TIMEOUT, //msec
            &error,
            G_TYPE_INVALID,
            DBUS_TYPE_MPRIS_STATUS,
            &s,
            G_TYPE_INVALID);

    aud_debug("GetStatus result = %d\n", result);

    if(result) {
        /* parse GValueArray here */
        GValue *val = g_value_array_get_nth(s, 0);
        int status = g_value_get_int(val);
        g_value_array_free(s);

        aud_debug("status = %d\n", status);

        if(status == STOPPED)
            return FALSE;
        else
            return TRUE;
    }

	return FALSE;
#endif

}

static void
signed_on_cb(PurpleConnection *gc, void *data)
{
    gchar *song_info = NULL;
    GHashTable *table = NULL;
    song_tuple *tuple = NULL;

    aud_debug("called\n");

    if(!is_app_playing())
        return;

    table = get_song_table();
    tuple = get_song_tuple(table);
    g_hash_table_destroy(table);

    if(!tuple) {
        aud_debug("tuple == NULL\n");
        return;
    }

    if(tuple->title)
        aud_debug("title = %s\n", tuple->title);

    /* set current song */
    purple_util_set_current_song(tuple->title ? tuple->title : "",
                                 tuple->artist ? tuple->artist : "",
                                 tuple->album ? tuple->album : "");

    song_info = format_song_info(tuple);
    free_song_tuple(tuple);

    aud_process(song_info);
    g_free(song_info);
}


static void
prefs_cb(const char *name, PurplePrefType type,
                    gconstpointer value, gpointer data)
{
	aud_debug("settings change detected at %s\n", name);
	signed_on_cb(NULL, NULL);
}

static gboolean
load_plugin(PurplePlugin *plugin)
{
    seed_status = g_hash_table_new_full(g_str_hash, g_str_equal,
                                        removekey, removeval);
    seed_userinfo = g_hash_table_new_full(g_str_hash, g_str_equal,
                                          removekey, removeval);

    pushed_status = g_hash_table_new_full(g_str_hash, g_str_equal,
                                          removekey, removeval);
    pushed_userinfo = g_hash_table_new_full(g_str_hash, g_str_equal,
                                            removekey, removeval);


    session = get_dbus_session();

    aud_debug("session = %p\n", session);

    /* connect to mpris signals */
    if(session)
        connect_dbus_signals();

    /* connect to purple signals */
    connect_purple_signals(plugin);

	status_signal_cb(NULL, -1, NULL);


    /* register /song command */
    cmdid_paste_current_song =
        purple_cmd_register("song", "", PURPLE_CMD_P_DEFAULT,
                            PURPLE_CMD_FLAG_IM | PURPLE_CMD_FLAG_CHAT,
                            NULL, paste_current_song,
                            "song: Paste currently plaing song", NULL);

    return TRUE;
}

static gboolean
unload_plugin(PurplePlugin *plugin)
{
    aud_debug("pidgin-audacious unload called\n");

    g_hash_table_destroy(seed_status);
    g_hash_table_destroy(seed_userinfo);

    g_hash_table_destroy(pushed_status);
    g_hash_table_destroy(pushed_userinfo);

    purple_cmd_unregister(cmdid_paste_current_song);

    if(session) {
        g_object_unref(session);
        session = NULL;
    }

    if(connection) {
        dbus_g_connection_unref(connection);
        connection = NULL;
    }

    return TRUE;
}

static PurplePluginPrefFrame *
get_plugin_pref_frame(PurplePlugin *plugin)
{
    PurplePluginPref *pref;
    PurplePluginPrefFrame *frame = purple_plugin_pref_frame_new();

    /* create gtk elements for the plugin preferences */
    pref = purple_plugin_pref_new_with_label(
        "Pidgin-Audacious Configuration");
    purple_plugin_pref_frame_add(frame, pref);

    /* player to connect */
    pref = purple_plugin_pref_new_with_name_and_label(
        OPT_PLAYER,
        "Player to connect");
    purple_plugin_pref_frame_add(frame, pref);

    pref = purple_plugin_pref_new_with_name_and_label(
        OPT_PROCESS_STATUS,
        "Expand " SONG_TOKEN " to song info in the status message");
    purple_plugin_pref_frame_add(frame, pref);

    pref = purple_plugin_pref_new_with_name_and_label(
        OPT_PROCESS_USERINFO,
        "Expand " SONG_TOKEN " to song info in the user info");
    purple_plugin_pref_frame_add(frame, pref);

    /* song template */
    pref = purple_plugin_pref_new_with_name_and_label(
        OPT_SONG_TEMPLATE,
        "%song compiled with");
    purple_plugin_pref_frame_add(frame, pref);

    /* paste template */
    pref = purple_plugin_pref_new_with_name_and_label(
        OPT_PASTE_TEMPLATE,
        "/song command template");
    purple_plugin_pref_frame_add(frame, pref);

    return frame;
}

static PurplePluginUiInfo pref_info =
{
    get_plugin_pref_frame
};

static PurplePluginInfo info =
{
    PURPLE_PLUGIN_MAGIC,
    PURPLE_MAJOR_VERSION,
    PURPLE_MINOR_VERSION,
    PURPLE_PLUGIN_STANDARD,     /**< type   */
    NULL,                       /**< ui_req */
    0,                          /**< flags  */
    NULL,                       /**< deps   */
    PURPLE_PRIORITY_DEFAULT,    /**< priority */
    PIDGINAUD_PLUGIN_ID,        /**< id     */
    "Pidgin-Audacious",         /**< name   */
    "3.0.0d1",                  /**< version */
    "Automatically updates your Pidgin status info with the currently "
    "playing music in Audacious.", /**  summary */
    "Automatically updates your Pidgin status info with the currently "
    "playing music in Audacious.", /**  desc */
    "Yoshiki Yazawa (yaz@honeyplanet.jp)", /**< author */
    "http://www.honeyplanet.jp",   /**< homepage */
    load_plugin,                /**< load   */
    unload_plugin,              /**< unload */
    NULL,                       /**< destroy */
    NULL,                       /**< ui_info */
    NULL,                       /**< extra_info */
    &pref_info,                 /**< pref info */
    NULL
};

static void
init_plugin(PurplePlugin *plugin)
{
    g_type_init();

    /* add plugin preferences */
    purple_prefs_add_none(OPT_PIDGINAUD);
    purple_prefs_add_bool(OPT_PROCESS_STATUS, TRUE);
    purple_prefs_add_bool(OPT_PROCESS_USERINFO, TRUE);
    purple_prefs_add_string(OPT_PLAYER, DEFAULT_PLAYER);
    purple_prefs_add_string(OPT_SONG_TEMPLATE, DEFAULT_SONG_TEMPLATE);
    purple_prefs_add_string(OPT_PASTE_TEMPLATE, SONG_TOKEN);
    purple_prefs_add_bool(OPT_LOG_OUTPUT, TRUE);

}

PURPLE_INIT_PLUGIN(pidgin_audacious, init_plugin, info)
