/*
 * Constants for DV codec
 * Copyright (c) 2002 Fabrice Bellard.
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file dvdata.h
 * Constants for DV codec.
 */

#ifndef FFMPEG_DVDATA_H
#define FFMPEG_DVDATA_H

#include "libavutil/rational.h"
#include "avcodec.h"

/*
 * DVprofile is used to express the differences between various
 * DV flavors. For now it's primarily used for differentiating
 * 525/60 and 625/50, but the plans are to use it for various
 * DV specs as well (e.g. SMPTE314M vs. IEC 61834).
 */
typedef struct DVprofile {
    int              dsf;                 /* value of the dsf in the DV header */
    int              video_stype;         /* stype for VAUX source pack */
    int              frame_size;          /* total size of one frame in bytes */
    int              difseg_size;         /* number of DIF segments per DIF channel */
    int              n_difchan;           /* number of DIF channels per frame */
    int              frame_rate;
    int              frame_rate_base;
    int              ltc_divisor;         /* FPS from the LTS standpoint */
    int              height;              /* picture height in pixels */
    int              width;               /* picture width in pixels */
    AVRational       sar[2];              /* sample aspect ratios for 4:3 and 16:9 */
    const uint16_t  *video_place;         /* positions of all DV macro blocks */
    enum PixelFormat pix_fmt;             /* picture pixel format */
    int              bpm;                 /* blocks per macroblock */
    const uint8_t   *block_sizes;         /* AC block sizes, in bits */
    int              audio_stride;        /* size of audio_shuffle table */
    int              audio_min_samples[3];/* min ammount of audio samples */
                                          /* for 48Khz, 44.1Khz and 32Khz */
    int              audio_samples_dist[5];/* how many samples are supposed to be */
                                         /* in each frame in a 5 frames window */
    const uint8_t  (*audio_shuffle)[9];  /* PCM shuffling table */
} DVprofile;

#define NB_DV_VLC 409

/*
 * There's a catch about the following three tables: the mapping they establish
 * between (run, level) and vlc is not 1-1. So you have to watch out for that
 * when building misc. tables. E.g. (1, 0) can be either 0x7cf or 0x1f82.
 */
static const uint16_t dv_vlc_bits[409] = {
 0x0000, 0x0002, 0x0007, 0x0008, 0x0009, 0x0014, 0x0015, 0x0016,
 0x0017, 0x0030, 0x0031, 0x0032, 0x0033, 0x0068, 0x0069, 0x006a,
 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x00e0, 0x00e1, 0x00e2,
 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, 0x00e8, 0x00e9, 0x00ea,
 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, 0x01e0, 0x01e1, 0x01e2,
 0x01e3, 0x01e4, 0x01e5, 0x01e6, 0x01e7, 0x01e8, 0x01e9, 0x01ea,
 0x01eb, 0x01ec, 0x01ed, 0x01ee, 0x01ef, 0x03e0, 0x03e1, 0x03e2,
 0x03e3, 0x03e4, 0x03e5, 0x03e6, 0x07ce, 0x07cf, 0x07d0, 0x07d1,
 0x07d2, 0x07d3, 0x07d4, 0x07d5, 0x0fac, 0x0fad, 0x0fae, 0x0faf,
 0x0fb0, 0x0fb1, 0x0fb2, 0x0fb3, 0x0fb4, 0x0fb5, 0x0fb6, 0x0fb7,
 0x0fb8, 0x0fb9, 0x0fba, 0x0fbb, 0x0fbc, 0x0fbd, 0x0fbe, 0x0fbf,
 0x1f80, 0x1f81, 0x1f82, 0x1f83, 0x1f84, 0x1f85, 0x1f86, 0x1f87,
 0x1f88, 0x1f89, 0x1f8a, 0x1f8b, 0x1f8c, 0x1f8d, 0x1f8e, 0x1f8f,
 0x1f90, 0x1f91, 0x1f92, 0x1f93, 0x1f94, 0x1f95, 0x1f96, 0x1f97,
 0x1f98, 0x1f99, 0x1f9a, 0x1f9b, 0x1f9c, 0x1f9d, 0x1f9e, 0x1f9f,
 0x1fa0, 0x1fa1, 0x1fa2, 0x1fa3, 0x1fa4, 0x1fa5, 0x1fa6, 0x1fa7,
 0x1fa8, 0x1fa9, 0x1faa, 0x1fab, 0x1fac, 0x1fad, 0x1fae, 0x1faf,
 0x1fb0, 0x1fb1, 0x1fb2, 0x1fb3, 0x1fb4, 0x1fb5, 0x1fb6, 0x1fb7,
 0x1fb8, 0x1fb9, 0x1fba, 0x1fbb, 0x1fbc, 0x1fbd, 0x1fbe, 0x1fbf,
 0x7f00, 0x7f01, 0x7f02, 0x7f03, 0x7f04, 0x7f05, 0x7f06, 0x7f07,
 0x7f08, 0x7f09, 0x7f0a, 0x7f0b, 0x7f0c, 0x7f0d, 0x7f0e, 0x7f0f,
 0x7f10, 0x7f11, 0x7f12, 0x7f13, 0x7f14, 0x7f15, 0x7f16, 0x7f17,
 0x7f18, 0x7f19, 0x7f1a, 0x7f1b, 0x7f1c, 0x7f1d, 0x7f1e, 0x7f1f,
 0x7f20, 0x7f21, 0x7f22, 0x7f23, 0x7f24, 0x7f25, 0x7f26, 0x7f27,
 0x7f28, 0x7f29, 0x7f2a, 0x7f2b, 0x7f2c, 0x7f2d, 0x7f2e, 0x7f2f,
 0x7f30, 0x7f31, 0x7f32, 0x7f33, 0x7f34, 0x7f35, 0x7f36, 0x7f37,
 0x7f38, 0x7f39, 0x7f3a, 0x7f3b, 0x7f3c, 0x7f3d, 0x7f3e, 0x7f3f,
 0x7f40, 0x7f41, 0x7f42, 0x7f43, 0x7f44, 0x7f45, 0x7f46, 0x7f47,
 0x7f48, 0x7f49, 0x7f4a, 0x7f4b, 0x7f4c, 0x7f4d, 0x7f4e, 0x7f4f,
 0x7f50, 0x7f51, 0x7f52, 0x7f53, 0x7f54, 0x7f55, 0x7f56, 0x7f57,
 0x7f58, 0x7f59, 0x7f5a, 0x7f5b, 0x7f5c, 0x7f5d, 0x7f5e, 0x7f5f,
 0x7f60, 0x7f61, 0x7f62, 0x7f63, 0x7f64, 0x7f65, 0x7f66, 0x7f67,
 0x7f68, 0x7f69, 0x7f6a, 0x7f6b, 0x7f6c, 0x7f6d, 0x7f6e, 0x7f6f,
 0x7f70, 0x7f71, 0x7f72, 0x7f73, 0x7f74, 0x7f75, 0x7f76, 0x7f77,
 0x7f78, 0x7f79, 0x7f7a, 0x7f7b, 0x7f7c, 0x7f7d, 0x7f7e, 0x7f7f,
 0x7f80, 0x7f81, 0x7f82, 0x7f83, 0x7f84, 0x7f85, 0x7f86, 0x7f87,
 0x7f88, 0x7f89, 0x7f8a, 0x7f8b, 0x7f8c, 0x7f8d, 0x7f8e, 0x7f8f,
 0x7f90, 0x7f91, 0x7f92, 0x7f93, 0x7f94, 0x7f95, 0x7f96, 0x7f97,
 0x7f98, 0x7f99, 0x7f9a, 0x7f9b, 0x7f9c, 0x7f9d, 0x7f9e, 0x7f9f,
 0x7fa0, 0x7fa1, 0x7fa2, 0x7fa3, 0x7fa4, 0x7fa5, 0x7fa6, 0x7fa7,
 0x7fa8, 0x7fa9, 0x7faa, 0x7fab, 0x7fac, 0x7fad, 0x7fae, 0x7faf,
 0x7fb0, 0x7fb1, 0x7fb2, 0x7fb3, 0x7fb4, 0x7fb5, 0x7fb6, 0x7fb7,
 0x7fb8, 0x7fb9, 0x7fba, 0x7fbb, 0x7fbc, 0x7fbd, 0x7fbe, 0x7fbf,
 0x7fc0, 0x7fc1, 0x7fc2, 0x7fc3, 0x7fc4, 0x7fc5, 0x7fc6, 0x7fc7,
 0x7fc8, 0x7fc9, 0x7fca, 0x7fcb, 0x7fcc, 0x7fcd, 0x7fce, 0x7fcf,
 0x7fd0, 0x7fd1, 0x7fd2, 0x7fd3, 0x7fd4, 0x7fd5, 0x7fd6, 0x7fd7,
 0x7fd8, 0x7fd9, 0x7fda, 0x7fdb, 0x7fdc, 0x7fdd, 0x7fde, 0x7fdf,
 0x7fe0, 0x7fe1, 0x7fe2, 0x7fe3, 0x7fe4, 0x7fe5, 0x7fe6, 0x7fe7,
 0x7fe8, 0x7fe9, 0x7fea, 0x7feb, 0x7fec, 0x7fed, 0x7fee, 0x7fef,
 0x7ff0, 0x7ff1, 0x7ff2, 0x7ff3, 0x7ff4, 0x7ff5, 0x7ff6, 0x7ff7,
 0x7ff8, 0x7ff9, 0x7ffa, 0x7ffb, 0x7ffc, 0x7ffd, 0x7ffe, 0x7fff,
 0x0006,
};

static const uint8_t dv_vlc_len[409] = {
  2,  3,  4,  4,  4,  5,  5,  5,
  5,  6,  6,  6,  6,  7,  7,  7,
  7,  7,  7,  7,  7,  8,  8,  8,
  8,  8,  8,  8,  8,  8,  8,  8,
  8,  8,  8,  8,  8,  9,  9,  9,
  9,  9,  9,  9,  9,  9,  9,  9,
  9,  9,  9,  9,  9, 10, 10, 10,
 10, 10, 10, 10, 11, 11, 11, 11,
 11, 11, 11, 11, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
  4,
};

static const uint8_t dv_vlc_run[409] = {
  0,  0,  1,  0,  0,  2,  1,  0,
  0,  3,  4,  0,  0,  5,  6,  2,
  1,  1,  0,  0,  0,  7,  8,  9,
 10,  3,  4,  2,  1,  1,  1,  0,
  0,  0,  0,  0,  0, 11, 12, 13,
 14,  5,  6,  3,  4,  2,  2,  1,
  0,  0,  0,  0,  0,  5,  3,  3,
  2,  1,  1,  1,  0,  1,  6,  4,
  3,  1,  1,  1,  2,  3,  4,  5,
  7,  8,  9, 10,  7,  8,  4,  3,
  2,  2,  2,  2,  2,  1,  1,  1,
  0,  1,  2,  3,  4,  5,  6,  7,
  8,  9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23,
 24, 25, 26, 27, 28, 29, 30, 31,
 32, 33, 34, 35, 36, 37, 38, 39,
 40, 41, 42, 43, 44, 45, 46, 47,
 48, 49, 50, 51, 52, 53, 54, 55,
 56, 57, 58, 59, 60, 61, 62, 63,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
127,
};

static const uint8_t dv_vlc_level[409] = {
  1,  2,  1,  3,  4,  1,  2,  5,
  6,  1,  1,  7,  8,  1,  1,  2,
  3,  4,  9, 10, 11,  1,  1,  1,
  1,  2,  2,  3,  5,  6,  7, 12,
 13, 14, 15, 16, 17,  1,  1,  1,
  1,  2,  2,  3,  3,  4,  5,  8,
 18, 19, 20, 21, 22,  3,  4,  5,
  6,  9, 10, 11,  0,  0,  3,  4,
  6, 12, 13, 14,  0,  0,  0,  0,
  2,  2,  2,  2,  3,  3,  5,  7,
  7,  8,  9, 10, 11, 15, 16, 17,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  1,  2,  3,  4,  5,  6,  7,
  8,  9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23,
 24, 25, 26, 27, 28, 29, 30, 31,
 32, 33, 34, 35, 36, 37, 38, 39,
 40, 41, 42, 43, 44, 45, 46, 47,
 48, 49, 50, 51, 52, 53, 54, 55,
 56, 57, 58, 59, 60, 61, 62, 63,
 64, 65, 66, 67, 68, 69, 70, 71,
 72, 73, 74, 75, 76, 77, 78, 79,
 80, 81, 82, 83, 84, 85, 86, 87,
 88, 89, 90, 91, 92, 93, 94, 95,
 96, 97, 98, 99, 100, 101, 102, 103,
 104, 105, 106, 107, 108, 109, 110, 111,
 112, 113, 114, 115, 116, 117, 118, 119,
 120, 121, 122, 123, 124, 125, 126, 127,
 128, 129, 130, 131, 132, 133, 134, 135,
 136, 137, 138, 139, 140, 141, 142, 143,
 144, 145, 146, 147, 148, 149, 150, 151,
 152, 153, 154, 155, 156, 157, 158, 159,
 160, 161, 162, 163, 164, 165, 166, 167,
 168, 169, 170, 171, 172, 173, 174, 175,
 176, 177, 178, 179, 180, 181, 182, 183,
 184, 185, 186, 187, 188, 189, 190, 191,
 192, 193, 194, 195, 196, 197, 198, 199,
 200, 201, 202, 203, 204, 205, 206, 207,
 208, 209, 210, 211, 212, 213, 214, 215,
 216, 217, 218, 219, 220, 221, 222, 223,
 224, 225, 226, 227, 228, 229, 230, 231,
 232, 233, 234, 235, 236, 237, 238, 239,
 240, 241, 242, 243, 244, 245, 246, 247,
 248, 249, 250, 251, 252, 253, 254, 255,
  0,
};

/* unquant tables (not used directly) */
static const uint8_t dv_quant_shifts[22][4] = {
  { 3,3,4,4 },
  { 3,3,4,4 },
  { 2,3,3,4 },
  { 2,3,3,4 },
  { 2,2,3,3 },
  { 2,2,3,3 },
  { 1,2,2,3 },
  { 1,2,2,3 },
  { 1,1,2,2 },
  { 1,1,2,2 },
  { 0,1,1,2 },
  { 0,1,1,2 },
  { 0,0,1,1 },
  { 0,0,1,1 },
  { 0,0,0,1 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
};

static const uint8_t dv_quant_offset[4] = { 6, 3, 0, 1 };
static const uint8_t dv_quant_areas[4] = { 6, 21, 43, 64 };

/* quantization quanta by QNO for DV100 */
static const uint8_t dv100_qstep[16] = {
    1, /* QNO = 0 and 1 both have no quantization */
    1,
    2, 3, 4, 5, 6, 7, 8, 16, 18, 20, 22, 24, 28, 52
};

/* NOTE: I prefer hardcoding the positioning of dv blocks, it is
   simpler :-) */

static const uint16_t dv_place_420[1620] = {
 0x0c24, 0x2412, 0x3036, 0x0000, 0x1848,
 0x0e24, 0x2612, 0x3236, 0x0200, 0x1a48,
 0x1024, 0x2812, 0x3436, 0x0400, 0x1c48,
 0x1026, 0x2814, 0x3438, 0x0402, 0x1c4a,
 0x0e26, 0x2614, 0x3238, 0x0202, 0x1a4a,
 0x0c26, 0x2414, 0x3038, 0x0002, 0x184a,
 0x0c28, 0x2416, 0x303a, 0x0004, 0x184c,
 0x0e28, 0x2616, 0x323a, 0x0204, 0x1a4c,
 0x1028, 0x2816, 0x343a, 0x0404, 0x1c4c,
 0x102a, 0x2818, 0x343c, 0x0406, 0x1c4e,
 0x0e2a, 0x2618, 0x323c, 0x0206, 0x1a4e,
 0x0c2a, 0x2418, 0x303c, 0x0006, 0x184e,
 0x0c2c, 0x241a, 0x303e, 0x0008, 0x1850,
 0x0e2c, 0x261a, 0x323e, 0x0208, 0x1a50,
 0x102c, 0x281a, 0x343e, 0x0408, 0x1c50,
 0x102e, 0x281c, 0x3440, 0x040a, 0x1c52,
 0x0e2e, 0x261c, 0x3240, 0x020a, 0x1a52,
 0x0c2e, 0x241c, 0x3040, 0x000a, 0x1852,
 0x0c30, 0x241e, 0x3042, 0x000c, 0x1854,
 0x0e30, 0x261e, 0x3242, 0x020c, 0x1a54,
 0x1030, 0x281e, 0x3442, 0x040c, 0x1c54,
 0x1032, 0x2820, 0x3444, 0x040e, 0x1c56,
 0x0e32, 0x2620, 0x3244, 0x020e, 0x1a56,
 0x0c32, 0x2420, 0x3044, 0x000e, 0x1856,
 0x0c34, 0x2422, 0x3046, 0x0010, 0x1858,
 0x0e34, 0x2622, 0x3246, 0x0210, 0x1a58,
 0x1034, 0x2822, 0x3446, 0x0410, 0x1c58,
 0x1224, 0x2a12, 0x3636, 0x0600, 0x1e48,
 0x1424, 0x2c12, 0x3836, 0x0800, 0x2048,
 0x1624, 0x2e12, 0x3a36, 0x0a00, 0x2248,
 0x1626, 0x2e14, 0x3a38, 0x0a02, 0x224a,
 0x1426, 0x2c14, 0x3838, 0x0802, 0x204a,
 0x1226, 0x2a14, 0x3638, 0x0602, 0x1e4a,
 0x1228, 0x2a16, 0x363a, 0x0604, 0x1e4c,
 0x1428, 0x2c16, 0x383a, 0x0804, 0x204c,
 0x1628, 0x2e16, 0x3a3a, 0x0a04, 0x224c,
 0x162a, 0x2e18, 0x3a3c, 0x0a06, 0x224e,
 0x142a, 0x2c18, 0x383c, 0x0806, 0x204e,
 0x122a, 0x2a18, 0x363c, 0x0606, 0x1e4e,
 0x122c, 0x2a1a, 0x363e, 0x0608, 0x1e50,
 0x142c, 0x2c1a, 0x383e, 0x0808, 0x2050,
 0x162c, 0x2e1a, 0x3a3e, 0x0a08, 0x2250,
 0x162e, 0x2e1c, 0x3a40, 0x0a0a, 0x2252,
 0x142e, 0x2c1c, 0x3840, 0x080a, 0x2052,
 0x122e, 0x2a1c, 0x3640, 0x060a, 0x1e52,
 0x1230, 0x2a1e, 0x3642, 0x060c, 0x1e54,
 0x1430, 0x2c1e, 0x3842, 0x080c, 0x2054,
 0x1630, 0x2e1e, 0x3a42, 0x0a0c, 0x2254,
 0x1632, 0x2e20, 0x3a44, 0x0a0e, 0x2256,
 0x1432, 0x2c20, 0x3844, 0x080e, 0x2056,
 0x1232, 0x2a20, 0x3644, 0x060e, 0x1e56,
 0x1234, 0x2a22, 0x3646, 0x0610, 0x1e58,
 0x1434, 0x2c22, 0x3846, 0x0810, 0x2058,
 0x1634, 0x2e22, 0x3a46, 0x0a10, 0x2258,
 0x1824, 0x3012, 0x3c36, 0x0c00, 0x2448,
 0x1a24, 0x3212, 0x3e36, 0x0e00, 0x2648,
 0x1c24, 0x3412, 0x4036, 0x1000, 0x2848,
 0x1c26, 0x3414, 0x4038, 0x1002, 0x284a,
 0x1a26, 0x3214, 0x3e38, 0x0e02, 0x264a,
 0x1826, 0x3014, 0x3c38, 0x0c02, 0x244a,
 0x1828, 0x3016, 0x3c3a, 0x0c04, 0x244c,
 0x1a28, 0x3216, 0x3e3a, 0x0e04, 0x264c,
 0x1c28, 0x3416, 0x403a, 0x1004, 0x284c,
 0x1c2a, 0x3418, 0x403c, 0x1006, 0x284e,
 0x1a2a, 0x3218, 0x3e3c, 0x0e06, 0x264e,
 0x182a, 0x3018, 0x3c3c, 0x0c06, 0x244e,
 0x182c, 0x301a, 0x3c3e, 0x0c08, 0x2450,
 0x1a2c, 0x321a, 0x3e3e, 0x0e08, 0x2650,
 0x1c2c, 0x341a, 0x403e, 0x1008, 0x2850,
 0x1c2e, 0x341c, 0x4040, 0x100a, 0x2852,
 0x1a2e, 0x321c, 0x3e40, 0x0e0a, 0x2652,
 0x182e, 0x301c, 0x3c40, 0x0c0a, 0x2452,
 0x1830, 0x301e, 0x3c42, 0x0c0c, 0x2454,
 0x1a30, 0x321e, 0x3e42, 0x0e0c, 0x2654,
 0x1c30, 0x341e, 0x4042, 0x100c, 0x2854,
 0x1c32, 0x3420, 0x4044, 0x100e, 0x2856,
 0x1a32, 0x3220, 0x3e44, 0x0e0e, 0x2656,
 0x1832, 0x3020, 0x3c44, 0x0c0e, 0x2456,
 0x1834, 0x3022, 0x3c46, 0x0c10, 0x2458,
 0x1a34, 0x3222, 0x3e46, 0x0e10, 0x2658,
 0x1c34, 0x3422, 0x4046, 0x1010, 0x2858,
 0x1e24, 0x3612, 0x4236, 0x1200, 0x2a48,
 0x2024, 0x3812, 0x4436, 0x1400, 0x2c48,
 0x2224, 0x3a12, 0x4636, 0x1600, 0x2e48,
 0x2226, 0x3a14, 0x4638, 0x1602, 0x2e4a,
 0x2026, 0x3814, 0x4438, 0x1402, 0x2c4a,
 0x1e26, 0x3614, 0x4238, 0x1202, 0x2a4a,
 0x1e28, 0x3616, 0x423a, 0x1204, 0x2a4c,
 0x2028, 0x3816, 0x443a, 0x1404, 0x2c4c,
 0x2228, 0x3a16, 0x463a, 0x1604, 0x2e4c,
 0x222a, 0x3a18, 0x463c, 0x1606, 0x2e4e,
 0x202a, 0x3818, 0x443c, 0x1406, 0x2c4e,
 0x1e2a, 0x3618, 0x423c, 0x1206, 0x2a4e,
 0x1e2c, 0x361a, 0x423e, 0x1208, 0x2a50,
 0x202c, 0x381a, 0x443e, 0x1408, 0x2c50,
 0x222c, 0x3a1a, 0x463e, 0x1608, 0x2e50,
 0x222e, 0x3a1c, 0x4640, 0x160a, 0x2e52,
 0x202e, 0x381c, 0x4440, 0x140a, 0x2c52,
 0x1e2e, 0x361c, 0x4240, 0x120a, 0x2a52,
 0x1e30, 0x361e, 0x4242, 0x120c, 0x2a54,
 0x2030, 0x381e, 0x4442, 0x140c, 0x2c54,
 0x2230, 0x3a1e, 0x4642, 0x160c, 0x2e54,
 0x2232, 0x3a20, 0x4644, 0x160e, 0x2e56,
 0x2032, 0x3820, 0x4444, 0x140e, 0x2c56,
 0x1e32, 0x3620, 0x4244, 0x120e, 0x2a56,
 0x1e34, 0x3622, 0x4246, 0x1210, 0x2a58,
 0x2034, 0x3822, 0x4446, 0x1410, 0x2c58,
 0x2234, 0x3a22, 0x4646, 0x1610, 0x2e58,
 0x2424, 0x3c12, 0x0036, 0x1800, 0x3048,
 0x2624, 0x3e12, 0x0236, 0x1a00, 0x3248,
 0x2824, 0x4012, 0x0436, 0x1c00, 0x3448,
 0x2826, 0x4014, 0x0438, 0x1c02, 0x344a,
 0x2626, 0x3e14, 0x0238, 0x1a02, 0x324a,
 0x2426, 0x3c14, 0x0038, 0x1802, 0x304a,
 0x2428, 0x3c16, 0x003a, 0x1804, 0x304c,
 0x2628, 0x3e16, 0x023a, 0x1a04, 0x324c,
 0x2828, 0x4016, 0x043a, 0x1c04, 0x344c,
 0x282a, 0x4018, 0x043c, 0x1c06, 0x344e,
 0x262a, 0x3e18, 0x023c, 0x1a06, 0x324e,
 0x242a, 0x3c18, 0x003c, 0x1806, 0x304e,
 0x242c, 0x3c1a, 0x003e, 0x1808, 0x3050,
 0x262c, 0x3e1a, 0x023e, 0x1a08, 0x3250,
 0x282c, 0x401a, 0x043e, 0x1c08, 0x3450,
 0x282e, 0x401c, 0x0440, 0x1c0a, 0x3452,
 0x262e, 0x3e1c, 0x0240, 0x1a0a, 0x3252,
 0x242e, 0x3c1c, 0x0040, 0x180a, 0x3052,
 0x2430, 0x3c1e, 0x0042, 0x180c, 0x3054,
 0x2630, 0x3e1e, 0x0242, 0x1a0c, 0x3254,
 0x2830, 0x401e, 0x0442, 0x1c0c, 0x3454,
 0x2832, 0x4020, 0x0444, 0x1c0e, 0x3456,
 0x2632, 0x3e20, 0x0244, 0x1a0e, 0x3256,
 0x2432, 0x3c20, 0x0044, 0x180e, 0x3056,
 0x2434, 0x3c22, 0x0046, 0x1810, 0x3058,
 0x2634, 0x3e22, 0x0246, 0x1a10, 0x3258,
 0x2834, 0x4022, 0x0446, 0x1c10, 0x3458,
 0x2a24, 0x4212, 0x0636, 0x1e00, 0x3648,
 0x2c24, 0x4412, 0x0836, 0x2000, 0x3848,
 0x2e24, 0x4612, 0x0a36, 0x2200, 0x3a48,
 0x2e26, 0x4614, 0x0a38, 0x2202, 0x3a4a,
 0x2c26, 0x4414, 0x0838, 0x2002, 0x384a,
 0x2a26, 0x4214, 0x0638, 0x1e02, 0x364a,
 0x2a28, 0x4216, 0x063a, 0x1e04, 0x364c,
 0x2c28, 0x4416, 0x083a, 0x2004, 0x384c,
 0x2e28, 0x4616, 0x0a3a, 0x2204, 0x3a4c,
 0x2e2a, 0x4618, 0x0a3c, 0x2206, 0x3a4e,
 0x2c2a, 0x4418, 0x083c, 0x2006, 0x384e,
 0x2a2a, 0x4218, 0x063c, 0x1e06, 0x364e,
 0x2a2c, 0x421a, 0x063e, 0x1e08, 0x3650,
 0x2c2c, 0x441a, 0x083e, 0x2008, 0x3850,
 0x2e2c, 0x461a, 0x0a3e, 0x2208, 0x3a50,
 0x2e2e, 0x461c, 0x0a40, 0x220a, 0x3a52,
 0x2c2e, 0x441c, 0x0840, 0x200a, 0x3852,
 0x2a2e, 0x421c, 0x0640, 0x1e0a, 0x3652,
 0x2a30, 0x421e, 0x0642, 0x1e0c, 0x3654,
 0x2c30, 0x441e, 0x0842, 0x200c, 0x3854,
 0x2e30, 0x461e, 0x0a42, 0x220c, 0x3a54,
 0x2e32, 0x4620, 0x0a44, 0x220e, 0x3a56,
 0x2c32, 0x4420, 0x0844, 0x200e, 0x3856,
 0x2a32, 0x4220, 0x0644, 0x1e0e, 0x3656,
 0x2a34, 0x4222, 0x0646, 0x1e10, 0x3658,
 0x2c34, 0x4422, 0x0846, 0x2010, 0x3858,
 0x2e34, 0x4622, 0x0a46, 0x2210, 0x3a58,
 0x3024, 0x0012, 0x0c36, 0x2400, 0x3c48,
 0x3224, 0x0212, 0x0e36, 0x2600, 0x3e48,
 0x3424, 0x0412, 0x1036, 0x2800, 0x4048,
 0x3426, 0x0414, 0x1038, 0x2802, 0x404a,
 0x3226, 0x0214, 0x0e38, 0x2602, 0x3e4a,
 0x3026, 0x0014, 0x0c38, 0x2402, 0x3c4a,
 0x3028, 0x0016, 0x0c3a, 0x2404, 0x3c4c,
 0x3228, 0x0216, 0x0e3a, 0x2604, 0x3e4c,
 0x3428, 0x0416, 0x103a, 0x2804, 0x404c,
 0x342a, 0x0418, 0x103c, 0x2806, 0x404e,
 0x322a, 0x0218, 0x0e3c, 0x2606, 0x3e4e,
 0x302a, 0x0018, 0x0c3c, 0x2406, 0x3c4e,
 0x302c, 0x001a, 0x0c3e, 0x2408, 0x3c50,
 0x322c, 0x021a, 0x0e3e, 0x2608, 0x3e50,
 0x342c, 0x041a, 0x103e, 0x2808, 0x4050,
 0x342e, 0x041c, 0x1040, 0x280a, 0x4052,
 0x322e, 0x021c, 0x0e40, 0x260a, 0x3e52,
 0x302e, 0x001c, 0x0c40, 0x240a, 0x3c52,
 0x3030, 0x001e, 0x0c42, 0x240c, 0x3c54,
 0x3230, 0x021e, 0x0e42, 0x260c, 0x3e54,
 0x3430, 0x041e, 0x1042, 0x280c, 0x4054,
 0x3432, 0x0420, 0x1044, 0x280e, 0x4056,
 0x3232, 0x0220, 0x0e44, 0x260e, 0x3e56,
 0x3032, 0x0020, 0x0c44, 0x240e, 0x3c56,
 0x3034, 0x0022, 0x0c46, 0x2410, 0x3c58,
 0x3234, 0x0222, 0x0e46, 0x2610, 0x3e58,
 0x3434, 0x0422, 0x1046, 0x2810, 0x4058,
 0x3624, 0x0612, 0x1236, 0x2a00, 0x4248,
 0x3824, 0x0812, 0x1436, 0x2c00, 0x4448,
 0x3a24, 0x0a12, 0x1636, 0x2e00, 0x4648,
 0x3a26, 0x0a14, 0x1638, 0x2e02, 0x464a,
 0x3826, 0x0814, 0x1438, 0x2c02, 0x444a,
 0x3626, 0x0614, 0x1238, 0x2a02, 0x424a,
 0x3628, 0x0616, 0x123a, 0x2a04, 0x424c,
 0x3828, 0x0816, 0x143a, 0x2c04, 0x444c,
 0x3a28, 0x0a16, 0x163a, 0x2e04, 0x464c,
 0x3a2a, 0x0a18, 0x163c, 0x2e06, 0x464e,
 0x382a, 0x0818, 0x143c, 0x2c06, 0x444e,
 0x362a, 0x0618, 0x123c, 0x2a06, 0x424e,
 0x362c, 0x061a, 0x123e, 0x2a08, 0x4250,
 0x382c, 0x081a, 0x143e, 0x2c08, 0x4450,
 0x3a2c, 0x0a1a, 0x163e, 0x2e08, 0x4650,
 0x3a2e, 0x0a1c, 0x1640, 0x2e0a, 0x4652,
 0x382e, 0x081c, 0x1440, 0x2c0a, 0x4452,
 0x362e, 0x061c, 0x1240, 0x2a0a, 0x4252,
 0x3630, 0x061e, 0x1242, 0x2a0c, 0x4254,
 0x3830, 0x081e, 0x1442, 0x2c0c, 0x4454,
 0x3a30, 0x0a1e, 0x1642, 0x2e0c, 0x4654,
 0x3a32, 0x0a20, 0x1644, 0x2e0e, 0x4656,
 0x3832, 0x0820, 0x1444, 0x2c0e, 0x4456,
 0x3632, 0x0620, 0x1244, 0x2a0e, 0x4256,
 0x3634, 0x0622, 0x1246, 0x2a10, 0x4258,
 0x3834, 0x0822, 0x1446, 0x2c10, 0x4458,
 0x3a34, 0x0a22, 0x1646, 0x2e10, 0x4658,
 0x3c24, 0x0c12, 0x1836, 0x3000, 0x0048,
 0x3e24, 0x0e12, 0x1a36, 0x3200, 0x0248,
 0x4024, 0x1012, 0x1c36, 0x3400, 0x0448,
 0x4026, 0x1014, 0x1c38, 0x3402, 0x044a,
 0x3e26, 0x0e14, 0x1a38, 0x3202, 0x024a,
 0x3c26, 0x0c14, 0x1838, 0x3002, 0x004a,
 0x3c28, 0x0c16, 0x183a, 0x3004, 0x004c,
 0x3e28, 0x0e16, 0x1a3a, 0x3204, 0x024c,
 0x4028, 0x1016, 0x1c3a, 0x3404, 0x044c,
 0x402a, 0x1018, 0x1c3c, 0x3406, 0x044e,
 0x3e2a, 0x0e18, 0x1a3c, 0x3206, 0x024e,
 0x3c2a, 0x0c18, 0x183c, 0x3006, 0x004e,
 0x3c2c, 0x0c1a, 0x183e, 0x3008, 0x0050,
 0x3e2c, 0x0e1a, 0x1a3e, 0x3208, 0x0250,
 0x402c, 0x101a, 0x1c3e, 0x3408, 0x0450,
 0x402e, 0x101c, 0x1c40, 0x340a, 0x0452,
 0x3e2e, 0x0e1c, 0x1a40, 0x320a, 0x0252,
 0x3c2e, 0x0c1c, 0x1840, 0x300a, 0x0052,
 0x3c30, 0x0c1e, 0x1842, 0x300c, 0x0054,
 0x3e30, 0x0e1e, 0x1a42, 0x320c, 0x0254,
 0x4030, 0x101e, 0x1c42, 0x340c, 0x0454,
 0x4032, 0x1020, 0x1c44, 0x340e, 0x0456,
 0x3e32, 0x0e20, 0x1a44, 0x320e, 0x0256,
 0x3c32, 0x0c20, 0x1844, 0x300e, 0x0056,
 0x3c34, 0x0c22, 0x1846, 0x3010, 0x0058,
 0x3e34, 0x0e22, 0x1a46, 0x3210, 0x0258,
 0x4034, 0x1022, 0x1c46, 0x3410, 0x0458,
 0x4224, 0x1212, 0x1e36, 0x3600, 0x0648,
 0x4424, 0x1412, 0x2036, 0x3800, 0x0848,
 0x4624, 0x1612, 0x2236, 0x3a00, 0x0a48,
 0x4626, 0x1614, 0x2238, 0x3a02, 0x0a4a,
 0x4426, 0x1414, 0x2038, 0x3802, 0x084a,
 0x4226, 0x1214, 0x1e38, 0x3602, 0x064a,
 0x4228, 0x1216, 0x1e3a, 0x3604, 0x064c,
 0x4428, 0x1416, 0x203a, 0x3804, 0x084c,
 0x4628, 0x1616, 0x223a, 0x3a04, 0x0a4c,
 0x462a, 0x1618, 0x223c, 0x3a06, 0x0a4e,
 0x442a, 0x1418, 0x203c, 0x3806, 0x084e,
 0x422a, 0x1218, 0x1e3c, 0x3606, 0x064e,
 0x422c, 0x121a, 0x1e3e, 0x3608, 0x0650,
 0x442c, 0x141a, 0x203e, 0x3808, 0x0850,
 0x462c, 0x161a, 0x223e, 0x3a08, 0x0a50,
 0x462e, 0x161c, 0x2240, 0x3a0a, 0x0a52,
 0x442e, 0x141c, 0x2040, 0x380a, 0x0852,
 0x422e, 0x121c, 0x1e40, 0x360a, 0x0652,
 0x4230, 0x121e, 0x1e42, 0x360c, 0x0654,
 0x4430, 0x141e, 0x2042, 0x380c, 0x0854,
 0x4630, 0x161e, 0x2242, 0x3a0c, 0x0a54,
 0x4632, 0x1620, 0x2244, 0x3a0e, 0x0a56,
 0x4432, 0x1420, 0x2044, 0x380e, 0x0856,
 0x4232, 0x1220, 0x1e44, 0x360e, 0x0656,
 0x4234, 0x1222, 0x1e46, 0x3610, 0x0658,
 0x4434, 0x1422, 0x2046, 0x3810, 0x0858,
 0x4634, 0x1622, 0x2246, 0x3a10, 0x0a58,
 0x0024, 0x1812, 0x2436, 0x3c00, 0x0c48,
 0x0224, 0x1a12, 0x2636, 0x3e00, 0x0e48,
 0x0424, 0x1c12, 0x2836, 0x4000, 0x1048,
 0x0426, 0x1c14, 0x2838, 0x4002, 0x104a,
 0x0226, 0x1a14, 0x2638, 0x3e02, 0x0e4a,
 0x0026, 0x1814, 0x2438, 0x3c02, 0x0c4a,
 0x0028, 0x1816, 0x243a, 0x3c04, 0x0c4c,
 0x0228, 0x1a16, 0x263a, 0x3e04, 0x0e4c,
 0x0428, 0x1c16, 0x283a, 0x4004, 0x104c,
 0x042a, 0x1c18, 0x283c, 0x4006, 0x104e,
 0x022a, 0x1a18, 0x263c, 0x3e06, 0x0e4e,
 0x002a, 0x1818, 0x243c, 0x3c06, 0x0c4e,
 0x002c, 0x181a, 0x243e, 0x3c08, 0x0c50,
 0x022c, 0x1a1a, 0x263e, 0x3e08, 0x0e50,
 0x042c, 0x1c1a, 0x283e, 0x4008, 0x1050,
 0x042e, 0x1c1c, 0x2840, 0x400a, 0x1052,
 0x022e, 0x1a1c, 0x2640, 0x3e0a, 0x0e52,
 0x002e, 0x181c, 0x2440, 0x3c0a, 0x0c52,
 0x0030, 0x181e, 0x2442, 0x3c0c, 0x0c54,
 0x0230, 0x1a1e, 0x2642, 0x3e0c, 0x0e54,
 0x0430, 0x1c1e, 0x2842, 0x400c, 0x1054,
 0x0432, 0x1c20, 0x2844, 0x400e, 0x1056,
 0x0232, 0x1a20, 0x2644, 0x3e0e, 0x0e56,
 0x0032, 0x1820, 0x2444, 0x3c0e, 0x0c56,
 0x0034, 0x1822, 0x2446, 0x3c10, 0x0c58,
 0x0234, 0x1a22, 0x2646, 0x3e10, 0x0e58,
 0x0434, 0x1c22, 0x2846, 0x4010, 0x1058,
 0x0624, 0x1e12, 0x2a36, 0x4200, 0x1248,
 0x0824, 0x2012, 0x2c36, 0x4400, 0x1448,
 0x0a24, 0x2212, 0x2e36, 0x4600, 0x1648,
 0x0a26, 0x2214, 0x2e38, 0x4602, 0x164a,
 0x0826, 0x2014, 0x2c38, 0x4402, 0x144a,
 0x0626, 0x1e14, 0x2a38, 0x4202, 0x124a,
 0x0628, 0x1e16, 0x2a3a, 0x4204, 0x124c,
 0x0828, 0x2016, 0x2c3a, 0x4404, 0x144c,
 0x0a28, 0x2216, 0x2e3a, 0x4604, 0x164c,
 0x0a2a, 0x2218, 0x2e3c, 0x4606, 0x164e,
 0x082a, 0x2018, 0x2c3c, 0x4406, 0x144e,
 0x062a, 0x1e18, 0x2a3c, 0x4206, 0x124e,
 0x062c, 0x1e1a, 0x2a3e, 0x4208, 0x1250,
 0x082c, 0x201a, 0x2c3e, 0x4408, 0x1450,
 0x0a2c, 0x221a, 0x2e3e, 0x4608, 0x1650,
 0x0a2e, 0x221c, 0x2e40, 0x460a, 0x1652,
 0x082e, 0x201c, 0x2c40, 0x440a, 0x1452,
 0x062e, 0x1e1c, 0x2a40, 0x420a, 0x1252,
 0x0630, 0x1e1e, 0x2a42, 0x420c, 0x1254,
 0x0830, 0x201e, 0x2c42, 0x440c, 0x1454,
 0x0a30, 0x221e, 0x2e42, 0x460c, 0x1654,
 0x0a32, 0x2220, 0x2e44, 0x460e, 0x1656,
 0x0832, 0x2020, 0x2c44, 0x440e, 0x1456,
 0x0632, 0x1e20, 0x2a44, 0x420e, 0x1256,
 0x0634, 0x1e22, 0x2a46, 0x4210, 0x1258,
 0x0834, 0x2022, 0x2c46, 0x4410, 0x1458,
 0x0a34, 0x2222, 0x2e46, 0x4610, 0x1658,
};

static const uint16_t dv_place_411P[1620] = {
 0x0c24, 0x2710, 0x3334, 0x0000, 0x1848,
 0x0d24, 0x2810, 0x3434, 0x0100, 0x1948,
 0x0e24, 0x2910, 0x3534, 0x0200, 0x1a48,
 0x0f24, 0x2914, 0x3538, 0x0300, 0x1b48,
 0x1024, 0x2814, 0x3438, 0x0400, 0x1c48,
 0x1124, 0x2714, 0x3338, 0x0500, 0x1d48,
 0x1128, 0x2614, 0x3238, 0x0504, 0x1d4c,
 0x1028, 0x2514, 0x3138, 0x0404, 0x1c4c,
 0x0f28, 0x2414, 0x3038, 0x0304, 0x1b4c,
 0x0e28, 0x2418, 0x303c, 0x0204, 0x1a4c,
 0x0d28, 0x2518, 0x313c, 0x0104, 0x194c,
 0x0c28, 0x2618, 0x323c, 0x0004, 0x184c,
 0x0c2c, 0x2718, 0x333c, 0x0008, 0x1850,
 0x0d2c, 0x2818, 0x343c, 0x0108, 0x1950,
 0x0e2c, 0x2918, 0x353c, 0x0208, 0x1a50,
 0x0f2c, 0x291c, 0x3540, 0x0308, 0x1b50,
 0x102c, 0x281c, 0x3440, 0x0408, 0x1c50,
 0x112c, 0x271c, 0x3340, 0x0508, 0x1d50,
 0x1130, 0x261c, 0x3240, 0x050c, 0x1d54,
 0x1030, 0x251c, 0x3140, 0x040c, 0x1c54,
 0x0f30, 0x241c, 0x3040, 0x030c, 0x1b54,
 0x0e30, 0x2420, 0x3044, 0x020c, 0x1a54,
 0x0d30, 0x2520, 0x3144, 0x010c, 0x1954,
 0x0c30, 0x2620, 0x3244, 0x000c, 0x1854,
 0x0c34, 0x2720, 0x3344, 0x0010, 0x1858,
 0x0d34, 0x2820, 0x3444, 0x0110, 0x1a58,
 0x0e34, 0x2920, 0x3544, 0x0210, 0x1c58,
 0x1224, 0x2d10, 0x3934, 0x0600, 0x1e48,
 0x1324, 0x2e10, 0x3a34, 0x0700, 0x1f48,
 0x1424, 0x2f10, 0x3b34, 0x0800, 0x2048,
 0x1524, 0x2f14, 0x3b38, 0x0900, 0x2148,
 0x1624, 0x2e14, 0x3a38, 0x0a00, 0x2248,
 0x1724, 0x2d14, 0x3938, 0x0b00, 0x2348,
 0x1728, 0x2c14, 0x3838, 0x0b04, 0x234c,
 0x1628, 0x2b14, 0x3738, 0x0a04, 0x224c,
 0x1528, 0x2a14, 0x3638, 0x0904, 0x214c,
 0x1428, 0x2a18, 0x363c, 0x0804, 0x204c,
 0x1328, 0x2b18, 0x373c, 0x0704, 0x1f4c,
 0x1228, 0x2c18, 0x383c, 0x0604, 0x1e4c,
 0x122c, 0x2d18, 0x393c, 0x0608, 0x1e50,
 0x132c, 0x2e18, 0x3a3c, 0x0708, 0x1f50,
 0x142c, 0x2f18, 0x3b3c, 0x0808, 0x2050,
 0x152c, 0x2f1c, 0x3b40, 0x0908, 0x2150,
 0x162c, 0x2e1c, 0x3a40, 0x0a08, 0x2250,
 0x172c, 0x2d1c, 0x3940, 0x0b08, 0x2350,
 0x1730, 0x2c1c, 0x3840, 0x0b0c, 0x2354,
 0x1630, 0x2b1c, 0x3740, 0x0a0c, 0x2254,
 0x1530, 0x2a1c, 0x3640, 0x090c, 0x2154,
 0x1430, 0x2a20, 0x3644, 0x080c, 0x2054,
 0x1330, 0x2b20, 0x3744, 0x070c, 0x1f54,
 0x1230, 0x2c20, 0x3844, 0x060c, 0x1e54,
 0x1234, 0x2d20, 0x3944, 0x0610, 0x1e58,
 0x1334, 0x2e20, 0x3a44, 0x0710, 0x2058,
 0x1434, 0x2f20, 0x3b44, 0x0810, 0x2258,
 0x1824, 0x3310, 0x3f34, 0x0c00, 0x2448,
 0x1924, 0x3410, 0x4034, 0x0d00, 0x2548,
 0x1a24, 0x3510, 0x4134, 0x0e00, 0x2648,
 0x1b24, 0x3514, 0x4138, 0x0f00, 0x2748,
 0x1c24, 0x3414, 0x4038, 0x1000, 0x2848,
 0x1d24, 0x3314, 0x3f38, 0x1100, 0x2948,
 0x1d28, 0x3214, 0x3e38, 0x1104, 0x294c,
 0x1c28, 0x3114, 0x3d38, 0x1004, 0x284c,
 0x1b28, 0x3014, 0x3c38, 0x0f04, 0x274c,
 0x1a28, 0x3018, 0x3c3c, 0x0e04, 0x264c,
 0x1928, 0x3118, 0x3d3c, 0x0d04, 0x254c,
 0x1828, 0x3218, 0x3e3c, 0x0c04, 0x244c,
 0x182c, 0x3318, 0x3f3c, 0x0c08, 0x2450,
 0x192c, 0x3418, 0x403c, 0x0d08, 0x2550,
 0x1a2c, 0x3518, 0x413c, 0x0e08, 0x2650,
 0x1b2c, 0x351c, 0x4140, 0x0f08, 0x2750,
 0x1c2c, 0x341c, 0x4040, 0x1008, 0x2850,
 0x1d2c, 0x331c, 0x3f40, 0x1108, 0x2950,
 0x1d30, 0x321c, 0x3e40, 0x110c, 0x2954,
 0x1c30, 0x311c, 0x3d40, 0x100c, 0x2854,
 0x1b30, 0x301c, 0x3c40, 0x0f0c, 0x2754,
 0x1a30, 0x3020, 0x3c44, 0x0e0c, 0x2654,
 0x1930, 0x3120, 0x3d44, 0x0d0c, 0x2554,
 0x1830, 0x3220, 0x3e44, 0x0c0c, 0x2454,
 0x1834, 0x3320, 0x3f44, 0x0c10, 0x2458,
 0x1934, 0x3420, 0x4044, 0x0d10, 0x2658,
 0x1a34, 0x3520, 0x4144, 0x0e10, 0x2858,
 0x1e24, 0x3910, 0x4534, 0x1200, 0x2a48,
 0x1f24, 0x3a10, 0x4634, 0x1300, 0x2b48,
 0x2024, 0x3b10, 0x4734, 0x1400, 0x2c48,
 0x2124, 0x3b14, 0x4738, 0x1500, 0x2d48,
 0x2224, 0x3a14, 0x4638, 0x1600, 0x2e48,
 0x2324, 0x3914, 0x4538, 0x1700, 0x2f48,
 0x2328, 0x3814, 0x4438, 0x1704, 0x2f4c,
 0x2228, 0x3714, 0x4338, 0x1604, 0x2e4c,
 0x2128, 0x3614, 0x4238, 0x1504, 0x2d4c,
 0x2028, 0x3618, 0x423c, 0x1404, 0x2c4c,
 0x1f28, 0x3718, 0x433c, 0x1304, 0x2b4c,
 0x1e28, 0x3818, 0x443c, 0x1204, 0x2a4c,
 0x1e2c, 0x3918, 0x453c, 0x1208, 0x2a50,
 0x1f2c, 0x3a18, 0x463c, 0x1308, 0x2b50,
 0x202c, 0x3b18, 0x473c, 0x1408, 0x2c50,
 0x212c, 0x3b1c, 0x4740, 0x1508, 0x2d50,
 0x222c, 0x3a1c, 0x4640, 0x1608, 0x2e50,
 0x232c, 0x391c, 0x4540, 0x1708, 0x2f50,
 0x2330, 0x381c, 0x4440, 0x170c, 0x2f54,
 0x2230, 0x371c, 0x4340, 0x160c, 0x2e54,
 0x2130, 0x361c, 0x4240, 0x150c, 0x2d54,
 0x2030, 0x3620, 0x4244, 0x140c, 0x2c54,
 0x1f30, 0x3720, 0x4344, 0x130c, 0x2b54,
 0x1e30, 0x3820, 0x4444, 0x120c, 0x2a54,
 0x1e34, 0x3920, 0x4544, 0x1210, 0x2a58,
 0x1f34, 0x3a20, 0x4644, 0x1310, 0x2c58,
 0x2034, 0x3b20, 0x4744, 0x1410, 0x2e58,
 0x2424, 0x3f10, 0x0334, 0x1800, 0x3048,
 0x2524, 0x4010, 0x0434, 0x1900, 0x3148,
 0x2624, 0x4110, 0x0534, 0x1a00, 0x3248,
 0x2724, 0x4114, 0x0538, 0x1b00, 0x3348,
 0x2824, 0x4014, 0x0438, 0x1c00, 0x3448,
 0x2924, 0x3f14, 0x0338, 0x1d00, 0x3548,
 0x2928, 0x3e14, 0x0238, 0x1d04, 0x354c,
 0x2828, 0x3d14, 0x0138, 0x1c04, 0x344c,
 0x2728, 0x3c14, 0x0038, 0x1b04, 0x334c,
 0x2628, 0x3c18, 0x003c, 0x1a04, 0x324c,
 0x2528, 0x3d18, 0x013c, 0x1904, 0x314c,
 0x2428, 0x3e18, 0x023c, 0x1804, 0x304c,
 0x242c, 0x3f18, 0x033c, 0x1808, 0x3050,
 0x252c, 0x4018, 0x043c, 0x1908, 0x3150,
 0x262c, 0x4118, 0x053c, 0x1a08, 0x3250,
 0x272c, 0x411c, 0x0540, 0x1b08, 0x3350,
 0x282c, 0x401c, 0x0440, 0x1c08, 0x3450,
 0x292c, 0x3f1c, 0x0340, 0x1d08, 0x3550,
 0x2930, 0x3e1c, 0x0240, 0x1d0c, 0x3554,
 0x2830, 0x3d1c, 0x0140, 0x1c0c, 0x3454,
 0x2730, 0x3c1c, 0x0040, 0x1b0c, 0x3354,
 0x2630, 0x3c20, 0x0044, 0x1a0c, 0x3254,
 0x2530, 0x3d20, 0x0144, 0x190c, 0x3154,
 0x2430, 0x3e20, 0x0244, 0x180c, 0x3054,
 0x2434, 0x3f20, 0x0344, 0x1810, 0x3058,
 0x2534, 0x4020, 0x0444, 0x1910, 0x3258,
 0x2634, 0x4120, 0x0544, 0x1a10, 0x3458,
 0x2a24, 0x4510, 0x0934, 0x1e00, 0x3648,
 0x2b24, 0x4610, 0x0a34, 0x1f00, 0x3748,
 0x2c24, 0x4710, 0x0b34, 0x2000, 0x3848,
 0x2d24, 0x4714, 0x0b38, 0x2100, 0x3948,
 0x2e24, 0x4614, 0x0a38, 0x2200, 0x3a48,
 0x2f24, 0x4514, 0x0938, 0x2300, 0x3b48,
 0x2f28, 0x4414, 0x0838, 0x2304, 0x3b4c,
 0x2e28, 0x4314, 0x0738, 0x2204, 0x3a4c,
 0x2d28, 0x4214, 0x0638, 0x2104, 0x394c,
 0x2c28, 0x4218, 0x063c, 0x2004, 0x384c,
 0x2b28, 0x4318, 0x073c, 0x1f04, 0x374c,
 0x2a28, 0x4418, 0x083c, 0x1e04, 0x364c,
 0x2a2c, 0x4518, 0x093c, 0x1e08, 0x3650,
 0x2b2c, 0x4618, 0x0a3c, 0x1f08, 0x3750,
 0x2c2c, 0x4718, 0x0b3c, 0x2008, 0x3850,
 0x2d2c, 0x471c, 0x0b40, 0x2108, 0x3950,
 0x2e2c, 0x461c, 0x0a40, 0x2208, 0x3a50,
 0x2f2c, 0x451c, 0x0940, 0x2308, 0x3b50,
 0x2f30, 0x441c, 0x0840, 0x230c, 0x3b54,
 0x2e30, 0x431c, 0x0740, 0x220c, 0x3a54,
 0x2d30, 0x421c, 0x0640, 0x210c, 0x3954,
 0x2c30, 0x4220, 0x0644, 0x200c, 0x3854,
 0x2b30, 0x4320, 0x0744, 0x1f0c, 0x3754,
 0x2a30, 0x4420, 0x0844, 0x1e0c, 0x3654,
 0x2a34, 0x4520, 0x0944, 0x1e10, 0x3658,
 0x2b34, 0x4620, 0x0a44, 0x1f10, 0x3858,
 0x2c34, 0x4720, 0x0b44, 0x2010, 0x3a58,
 0x3024, 0x0310, 0x0f34, 0x2400, 0x3c48,
 0x3124, 0x0410, 0x1034, 0x2500, 0x3d48,
 0x3224, 0x0510, 0x1134, 0x2600, 0x3e48,
 0x3324, 0x0514, 0x1138, 0x2700, 0x3f48,
 0x3424, 0x0414, 0x1038, 0x2800, 0x4048,
 0x3524, 0x0314, 0x0f38, 0x2900, 0x4148,
 0x3528, 0x0214, 0x0e38, 0x2904, 0x414c,
 0x3428, 0x0114, 0x0d38, 0x2804, 0x404c,
 0x3328, 0x0014, 0x0c38, 0x2704, 0x3f4c,
 0x3228, 0x0018, 0x0c3c, 0x2604, 0x3e4c,
 0x3128, 0x0118, 0x0d3c, 0x2504, 0x3d4c,
 0x3028, 0x0218, 0x0e3c, 0x2404, 0x3c4c,
 0x302c, 0x0318, 0x0f3c, 0x2408, 0x3c50,
 0x312c, 0x0418, 0x103c, 0x2508, 0x3d50,
 0x322c, 0x0518, 0x113c, 0x2608, 0x3e50,
 0x332c, 0x051c, 0x1140, 0x2708, 0x3f50,
 0x342c, 0x041c, 0x1040, 0x2808, 0x4050,
 0x352c, 0x031c, 0x0f40, 0x2908, 0x4150,
 0x3530, 0x021c, 0x0e40, 0x290c, 0x4154,
 0x3430, 0x011c, 0x0d40, 0x280c, 0x4054,
 0x3330, 0x001c, 0x0c40, 0x270c, 0x3f54,
 0x3230, 0x0020, 0x0c44, 0x260c, 0x3e54,
 0x3130, 0x0120, 0x0d44, 0x250c, 0x3d54,
 0x3030, 0x0220, 0x0e44, 0x240c, 0x3c54,
 0x3034, 0x0320, 0x0f44, 0x2410, 0x3c58,
 0x3134, 0x0420, 0x1044, 0x2510, 0x3e58,
 0x3234, 0x0520, 0x1144, 0x2610, 0x4058,
 0x3624, 0x0910, 0x1534, 0x2a00, 0x4248,
 0x3724, 0x0a10, 0x1634, 0x2b00, 0x4348,
 0x3824, 0x0b10, 0x1734, 0x2c00, 0x4448,
 0x3924, 0x0b14, 0x1738, 0x2d00, 0x4548,
 0x3a24, 0x0a14, 0x1638, 0x2e00, 0x4648,
 0x3b24, 0x0914, 0x1538, 0x2f00, 0x4748,
 0x3b28, 0x0814, 0x1438, 0x2f04, 0x474c,
 0x3a28, 0x0714, 0x1338, 0x2e04, 0x464c,
 0x3928, 0x0614, 0x1238, 0x2d04, 0x454c,
 0x3828, 0x0618, 0x123c, 0x2c04, 0x444c,
 0x3728, 0x0718, 0x133c, 0x2b04, 0x434c,
 0x3628, 0x0818, 0x143c, 0x2a04, 0x424c,
 0x362c, 0x0918, 0x153c, 0x2a08, 0x4250,
 0x372c, 0x0a18, 0x163c, 0x2b08, 0x4350,
 0x382c, 0x0b18, 0x173c, 0x2c08, 0x4450,
 0x392c, 0x0b1c, 0x1740, 0x2d08, 0x4550,
 0x3a2c, 0x0a1c, 0x1640, 0x2e08, 0x4650,
 0x3b2c, 0x091c, 0x1540, 0x2f08, 0x4750,
 0x3b30, 0x081c, 0x1440, 0x2f0c, 0x4754,
 0x3a30, 0x071c, 0x1340, 0x2e0c, 0x4654,
 0x3930, 0x061c, 0x1240, 0x2d0c, 0x4554,
 0x3830, 0x0620, 0x1244, 0x2c0c, 0x4454,
 0x3730, 0x0720, 0x1344, 0x2b0c, 0x4354,
 0x3630, 0x0820, 0x1444, 0x2a0c, 0x4254,
 0x3634, 0x0920, 0x1544, 0x2a10, 0x4258,
 0x3734, 0x0a20, 0x1644, 0x2b10, 0x4458,
 0x3834, 0x0b20, 0x1744, 0x2c10, 0x4658,
 0x3c24, 0x0f10, 0x1b34, 0x3000, 0x0048,
 0x3d24, 0x1010, 0x1c34, 0x3100, 0x0148,
 0x3e24, 0x1110, 0x1d34, 0x3200, 0x0248,
 0x3f24, 0x1114, 0x1d38, 0x3300, 0x0348,
 0x4024, 0x1014, 0x1c38, 0x3400, 0x0448,
 0x4124, 0x0f14, 0x1b38, 0x3500, 0x0548,
 0x4128, 0x0e14, 0x1a38, 0x3504, 0x054c,
 0x4028, 0x0d14, 0x1938, 0x3404, 0x044c,
 0x3f28, 0x0c14, 0x1838, 0x3304, 0x034c,
 0x3e28, 0x0c18, 0x183c, 0x3204, 0x024c,
 0x3d28, 0x0d18, 0x193c, 0x3104, 0x014c,
 0x3c28, 0x0e18, 0x1a3c, 0x3004, 0x004c,
 0x3c2c, 0x0f18, 0x1b3c, 0x3008, 0x0050,
 0x3d2c, 0x1018, 0x1c3c, 0x3108, 0x0150,
 0x3e2c, 0x1118, 0x1d3c, 0x3208, 0x0250,
 0x3f2c, 0x111c, 0x1d40, 0x3308, 0x0350,
 0x402c, 0x101c, 0x1c40, 0x3408, 0x0450,
 0x412c, 0x0f1c, 0x1b40, 0x3508, 0x0550,
 0x4130, 0x0e1c, 0x1a40, 0x350c, 0x0554,
 0x4030, 0x0d1c, 0x1940, 0x340c, 0x0454,
 0x3f30, 0x0c1c, 0x1840, 0x330c, 0x0354,
 0x3e30, 0x0c20, 0x1844, 0x320c, 0x0254,
 0x3d30, 0x0d20, 0x1944, 0x310c, 0x0154,
 0x3c30, 0x0e20, 0x1a44, 0x300c, 0x0054,
 0x3c34, 0x0f20, 0x1b44, 0x3010, 0x0058,
 0x3d34, 0x1020, 0x1c44, 0x3110, 0x0258,
 0x3e34, 0x1120, 0x1d44, 0x3210, 0x0458,
 0x4224, 0x1510, 0x2134, 0x3600, 0x0648,
 0x4324, 0x1610, 0x2234, 0x3700, 0x0748,
 0x4424, 0x1710, 0x2334, 0x3800, 0x0848,
 0x4524, 0x1714, 0x2338, 0x3900, 0x0948,
 0x4624, 0x1614, 0x2238, 0x3a00, 0x0a48,
 0x4724, 0x1514, 0x2138, 0x3b00, 0x0b48,
 0x4728, 0x1414, 0x2038, 0x3b04, 0x0b4c,
 0x4628, 0x1314, 0x1f38, 0x3a04, 0x0a4c,
 0x4528, 0x1214, 0x1e38, 0x3904, 0x094c,
 0x4428, 0x1218, 0x1e3c, 0x3804, 0x084c,
 0x4328, 0x1318, 0x1f3c, 0x3704, 0x074c,
 0x4228, 0x1418, 0x203c, 0x3604, 0x064c,
 0x422c, 0x1518, 0x213c, 0x3608, 0x0650,
 0x432c, 0x1618, 0x223c, 0x3708, 0x0750,
 0x442c, 0x1718, 0x233c, 0x3808, 0x0850,
 0x452c, 0x171c, 0x2340, 0x3908, 0x0950,
 0x462c, 0x161c, 0x2240, 0x3a08, 0x0a50,
 0x472c, 0x151c, 0x2140, 0x3b08, 0x0b50,
 0x4730, 0x141c, 0x2040, 0x3b0c, 0x0b54,
 0x4630, 0x131c, 0x1f40, 0x3a0c, 0x0a54,
 0x4530, 0x121c, 0x1e40, 0x390c, 0x0954,
 0x4430, 0x1220, 0x1e44, 0x380c, 0x0854,
 0x4330, 0x1320, 0x1f44, 0x370c, 0x0754,
 0x4230, 0x1420, 0x2044, 0x360c, 0x0654,
 0x4234, 0x1520, 0x2144, 0x3610, 0x0658,
 0x4334, 0x1620, 0x2244, 0x3710, 0x0858,
 0x4434, 0x1720, 0x2344, 0x3810, 0x0a58,
 0x0024, 0x1b10, 0x2734, 0x3c00, 0x0c48,
 0x0124, 0x1c10, 0x2834, 0x3d00, 0x0d48,
 0x0224, 0x1d10, 0x2934, 0x3e00, 0x0e48,
 0x0324, 0x1d14, 0x2938, 0x3f00, 0x0f48,
 0x0424, 0x1c14, 0x2838, 0x4000, 0x1048,
 0x0524, 0x1b14, 0x2738, 0x4100, 0x1148,
 0x0528, 0x1a14, 0x2638, 0x4104, 0x114c,
 0x0428, 0x1914, 0x2538, 0x4004, 0x104c,
 0x0328, 0x1814, 0x2438, 0x3f04, 0x0f4c,
 0x0228, 0x1818, 0x243c, 0x3e04, 0x0e4c,
 0x0128, 0x1918, 0x253c, 0x3d04, 0x0d4c,
 0x0028, 0x1a18, 0x263c, 0x3c04, 0x0c4c,
 0x002c, 0x1b18, 0x273c, 0x3c08, 0x0c50,
 0x012c, 0x1c18, 0x283c, 0x3d08, 0x0d50,
 0x022c, 0x1d18, 0x293c, 0x3e08, 0x0e50,
 0x032c, 0x1d1c, 0x2940, 0x3f08, 0x0f50,
 0x042c, 0x1c1c, 0x2840, 0x4008, 0x1050,
 0x052c, 0x1b1c, 0x2740, 0x4108, 0x1150,
 0x0530, 0x1a1c, 0x2640, 0x410c, 0x1154,
 0x0430, 0x191c, 0x2540, 0x400c, 0x1054,
 0x0330, 0x181c, 0x2440, 0x3f0c, 0x0f54,
 0x0230, 0x1820, 0x2444, 0x3e0c, 0x0e54,
 0x0130, 0x1920, 0x2544, 0x3d0c, 0x0d54,
 0x0030, 0x1a20, 0x2644, 0x3c0c, 0x0c54,
 0x0034, 0x1b20, 0x2744, 0x3c10, 0x0c58,
 0x0134, 0x1c20, 0x2844, 0x3d10, 0x0e58,
 0x0234, 0x1d20, 0x2944, 0x3e10, 0x1058,
 0x0624, 0x2110, 0x2d34, 0x4200, 0x1248,
 0x0724, 0x2210, 0x2e34, 0x4300, 0x1348,
 0x0824, 0x2310, 0x2f34, 0x4400, 0x1448,
 0x0924, 0x2314, 0x2f38, 0x4500, 0x1548,
 0x0a24, 0x2214, 0x2e38, 0x4600, 0x1648,
 0x0b24, 0x2114, 0x2d38, 0x4700, 0x1748,
 0x0b28, 0x2014, 0x2c38, 0x4704, 0x174c,
 0x0a28, 0x1f14, 0x2b38, 0x4604, 0x164c,
 0x0928, 0x1e14, 0x2a38, 0x4504, 0x154c,
 0x0828, 0x1e18, 0x2a3c, 0x4404, 0x144c,
 0x0728, 0x1f18, 0x2b3c, 0x4304, 0x134c,
 0x0628, 0x2018, 0x2c3c, 0x4204, 0x124c,
 0x062c, 0x2118, 0x2d3c, 0x4208, 0x1250,
 0x072c, 0x2218, 0x2e3c, 0x4308, 0x1350,
 0x082c, 0x2318, 0x2f3c, 0x4408, 0x1450,
 0x092c, 0x231c, 0x2f40, 0x4508, 0x1550,
 0x0a2c, 0x221c, 0x2e40, 0x4608, 0x1650,
 0x0b2c, 0x211c, 0x2d40, 0x4708, 0x1750,
 0x0b30, 0x201c, 0x2c40, 0x470c, 0x1754,
 0x0a30, 0x1f1c, 0x2b40, 0x460c, 0x1654,
 0x0930, 0x1e1c, 0x2a40, 0x450c, 0x1554,
 0x0830, 0x1e20, 0x2a44, 0x440c, 0x1454,
 0x0730, 0x1f20, 0x2b44, 0x430c, 0x1354,
 0x0630, 0x2020, 0x2c44, 0x420c, 0x1254,
 0x0634, 0x2120, 0x2d44, 0x4210, 0x1258,
 0x0734, 0x2220, 0x2e44, 0x4310, 0x1458,
 0x0834, 0x2320, 0x2f44, 0x4410, 0x1658,
};

static const uint16_t dv_place_411[1350] = {
 0x0c24, 0x2710, 0x3334, 0x0000, 0x1848,
 0x0d24, 0x2810, 0x3434, 0x0100, 0x1948,
 0x0e24, 0x2910, 0x3534, 0x0200, 0x1a48,
 0x0f24, 0x2914, 0x3538, 0x0300, 0x1b48,
 0x1024, 0x2814, 0x3438, 0x0400, 0x1c48,
 0x1124, 0x2714, 0x3338, 0x0500, 0x1d48,
 0x1128, 0x2614, 0x3238, 0x0504, 0x1d4c,
 0x1028, 0x2514, 0x3138, 0x0404, 0x1c4c,
 0x0f28, 0x2414, 0x3038, 0x0304, 0x1b4c,
 0x0e28, 0x2418, 0x303c, 0x0204, 0x1a4c,
 0x0d28, 0x2518, 0x313c, 0x0104, 0x194c,
 0x0c28, 0x2618, 0x323c, 0x0004, 0x184c,
 0x0c2c, 0x2718, 0x333c, 0x0008, 0x1850,
 0x0d2c, 0x2818, 0x343c, 0x0108, 0x1950,
 0x0e2c, 0x2918, 0x353c, 0x0208, 0x1a50,
 0x0f2c, 0x291c, 0x3540, 0x0308, 0x1b50,
 0x102c, 0x281c, 0x3440, 0x0408, 0x1c50,
 0x112c, 0x271c, 0x3340, 0x0508, 0x1d50,
 0x1130, 0x261c, 0x3240, 0x050c, 0x1d54,
 0x1030, 0x251c, 0x3140, 0x040c, 0x1c54,
 0x0f30, 0x241c, 0x3040, 0x030c, 0x1b54,
 0x0e30, 0x2420, 0x3044, 0x020c, 0x1a54,
 0x0d30, 0x2520, 0x3144, 0x010c, 0x1954,
 0x0c30, 0x2620, 0x3244, 0x000c, 0x1854,
 0x0c34, 0x2720, 0x3344, 0x0010, 0x1858,
 0x0d34, 0x2820, 0x3444, 0x0110, 0x1a58,
 0x0e34, 0x2920, 0x3544, 0x0210, 0x1c58,
 0x1224, 0x2d10, 0x3934, 0x0600, 0x1e48,
 0x1324, 0x2e10, 0x3a34, 0x0700, 0x1f48,
 0x1424, 0x2f10, 0x3b34, 0x0800, 0x2048,
 0x1524, 0x2f14, 0x3b38, 0x0900, 0x2148,
 0x1624, 0x2e14, 0x3a38, 0x0a00, 0x2248,
 0x1724, 0x2d14, 0x3938, 0x0b00, 0x2348,
 0x1728, 0x2c14, 0x3838, 0x0b04, 0x234c,
 0x1628, 0x2b14, 0x3738, 0x0a04, 0x224c,
 0x1528, 0x2a14, 0x3638, 0x0904, 0x214c,
 0x1428, 0x2a18, 0x363c, 0x0804, 0x204c,
 0x1328, 0x2b18, 0x373c, 0x0704, 0x1f4c,
 0x1228, 0x2c18, 0x383c, 0x0604, 0x1e4c,
 0x122c, 0x2d18, 0x393c, 0x0608, 0x1e50,
 0x132c, 0x2e18, 0x3a3c, 0x0708, 0x1f50,
 0x142c, 0x2f18, 0x3b3c, 0x0808, 0x2050,
 0x152c, 0x2f1c, 0x3b40, 0x0908, 0x2150,
 0x162c, 0x2e1c, 0x3a40, 0x0a08, 0x2250,
 0x172c, 0x2d1c, 0x3940, 0x0b08, 0x2350,
 0x1730, 0x2c1c, 0x3840, 0x0b0c, 0x2354,
 0x1630, 0x2b1c, 0x3740, 0x0a0c, 0x2254,
 0x1530, 0x2a1c, 0x3640, 0x090c, 0x2154,
 0x1430, 0x2a20, 0x3644, 0x080c, 0x2054,
 0x1330, 0x2b20, 0x3744, 0x070c, 0x1f54,
 0x1230, 0x2c20, 0x3844, 0x060c, 0x1e54,
 0x1234, 0x2d20, 0x3944, 0x0610, 0x1e58,
 0x1334, 0x2e20, 0x3a44, 0x0710, 0x2058,
 0x1434, 0x2f20, 0x3b44, 0x0810, 0x2258,
 0x1824, 0x3310, 0x0334, 0x0c00, 0x2448,
 0x1924, 0x3410, 0x0434, 0x0d00, 0x2548,
 0x1a24, 0x3510, 0x0534, 0x0e00, 0x2648,
 0x1b24, 0x3514, 0x0538, 0x0f00, 0x2748,
 0x1c24, 0x3414, 0x0438, 0x1000, 0x2848,
 0x1d24, 0x3314, 0x0338, 0x1100, 0x2948,
 0x1d28, 0x3214, 0x0238, 0x1104, 0x294c,
 0x1c28, 0x3114, 0x0138, 0x1004, 0x284c,
 0x1b28, 0x3014, 0x0038, 0x0f04, 0x274c,
 0x1a28, 0x3018, 0x003c, 0x0e04, 0x264c,
 0x1928, 0x3118, 0x013c, 0x0d04, 0x254c,
 0x1828, 0x3218, 0x023c, 0x0c04, 0x244c,
 0x182c, 0x3318, 0x033c, 0x0c08, 0x2450,
 0x192c, 0x3418, 0x043c, 0x0d08, 0x2550,
 0x1a2c, 0x3518, 0x053c, 0x0e08, 0x2650,
 0x1b2c, 0x351c, 0x0540, 0x0f08, 0x2750,
 0x1c2c, 0x341c, 0x0440, 0x1008, 0x2850,
 0x1d2c, 0x331c, 0x0340, 0x1108, 0x2950,
 0x1d30, 0x321c, 0x0240, 0x110c, 0x2954,
 0x1c30, 0x311c, 0x0140, 0x100c, 0x2854,
 0x1b30, 0x301c, 0x0040, 0x0f0c, 0x2754,
 0x1a30, 0x3020, 0x0044, 0x0e0c, 0x2654,
 0x1930, 0x3120, 0x0144, 0x0d0c, 0x2554,
 0x1830, 0x3220, 0x0244, 0x0c0c, 0x2454,
 0x1834, 0x3320, 0x0344, 0x0c10, 0x2458,
 0x1934, 0x3420, 0x0444, 0x0d10, 0x2658,
 0x1a34, 0x3520, 0x0544, 0x0e10, 0x2858,
 0x1e24, 0x3910, 0x0934, 0x1200, 0x2a48,
 0x1f24, 0x3a10, 0x0a34, 0x1300, 0x2b48,
 0x2024, 0x3b10, 0x0b34, 0x1400, 0x2c48,
 0x2124, 0x3b14, 0x0b38, 0x1500, 0x2d48,
 0x2224, 0x3a14, 0x0a38, 0x1600, 0x2e48,
 0x2324, 0x3914, 0x0938, 0x1700, 0x2f48,
 0x2328, 0x3814, 0x0838, 0x1704, 0x2f4c,
 0x2228, 0x3714, 0x0738, 0x1604, 0x2e4c,
 0x2128, 0x3614, 0x0638, 0x1504, 0x2d4c,
 0x2028, 0x3618, 0x063c, 0x1404, 0x2c4c,
 0x1f28, 0x3718, 0x073c, 0x1304, 0x2b4c,
 0x1e28, 0x3818, 0x083c, 0x1204, 0x2a4c,
 0x1e2c, 0x3918, 0x093c, 0x1208, 0x2a50,
 0x1f2c, 0x3a18, 0x0a3c, 0x1308, 0x2b50,
 0x202c, 0x3b18, 0x0b3c, 0x1408, 0x2c50,
 0x212c, 0x3b1c, 0x0b40, 0x1508, 0x2d50,
 0x222c, 0x3a1c, 0x0a40, 0x1608, 0x2e50,
 0x232c, 0x391c, 0x0940, 0x1708, 0x2f50,
 0x2330, 0x381c, 0x0840, 0x170c, 0x2f54,
 0x2230, 0x371c, 0x0740, 0x160c, 0x2e54,
 0x2130, 0x361c, 0x0640, 0x150c, 0x2d54,
 0x2030, 0x3620, 0x0644, 0x140c, 0x2c54,
 0x1f30, 0x3720, 0x0744, 0x130c, 0x2b54,
 0x1e30, 0x3820, 0x0844, 0x120c, 0x2a54,
 0x1e34, 0x3920, 0x0944, 0x1210, 0x2a58,
 0x1f34, 0x3a20, 0x0a44, 0x1310, 0x2c58,
 0x2034, 0x3b20, 0x0b44, 0x1410, 0x2e58,
 0x2424, 0x0310, 0x0f34, 0x1800, 0x3048,
 0x2524, 0x0410, 0x1034, 0x1900, 0x3148,
 0x2624, 0x0510, 0x1134, 0x1a00, 0x3248,
 0x2724, 0x0514, 0x1138, 0x1b00, 0x3348,
 0x2824, 0x0414, 0x1038, 0x1c00, 0x3448,
 0x2924, 0x0314, 0x0f38, 0x1d00, 0x3548,
 0x2928, 0x0214, 0x0e38, 0x1d04, 0x354c,
 0x2828, 0x0114, 0x0d38, 0x1c04, 0x344c,
 0x2728, 0x0014, 0x0c38, 0x1b04, 0x334c,
 0x2628, 0x0018, 0x0c3c, 0x1a04, 0x324c,
 0x2528, 0x0118, 0x0d3c, 0x1904, 0x314c,
 0x2428, 0x0218, 0x0e3c, 0x1804, 0x304c,
 0x242c, 0x0318, 0x0f3c, 0x1808, 0x3050,
 0x252c, 0x0418, 0x103c, 0x1908, 0x3150,
 0x262c, 0x0518, 0x113c, 0x1a08, 0x3250,
 0x272c, 0x051c, 0x1140, 0x1b08, 0x3350,
 0x282c, 0x041c, 0x1040, 0x1c08, 0x3450,
 0x292c, 0x031c, 0x0f40, 0x1d08, 0x3550,
 0x2930, 0x021c, 0x0e40, 0x1d0c, 0x3554,
 0x2830, 0x011c, 0x0d40, 0x1c0c, 0x3454,
 0x2730, 0x001c, 0x0c40, 0x1b0c, 0x3354,
 0x2630, 0x0020, 0x0c44, 0x1a0c, 0x3254,
 0x2530, 0x0120, 0x0d44, 0x190c, 0x3154,
 0x2430, 0x0220, 0x0e44, 0x180c, 0x3054,
 0x2434, 0x0320, 0x0f44, 0x1810, 0x3058,
 0x2534, 0x0420, 0x1044, 0x1910, 0x3258,
 0x2634, 0x0520, 0x1144, 0x1a10, 0x3458,
 0x2a24, 0x0910, 0x1534, 0x1e00, 0x3648,
 0x2b24, 0x0a10, 0x1634, 0x1f00, 0x3748,
 0x2c24, 0x0b10, 0x1734, 0x2000, 0x3848,
 0x2d24, 0x0b14, 0x1738, 0x2100, 0x3948,
 0x2e24, 0x0a14, 0x1638, 0x2200, 0x3a48,
 0x2f24, 0x0914, 0x1538, 0x2300, 0x3b48,
 0x2f28, 0x0814, 0x1438, 0x2304, 0x3b4c,
 0x2e28, 0x0714, 0x1338, 0x2204, 0x3a4c,
 0x2d28, 0x0614, 0x1238, 0x2104, 0x394c,
 0x2c28, 0x0618, 0x123c, 0x2004, 0x384c,
 0x2b28, 0x0718, 0x133c, 0x1f04, 0x374c,
 0x2a28, 0x0818, 0x143c, 0x1e04, 0x364c,
 0x2a2c, 0x0918, 0x153c, 0x1e08, 0x3650,
 0x2b2c, 0x0a18, 0x163c, 0x1f08, 0x3750,
 0x2c2c, 0x0b18, 0x173c, 0x2008, 0x3850,
 0x2d2c, 0x0b1c, 0x1740, 0x2108, 0x3950,
 0x2e2c, 0x0a1c, 0x1640, 0x2208, 0x3a50,
 0x2f2c, 0x091c, 0x1540, 0x2308, 0x3b50,
 0x2f30, 0x081c, 0x1440, 0x230c, 0x3b54,
 0x2e30, 0x071c, 0x1340, 0x220c, 0x3a54,
 0x2d30, 0x061c, 0x1240, 0x210c, 0x3954,
 0x2c30, 0x0620, 0x1244, 0x200c, 0x3854,
 0x2b30, 0x0720, 0x1344, 0x1f0c, 0x3754,
 0x2a30, 0x0820, 0x1444, 0x1e0c, 0x3654,
 0x2a34, 0x0920, 0x1544, 0x1e10, 0x3658,
 0x2b34, 0x0a20, 0x1644, 0x1f10, 0x3858,
 0x2c34, 0x0b20, 0x1744, 0x2010, 0x3a58,
 0x3024, 0x0f10, 0x1b34, 0x2400, 0x0048,
 0x3124, 0x1010, 0x1c34, 0x2500, 0x0148,
 0x3224, 0x1110, 0x1d34, 0x2600, 0x0248,
 0x3324, 0x1114, 0x1d38, 0x2700, 0x0348,
 0x3424, 0x1014, 0x1c38, 0x2800, 0x0448,
 0x3524, 0x0f14, 0x1b38, 0x2900, 0x0548,
 0x3528, 0x0e14, 0x1a38, 0x2904, 0x054c,
 0x3428, 0x0d14, 0x1938, 0x2804, 0x044c,
 0x3328, 0x0c14, 0x1838, 0x2704, 0x034c,
 0x3228, 0x0c18, 0x183c, 0x2604, 0x024c,
 0x3128, 0x0d18, 0x193c, 0x2504, 0x014c,
 0x3028, 0x0e18, 0x1a3c, 0x2404, 0x004c,
 0x302c, 0x0f18, 0x1b3c, 0x2408, 0x0050,
 0x312c, 0x1018, 0x1c3c, 0x2508, 0x0150,
 0x322c, 0x1118, 0x1d3c, 0x2608, 0x0250,
 0x332c, 0x111c, 0x1d40, 0x2708, 0x0350,
 0x342c, 0x101c, 0x1c40, 0x2808, 0x0450,
 0x352c, 0x0f1c, 0x1b40, 0x2908, 0x0550,
 0x3530, 0x0e1c, 0x1a40, 0x290c, 0x0554,
 0x3430, 0x0d1c, 0x1940, 0x280c, 0x0454,
 0x3330, 0x0c1c, 0x1840, 0x270c, 0x0354,
 0x3230, 0x0c20, 0x1844, 0x260c, 0x0254,
 0x3130, 0x0d20, 0x1944, 0x250c, 0x0154,
 0x3030, 0x0e20, 0x1a44, 0x240c, 0x0054,
 0x3034, 0x0f20, 0x1b44, 0x2410, 0x0058,
 0x3134, 0x1020, 0x1c44, 0x2510, 0x0258,
 0x3234, 0x1120, 0x1d44, 0x2610, 0x0458,
 0x3624, 0x1510, 0x2134, 0x2a00, 0x0648,
 0x3724, 0x1610, 0x2234, 0x2b00, 0x0748,
 0x3824, 0x1710, 0x2334, 0x2c00, 0x0848,
 0x3924, 0x1714, 0x2338, 0x2d00, 0x0948,
 0x3a24, 0x1614, 0x2238, 0x2e00, 0x0a48,
 0x3b24, 0x1514, 0x2138, 0x2f00, 0x0b48,
 0x3b28, 0x1414, 0x2038, 0x2f04, 0x0b4c,
 0x3a28, 0x1314, 0x1f38, 0x2e04, 0x0a4c,
 0x3928, 0x1214, 0x1e38, 0x2d04, 0x094c,
 0x3828, 0x1218, 0x1e3c, 0x2c04, 0x084c,
 0x3728, 0x1318, 0x1f3c, 0x2b04, 0x074c,
 0x3628, 0x1418, 0x203c, 0x2a04, 0x064c,
 0x362c, 0x1518, 0x213c, 0x2a08, 0x0650,
 0x372c, 0x1618, 0x223c, 0x2b08, 0x0750,
 0x382c, 0x1718, 0x233c, 0x2c08, 0x0850,
 0x392c, 0x171c, 0x2340, 0x2d08, 0x0950,
 0x3a2c, 0x161c, 0x2240, 0x2e08, 0x0a50,
 0x3b2c, 0x151c, 0x2140, 0x2f08, 0x0b50,
 0x3b30, 0x141c, 0x2040, 0x2f0c, 0x0b54,
 0x3a30, 0x131c, 0x1f40, 0x2e0c, 0x0a54,
 0x3930, 0x121c, 0x1e40, 0x2d0c, 0x0954,
 0x3830, 0x1220, 0x1e44, 0x2c0c, 0x0854,
 0x3730, 0x1320, 0x1f44, 0x2b0c, 0x0754,
 0x3630, 0x1420, 0x2044, 0x2a0c, 0x0654,
 0x3634, 0x1520, 0x2144, 0x2a10, 0x0658,
 0x3734, 0x1620, 0x2244, 0x2b10, 0x0858,
 0x3834, 0x1720, 0x2344, 0x2c10, 0x0a58,
 0x0024, 0x1b10, 0x2734, 0x3000, 0x0c48,
 0x0124, 0x1c10, 0x2834, 0x3100, 0x0d48,
 0x0224, 0x1d10, 0x2934, 0x3200, 0x0e48,
 0x0324, 0x1d14, 0x2938, 0x3300, 0x0f48,
 0x0424, 0x1c14, 0x2838, 0x3400, 0x1048,
 0x0524, 0x1b14, 0x2738, 0x3500, 0x1148,
 0x0528, 0x1a14, 0x2638, 0x3504, 0x114c,
 0x0428, 0x1914, 0x2538, 0x3404, 0x104c,
 0x0328, 0x1814, 0x2438, 0x3304, 0x0f4c,
 0x0228, 0x1818, 0x243c, 0x3204, 0x0e4c,
 0x0128, 0x1918, 0x253c, 0x3104, 0x0d4c,
 0x0028, 0x1a18, 0x263c, 0x3004, 0x0c4c,
 0x002c, 0x1b18, 0x273c, 0x3008, 0x0c50,
 0x012c, 0x1c18, 0x283c, 0x3108, 0x0d50,
 0x022c, 0x1d18, 0x293c, 0x3208, 0x0e50,
 0x032c, 0x1d1c, 0x2940, 0x3308, 0x0f50,
 0x042c, 0x1c1c, 0x2840, 0x3408, 0x1050,
 0x052c, 0x1b1c, 0x2740, 0x3508, 0x1150,
 0x0530, 0x1a1c, 0x2640, 0x350c, 0x1154,
 0x0430, 0x191c, 0x2540, 0x340c, 0x1054,
 0x0330, 0x181c, 0x2440, 0x330c, 0x0f54,
 0x0230, 0x1820, 0x2444, 0x320c, 0x0e54,
 0x0130, 0x1920, 0x2544, 0x310c, 0x0d54,
 0x0030, 0x1a20, 0x2644, 0x300c, 0x0c54,
 0x0034, 0x1b20, 0x2744, 0x3010, 0x0c58,
 0x0134, 0x1c20, 0x2844, 0x3110, 0x0e58,
 0x0234, 0x1d20, 0x2944, 0x3210, 0x1058,
 0x0624, 0x2110, 0x2d34, 0x3600, 0x1248,
 0x0724, 0x2210, 0x2e34, 0x3700, 0x1348,
 0x0824, 0x2310, 0x2f34, 0x3800, 0x1448,
 0x0924, 0x2314, 0x2f38, 0x3900, 0x1548,
 0x0a24, 0x2214, 0x2e38, 0x3a00, 0x1648,
 0x0b24, 0x2114, 0x2d38, 0x3b00, 0x1748,
 0x0b28, 0x2014, 0x2c38, 0x3b04, 0x174c,
 0x0a28, 0x1f14, 0x2b38, 0x3a04, 0x164c,
 0x0928, 0x1e14, 0x2a38, 0x3904, 0x154c,
 0x0828, 0x1e18, 0x2a3c, 0x3804, 0x144c,
 0x0728, 0x1f18, 0x2b3c, 0x3704, 0x134c,
 0x0628, 0x2018, 0x2c3c, 0x3604, 0x124c,
 0x062c, 0x2118, 0x2d3c, 0x3608, 0x1250,
 0x072c, 0x2218, 0x2e3c, 0x3708, 0x1350,
 0x082c, 0x2318, 0x2f3c, 0x3808, 0x1450,
 0x092c, 0x231c, 0x2f40, 0x3908, 0x1550,
 0x0a2c, 0x221c, 0x2e40, 0x3a08, 0x1650,
 0x0b2c, 0x211c, 0x2d40, 0x3b08, 0x1750,
 0x0b30, 0x201c, 0x2c40, 0x3b0c, 0x1754,
 0x0a30, 0x1f1c, 0x2b40, 0x3a0c, 0x1654,
 0x0930, 0x1e1c, 0x2a40, 0x390c, 0x1554,
 0x0830, 0x1e20, 0x2a44, 0x380c, 0x1454,
 0x0730, 0x1f20, 0x2b44, 0x370c, 0x1354,
 0x0630, 0x2020, 0x2c44, 0x360c, 0x1254,
 0x0634, 0x2120, 0x2d44, 0x3610, 0x1258,
 0x0734, 0x2220, 0x2e44, 0x3710, 0x1458,
 0x0834, 0x2320, 0x2f44, 0x3810, 0x1658,
};

/* 2 channels per frame, 10 DIF sequences per channel,
   27 video segments per DIF sequence, 5 macroblocks per video segment */
static const uint16_t dv_place_422_525[2*10*27*5] = {
 0x0c24, 0x2412, 0x3036, 0x0000, 0x1848,
 0x0d24, 0x2512, 0x3136, 0x0100, 0x1948,
 0x0e24, 0x2612, 0x3236, 0x0200, 0x1a48,
 0x0e26, 0x2614, 0x3238, 0x0202, 0x1a4a,
 0x0d26, 0x2514, 0x3138, 0x0102, 0x194a,
 0x0c26, 0x2414, 0x3038, 0x0002, 0x184a,
 0x0c28, 0x2416, 0x303a, 0x0004, 0x184c,
 0x0d28, 0x2516, 0x313a, 0x0104, 0x194c,
 0x0e28, 0x2616, 0x323a, 0x0204, 0x1a4c,
 0x0e2a, 0x2618, 0x323c, 0x0206, 0x1a4e,
 0x0d2a, 0x2518, 0x313c, 0x0106, 0x194e,
 0x0c2a, 0x2418, 0x303c, 0x0006, 0x184e,
 0x0c2c, 0x241a, 0x303e, 0x0008, 0x1850,
 0x0d2c, 0x251a, 0x313e, 0x0108, 0x1950,
 0x0e2c, 0x261a, 0x323e, 0x0208, 0x1a50,
 0x0e2e, 0x261c, 0x3240, 0x020a, 0x1a52,
 0x0d2e, 0x251c, 0x3140, 0x010a, 0x1952,
 0x0c2e, 0x241c, 0x3040, 0x000a, 0x1852,
 0x0c30, 0x241e, 0x3042, 0x000c, 0x1854,
 0x0d30, 0x251e, 0x3142, 0x010c, 0x1954,
 0x0e30, 0x261e, 0x3242, 0x020c, 0x1a54,
 0x0e32, 0x2620, 0x3244, 0x020e, 0x1a56,
 0x0d32, 0x2520, 0x3144, 0x010e, 0x1956,
 0x0c32, 0x2420, 0x3044, 0x000e, 0x1856,
 0x0c34, 0x2422, 0x3046, 0x0010, 0x1858,
 0x0d34, 0x2522, 0x3146, 0x0110, 0x1958,
 0x0e34, 0x2622, 0x3246, 0x0210, 0x1a58,
 0x1224, 0x2a12, 0x3636, 0x0600, 0x1e48,
 0x1324, 0x2b12, 0x3736, 0x0700, 0x1f48,
 0x1424, 0x2c12, 0x3836, 0x0800, 0x2048,
 0x1426, 0x2c14, 0x3838, 0x0802, 0x204a,
 0x1326, 0x2b14, 0x3738, 0x0702, 0x1f4a,
 0x1226, 0x2a14, 0x3638, 0x0602, 0x1e4a,
 0x1228, 0x2a16, 0x363a, 0x0604, 0x1e4c,
 0x1328, 0x2b16, 0x373a, 0x0704, 0x1f4c,
 0x1428, 0x2c16, 0x383a, 0x0804, 0x204c,
 0x142a, 0x2c18, 0x383c, 0x0806, 0x204e,
 0x132a, 0x2b18, 0x373c, 0x0706, 0x1f4e,
 0x122a, 0x2a18, 0x363c, 0x0606, 0x1e4e,
 0x122c, 0x2a1a, 0x363e, 0x0608, 0x1e50,
 0x132c, 0x2b1a, 0x373e, 0x0708, 0x1f50,
 0x142c, 0x2c1a, 0x383e, 0x0808, 0x2050,
 0x142e, 0x2c1c, 0x3840, 0x080a, 0x2052,
 0x132e, 0x2b1c, 0x3740, 0x070a, 0x1f52,
 0x122e, 0x2a1c, 0x3640, 0x060a, 0x1e52,
 0x1230, 0x2a1e, 0x3642, 0x060c, 0x1e54,
 0x1330, 0x2b1e, 0x3742, 0x070c, 0x1f54,
 0x1430, 0x2c1e, 0x3842, 0x080c, 0x2054,
 0x1432, 0x2c20, 0x3844, 0x080e, 0x2056,
 0x1332, 0x2b20, 0x3744, 0x070e, 0x1f56,
 0x1232, 0x2a20, 0x3644, 0x060e, 0x1e56,
 0x1234, 0x2a22, 0x3646, 0x0610, 0x1e58,
 0x1334, 0x2b22, 0x3746, 0x0710, 0x1f58,
 0x1434, 0x2c22, 0x3846, 0x0810, 0x2058,
 0x1824, 0x3012, 0x0036, 0x0c00, 0x2448,
 0x1924, 0x3112, 0x0136, 0x0d00, 0x2548,
 0x1a24, 0x3212, 0x0236, 0x0e00, 0x2648,
 0x1a26, 0x3214, 0x0238, 0x0e02, 0x264a,
 0x1926, 0x3114, 0x0138, 0x0d02, 0x254a,
 0x1826, 0x3014, 0x0038, 0x0c02, 0x244a,
 0x1828, 0x3016, 0x003a, 0x0c04, 0x244c,
 0x1928, 0x3116, 0x013a, 0x0d04, 0x254c,
 0x1a28, 0x3216, 0x023a, 0x0e04, 0x264c,
 0x1a2a, 0x3218, 0x023c, 0x0e06, 0x264e,
 0x192a, 0x3118, 0x013c, 0x0d06, 0x254e,
 0x182a, 0x3018, 0x003c, 0x0c06, 0x244e,
 0x182c, 0x301a, 0x003e, 0x0c08, 0x2450,
 0x192c, 0x311a, 0x013e, 0x0d08, 0x2550,
 0x1a2c, 0x321a, 0x023e, 0x0e08, 0x2650,
 0x1a2e, 0x321c, 0x0240, 0x0e0a, 0x2652,
 0x192e, 0x311c, 0x0140, 0x0d0a, 0x2552,
 0x182e, 0x301c, 0x0040, 0x0c0a, 0x2452,
 0x1830, 0x301e, 0x0042, 0x0c0c, 0x2454,
 0x1930, 0x311e, 0x0142, 0x0d0c, 0x2554,
 0x1a30, 0x321e, 0x0242, 0x0e0c, 0x2654,
 0x1a32, 0x3220, 0x0244, 0x0e0e, 0x2656,
 0x1932, 0x3120, 0x0144, 0x0d0e, 0x2556,
 0x1832, 0x3020, 0x0044, 0x0c0e, 0x2456,
 0x1834, 0x3022, 0x0046, 0x0c10, 0x2458,
 0x1934, 0x3122, 0x0146, 0x0d10, 0x2558,
 0x1a34, 0x3222, 0x0246, 0x0e10, 0x2658,
 0x1e24, 0x3612, 0x0636, 0x1200, 0x2a48,
 0x1f24, 0x3712, 0x0736, 0x1300, 0x2b48,
 0x2024, 0x3812, 0x0836, 0x1400, 0x2c48,
 0x2026, 0x3814, 0x0838, 0x1402, 0x2c4a,
 0x1f26, 0x3714, 0x0738, 0x1302, 0x2b4a,
 0x1e26, 0x3614, 0x0638, 0x1202, 0x2a4a,
 0x1e28, 0x3616, 0x063a, 0x1204, 0x2a4c,
 0x1f28, 0x3716, 0x073a, 0x1304, 0x2b4c,
 0x2028, 0x3816, 0x083a, 0x1404, 0x2c4c,
 0x202a, 0x3818, 0x083c, 0x1406, 0x2c4e,
 0x1f2a, 0x3718, 0x073c, 0x1306, 0x2b4e,
 0x1e2a, 0x3618, 0x063c, 0x1206, 0x2a4e,
 0x1e2c, 0x361a, 0x063e, 0x1208, 0x2a50,
 0x1f2c, 0x371a, 0x073e, 0x1308, 0x2b50,
 0x202c, 0x381a, 0x083e, 0x1408, 0x2c50,
 0x202e, 0x381c, 0x0840, 0x140a, 0x2c52,
 0x1f2e, 0x371c, 0x0740, 0x130a, 0x2b52,
 0x1e2e, 0x361c, 0x0640, 0x120a, 0x2a52,
 0x1e30, 0x361e, 0x0642, 0x120c, 0x2a54,
 0x1f30, 0x371e, 0x0742, 0x130c, 0x2b54,
 0x2030, 0x381e, 0x0842, 0x140c, 0x2c54,
 0x2032, 0x3820, 0x0844, 0x140e, 0x2c56,
 0x1f32, 0x3720, 0x0744, 0x130e, 0x2b56,
 0x1e32, 0x3620, 0x0644, 0x120e, 0x2a56,
 0x1e34, 0x3622, 0x0646, 0x1210, 0x2a58,
 0x1f34, 0x3722, 0x0746, 0x1310, 0x2b58,
 0x2034, 0x3822, 0x0846, 0x1410, 0x2c58,
 0x2424, 0x0012, 0x0c36, 0x1800, 0x3048,
 0x2524, 0x0112, 0x0d36, 0x1900, 0x3148,
 0x2624, 0x0212, 0x0e36, 0x1a00, 0x3248,
 0x2626, 0x0214, 0x0e38, 0x1a02, 0x324a,
 0x2526, 0x0114, 0x0d38, 0x1902, 0x314a,
 0x2426, 0x0014, 0x0c38, 0x1802, 0x304a,
 0x2428, 0x0016, 0x0c3a, 0x1804, 0x304c,
 0x2528, 0x0116, 0x0d3a, 0x1904, 0x314c,
 0x2628, 0x0216, 0x0e3a, 0x1a04, 0x324c,
 0x262a, 0x0218, 0x0e3c, 0x1a06, 0x324e,
 0x252a, 0x0118, 0x0d3c, 0x1906, 0x314e,
 0x242a, 0x0018, 0x0c3c, 0x1806, 0x304e,
 0x242c, 0x001a, 0x0c3e, 0x1808, 0x3050,
 0x252c, 0x011a, 0x0d3e, 0x1908, 0x3150,
 0x262c, 0x021a, 0x0e3e, 0x1a08, 0x3250,
 0x262e, 0x021c, 0x0e40, 0x1a0a, 0x3252,
 0x252e, 0x011c, 0x0d40, 0x190a, 0x3152,
 0x242e, 0x001c, 0x0c40, 0x180a, 0x3052,
 0x2430, 0x001e, 0x0c42, 0x180c, 0x3054,
 0x2530, 0x011e, 0x0d42, 0x190c, 0x3154,
 0x2630, 0x021e, 0x0e42, 0x1a0c, 0x3254,
 0x2632, 0x0220, 0x0e44, 0x1a0e, 0x3256,
 0x2532, 0x0120, 0x0d44, 0x190e, 0x3156,
 0x2432, 0x0020, 0x0c44, 0x180e, 0x3056,
 0x2434, 0x0022, 0x0c46, 0x1810, 0x3058,
 0x2534, 0x0122, 0x0d46, 0x1910, 0x3158,
 0x2634, 0x0222, 0x0e46, 0x1a10, 0x3258,
 0x2a24, 0x0612, 0x1236, 0x1e00, 0x3648,
 0x2b24, 0x0712, 0x1336, 0x1f00, 0x3748,
 0x2c24, 0x0812, 0x1436, 0x2000, 0x3848,
 0x2c26, 0x0814, 0x1438, 0x2002, 0x384a,
 0x2b26, 0x0714, 0x1338, 0x1f02, 0x374a,
 0x2a26, 0x0614, 0x1238, 0x1e02, 0x364a,
 0x2a28, 0x0616, 0x123a, 0x1e04, 0x364c,
 0x2b28, 0x0716, 0x133a, 0x1f04, 0x374c,
 0x2c28, 0x0816, 0x143a, 0x2004, 0x384c,
 0x2c2a, 0x0818, 0x143c, 0x2006, 0x384e,
 0x2b2a, 0x0718, 0x133c, 0x1f06, 0x374e,
 0x2a2a, 0x0618, 0x123c, 0x1e06, 0x364e,
 0x2a2c, 0x061a, 0x123e, 0x1e08, 0x3650,
 0x2b2c, 0x071a, 0x133e, 0x1f08, 0x3750,
 0x2c2c, 0x081a, 0x143e, 0x2008, 0x3850,
 0x2c2e, 0x081c, 0x1440, 0x200a, 0x3852,
 0x2b2e, 0x071c, 0x1340, 0x1f0a, 0x3752,
 0x2a2e, 0x061c, 0x1240, 0x1e0a, 0x3652,
 0x2a30, 0x061e, 0x1242, 0x1e0c, 0x3654,
 0x2b30, 0x071e, 0x1342, 0x1f0c, 0x3754,
 0x2c30, 0x081e, 0x1442, 0x200c, 0x3854,
 0x2c32, 0x0820, 0x1444, 0x200e, 0x3856,
 0x2b32, 0x0720, 0x1344, 0x1f0e, 0x3756,
 0x2a32, 0x0620, 0x1244, 0x1e0e, 0x3656,
 0x2a34, 0x0622, 0x1246, 0x1e10, 0x3658,
 0x2b34, 0x0722, 0x1346, 0x1f10, 0x3758,
 0x2c34, 0x0822, 0x1446, 0x2010, 0x3858,
 0x3024, 0x0c12, 0x1836, 0x2400, 0x0048,
 0x3124, 0x0d12, 0x1936, 0x2500, 0x0148,
 0x3224, 0x0e12, 0x1a36, 0x2600, 0x0248,
 0x3226, 0x0e14, 0x1a38, 0x2602, 0x024a,
 0x3126, 0x0d14, 0x1938, 0x2502, 0x014a,
 0x3026, 0x0c14, 0x1838, 0x2402, 0x004a,
 0x3028, 0x0c16, 0x183a, 0x2404, 0x004c,
 0x3128, 0x0d16, 0x193a, 0x2504, 0x014c,
 0x3228, 0x0e16, 0x1a3a, 0x2604, 0x024c,
 0x322a, 0x0e18, 0x1a3c, 0x2606, 0x024e,
 0x312a, 0x0d18, 0x193c, 0x2506, 0x014e,
 0x302a, 0x0c18, 0x183c, 0x2406, 0x004e,
 0x302c, 0x0c1a, 0x183e, 0x2408, 0x0050,
 0x312c, 0x0d1a, 0x193e, 0x2508, 0x0150,
 0x322c, 0x0e1a, 0x1a3e, 0x2608, 0x0250,
 0x322e, 0x0e1c, 0x1a40, 0x260a, 0x0252,
 0x312e, 0x0d1c, 0x1940, 0x250a, 0x0152,
 0x302e, 0x0c1c, 0x1840, 0x240a, 0x0052,
 0x3030, 0x0c1e, 0x1842, 0x240c, 0x0054,
 0x3130, 0x0d1e, 0x1942, 0x250c, 0x0154,
 0x3230, 0x0e1e, 0x1a42, 0x260c, 0x0254,
 0x3232, 0x0e20, 0x1a44, 0x260e, 0x0256,
 0x3132, 0x0d20, 0x1944, 0x250e, 0x0156,
 0x3032, 0x0c20, 0x1844, 0x240e, 0x0056,
 0x3034, 0x0c22, 0x1846, 0x2410, 0x0058,
 0x3134, 0x0d22, 0x1946, 0x2510, 0x0158,
 0x3234, 0x0e22, 0x1a46, 0x2610, 0x0258,
 0x3624, 0x1212, 0x1e36, 0x2a00, 0x0648,
 0x3724, 0x1312, 0x1f36, 0x2b00, 0x0748,
 0x3824, 0x1412, 0x2036, 0x2c00, 0x0848,
 0x3826, 0x1414, 0x2038, 0x2c02, 0x084a,
 0x3726, 0x1314, 0x1f38, 0x2b02, 0x074a,
 0x3626, 0x1214, 0x1e38, 0x2a02, 0x064a,
 0x3628, 0x1216, 0x1e3a, 0x2a04, 0x064c,
 0x3728, 0x1316, 0x1f3a, 0x2b04, 0x074c,
 0x3828, 0x1416, 0x203a, 0x2c04, 0x084c,
 0x382a, 0x1418, 0x203c, 0x2c06, 0x084e,
 0x372a, 0x1318, 0x1f3c, 0x2b06, 0x074e,
 0x362a, 0x1218, 0x1e3c, 0x2a06, 0x064e,
 0x362c, 0x121a, 0x1e3e, 0x2a08, 0x0650,
 0x372c, 0x131a, 0x1f3e, 0x2b08, 0x0750,
 0x382c, 0x141a, 0x203e, 0x2c08, 0x0850,
 0x382e, 0x141c, 0x2040, 0x2c0a, 0x0852,
 0x372e, 0x131c, 0x1f40, 0x2b0a, 0x0752,
 0x362e, 0x121c, 0x1e40, 0x2a0a, 0x0652,
 0x3630, 0x121e, 0x1e42, 0x2a0c, 0x0654,
 0x3730, 0x131e, 0x1f42, 0x2b0c, 0x0754,
 0x3830, 0x141e, 0x2042, 0x2c0c, 0x0854,
 0x3832, 0x1420, 0x2044, 0x2c0e, 0x0856,
 0x3732, 0x1320, 0x1f44, 0x2b0e, 0x0756,
 0x3632, 0x1220, 0x1e44, 0x2a0e, 0x0656,
 0x3634, 0x1222, 0x1e46, 0x2a10, 0x0658,
 0x3734, 0x1322, 0x1f46, 0x2b10, 0x0758,
 0x3834, 0x1422, 0x2046, 0x2c10, 0x0858,
 0x0024, 0x1812, 0x2436, 0x3000, 0x0c48,
 0x0124, 0x1912, 0x2536, 0x3100, 0x0d48,
 0x0224, 0x1a12, 0x2636, 0x3200, 0x0e48,
 0x0226, 0x1a14, 0x2638, 0x3202, 0x0e4a,
 0x0126, 0x1914, 0x2538, 0x3102, 0x0d4a,
 0x0026, 0x1814, 0x2438, 0x3002, 0x0c4a,
 0x0028, 0x1816, 0x243a, 0x3004, 0x0c4c,
 0x0128, 0x1916, 0x253a, 0x3104, 0x0d4c,
 0x0228, 0x1a16, 0x263a, 0x3204, 0x0e4c,
 0x022a, 0x1a18, 0x263c, 0x3206, 0x0e4e,
 0x012a, 0x1918, 0x253c, 0x3106, 0x0d4e,
 0x002a, 0x1818, 0x243c, 0x3006, 0x0c4e,
 0x002c, 0x181a, 0x243e, 0x3008, 0x0c50,
 0x012c, 0x191a, 0x253e, 0x3108, 0x0d50,
 0x022c, 0x1a1a, 0x263e, 0x3208, 0x0e50,
 0x022e, 0x1a1c, 0x2640, 0x320a, 0x0e52,
 0x012e, 0x191c, 0x2540, 0x310a, 0x0d52,
 0x002e, 0x181c, 0x2440, 0x300a, 0x0c52,
 0x0030, 0x181e, 0x2442, 0x300c, 0x0c54,
 0x0130, 0x191e, 0x2542, 0x310c, 0x0d54,
 0x0230, 0x1a1e, 0x2642, 0x320c, 0x0e54,
 0x0232, 0x1a20, 0x2644, 0x320e, 0x0e56,
 0x0132, 0x1920, 0x2544, 0x310e, 0x0d56,
 0x0032, 0x1820, 0x2444, 0x300e, 0x0c56,
 0x0034, 0x1822, 0x2446, 0x3010, 0x0c58,
 0x0134, 0x1922, 0x2546, 0x3110, 0x0d58,
 0x0234, 0x1a22, 0x2646, 0x3210, 0x0e58,
 0x0624, 0x1e12, 0x2a36, 0x3600, 0x1248,
 0x0724, 0x1f12, 0x2b36, 0x3700, 0x1348,
 0x0824, 0x2012, 0x2c36, 0x3800, 0x1448,
 0x0826, 0x2014, 0x2c38, 0x3802, 0x144a,
 0x0726, 0x1f14, 0x2b38, 0x3702, 0x134a,
 0x0626, 0x1e14, 0x2a38, 0x3602, 0x124a,
 0x0628, 0x1e16, 0x2a3a, 0x3604, 0x124c,
 0x0728, 0x1f16, 0x2b3a, 0x3704, 0x134c,
 0x0828, 0x2016, 0x2c3a, 0x3804, 0x144c,
 0x082a, 0x2018, 0x2c3c, 0x3806, 0x144e,
 0x072a, 0x1f18, 0x2b3c, 0x3706, 0x134e,
 0x062a, 0x1e18, 0x2a3c, 0x3606, 0x124e,
 0x062c, 0x1e1a, 0x2a3e, 0x3608, 0x1250,
 0x072c, 0x1f1a, 0x2b3e, 0x3708, 0x1350,
 0x082c, 0x201a, 0x2c3e, 0x3808, 0x1450,
 0x082e, 0x201c, 0x2c40, 0x380a, 0x1452,
 0x072e, 0x1f1c, 0x2b40, 0x370a, 0x1352,
 0x062e, 0x1e1c, 0x2a40, 0x360a, 0x1252,
 0x0630, 0x1e1e, 0x2a42, 0x360c, 0x1254,
 0x0730, 0x1f1e, 0x2b42, 0x370c, 0x1354,
 0x0830, 0x201e, 0x2c42, 0x380c, 0x1454,
 0x0832, 0x2020, 0x2c44, 0x380e, 0x1456,
 0x0732, 0x1f20, 0x2b44, 0x370e, 0x1356,
 0x0632, 0x1e20, 0x2a44, 0x360e, 0x1256,
 0x0634, 0x1e22, 0x2a46, 0x3610, 0x1258,
 0x0734, 0x1f22, 0x2b46, 0x3710, 0x1358,
 0x0834, 0x2022, 0x2c46, 0x3810, 0x1458,
 0x0f24, 0x2712, 0x3336, 0x0300, 0x1b48,
 0x1024, 0x2812, 0x3436, 0x0400, 0x1c48,
 0x1124, 0x2912, 0x3536, 0x0500, 0x1d48,
 0x1126, 0x2914, 0x3538, 0x0502, 0x1d4a,
 0x1026, 0x2814, 0x3438, 0x0402, 0x1c4a,
 0x0f26, 0x2714, 0x3338, 0x0302, 0x1b4a,
 0x0f28, 0x2716, 0x333a, 0x0304, 0x1b4c,
 0x1028, 0x2816, 0x343a, 0x0404, 0x1c4c,
 0x1128, 0x2916, 0x353a, 0x0504, 0x1d4c,
 0x112a, 0x2918, 0x353c, 0x0506, 0x1d4e,
 0x102a, 0x2818, 0x343c, 0x0406, 0x1c4e,
 0x0f2a, 0x2718, 0x333c, 0x0306, 0x1b4e,
 0x0f2c, 0x271a, 0x333e, 0x0308, 0x1b50,
 0x102c, 0x281a, 0x343e, 0x0408, 0x1c50,
 0x112c, 0x291a, 0x353e, 0x0508, 0x1d50,
 0x112e, 0x291c, 0x3540, 0x050a, 0x1d52,
 0x102e, 0x281c, 0x3440, 0x040a, 0x1c52,
 0x0f2e, 0x271c, 0x3340, 0x030a, 0x1b52,
 0x0f30, 0x271e, 0x3342, 0x030c, 0x1b54,
 0x1030, 0x281e, 0x3442, 0x040c, 0x1c54,
 0x1130, 0x291e, 0x3542, 0x050c, 0x1d54,
 0x1132, 0x2920, 0x3544, 0x050e, 0x1d56,
 0x1032, 0x2820, 0x3444, 0x040e, 0x1c56,
 0x0f32, 0x2720, 0x3344, 0x030e, 0x1b56,
 0x0f34, 0x2722, 0x3346, 0x0310, 0x1b58,
 0x1034, 0x2822, 0x3446, 0x0410, 0x1c58,
 0x1134, 0x2922, 0x3546, 0x0510, 0x1d58,
 0x1524, 0x2d12, 0x3936, 0x0900, 0x2148,
 0x1624, 0x2e12, 0x3a36, 0x0a00, 0x2248,
 0x1724, 0x2f12, 0x3b36, 0x0b00, 0x2348,
 0x1726, 0x2f14, 0x3b38, 0x0b02, 0x234a,
 0x1626, 0x2e14, 0x3a38, 0x0a02, 0x224a,
 0x1526, 0x2d14, 0x3938, 0x0902, 0x214a,
 0x1528, 0x2d16, 0x393a, 0x0904, 0x214c,
 0x1628, 0x2e16, 0x3a3a, 0x0a04, 0x224c,
 0x1728, 0x2f16, 0x3b3a, 0x0b04, 0x234c,
 0x172a, 0x2f18, 0x3b3c, 0x0b06, 0x234e,
 0x162a, 0x2e18, 0x3a3c, 0x0a06, 0x224e,
 0x152a, 0x2d18, 0x393c, 0x0906, 0x214e,
 0x152c, 0x2d1a, 0x393e, 0x0908, 0x2150,
 0x162c, 0x2e1a, 0x3a3e, 0x0a08, 0x2250,
 0x172c, 0x2f1a, 0x3b3e, 0x0b08, 0x2350,
 0x172e, 0x2f1c, 0x3b40, 0x0b0a, 0x2352,
 0x162e, 0x2e1c, 0x3a40, 0x0a0a, 0x2252,
 0x152e, 0x2d1c, 0x3940, 0x090a, 0x2152,
 0x1530, 0x2d1e, 0x3942, 0x090c, 0x2154,
 0x1630, 0x2e1e, 0x3a42, 0x0a0c, 0x2254,
 0x1730, 0x2f1e, 0x3b42, 0x0b0c, 0x2354,
 0x1732, 0x2f20, 0x3b44, 0x0b0e, 0x2356,
 0x1632, 0x2e20, 0x3a44, 0x0a0e, 0x2256,
 0x1532, 0x2d20, 0x3944, 0x090e, 0x2156,
 0x1534, 0x2d22, 0x3946, 0x0910, 0x2158,
 0x1634, 0x2e22, 0x3a46, 0x0a10, 0x2258,
 0x1734, 0x2f22, 0x3b46, 0x0b10, 0x2358,
 0x1b24, 0x3312, 0x0336, 0x0f00, 0x2748,
 0x1c24, 0x3412, 0x0436, 0x1000, 0x2848,
 0x1d24, 0x3512, 0x0536, 0x1100, 0x2948,
 0x1d26, 0x3514, 0x0538, 0x1102, 0x294a,
 0x1c26, 0x3414, 0x0438, 0x1002, 0x284a,
 0x1b26, 0x3314, 0x0338, 0x0f02, 0x274a,
 0x1b28, 0x3316, 0x033a, 0x0f04, 0x274c,
 0x1c28, 0x3416, 0x043a, 0x1004, 0x284c,
 0x1d28, 0x3516, 0x053a, 0x1104, 0x294c,
 0x1d2a, 0x3518, 0x053c, 0x1106, 0x294e,
 0x1c2a, 0x3418, 0x043c, 0x1006, 0x284e,
 0x1b2a, 0x3318, 0x033c, 0x0f06, 0x274e,
 0x1b2c, 0x331a, 0x033e, 0x0f08, 0x2750,
 0x1c2c, 0x341a, 0x043e, 0x1008, 0x2850,
 0x1d2c, 0x351a, 0x053e, 0x1108, 0x2950,
 0x1d2e, 0x351c, 0x0540, 0x110a, 0x2952,
 0x1c2e, 0x341c, 0x0440, 0x100a, 0x2852,
 0x1b2e, 0x331c, 0x0340, 0x0f0a, 0x2752,
 0x1b30, 0x331e, 0x0342, 0x0f0c, 0x2754,
 0x1c30, 0x341e, 0x0442, 0x100c, 0x2854,
 0x1d30, 0x351e, 0x0542, 0x110c, 0x2954,
 0x1d32, 0x3520, 0x0544, 0x110e, 0x2956,
 0x1c32, 0x3420, 0x0444, 0x100e, 0x2856,
 0x1b32, 0x3320, 0x0344, 0x0f0e, 0x2756,
 0x1b34, 0x3322, 0x0346, 0x0f10, 0x2758,
 0x1c34, 0x3422, 0x0446, 0x1010, 0x2858,
 0x1d34, 0x3522, 0x0546, 0x1110, 0x2958,
 0x2124, 0x3912, 0x0936, 0x1500, 0x2d48,
 0x2224, 0x3a12, 0x0a36, 0x1600, 0x2e48,
 0x2324, 0x3b12, 0x0b36, 0x1700, 0x2f48,
 0x2326, 0x3b14, 0x0b38, 0x1702, 0x2f4a,
 0x2226, 0x3a14, 0x0a38, 0x1602, 0x2e4a,
 0x2126, 0x3914, 0x0938, 0x1502, 0x2d4a,
 0x2128, 0x3916, 0x093a, 0x1504, 0x2d4c,
 0x2228, 0x3a16, 0x0a3a, 0x1604, 0x2e4c,
 0x2328, 0x3b16, 0x0b3a, 0x1704, 0x2f4c,
 0x232a, 0x3b18, 0x0b3c, 0x1706, 0x2f4e,
 0x222a, 0x3a18, 0x0a3c, 0x1606, 0x2e4e,
 0x212a, 0x3918, 0x093c, 0x1506, 0x2d4e,
 0x212c, 0x391a, 0x093e, 0x1508, 0x2d50,
 0x222c, 0x3a1a, 0x0a3e, 0x1608, 0x2e50,
 0x232c, 0x3b1a, 0x0b3e, 0x1708, 0x2f50,
 0x232e, 0x3b1c, 0x0b40, 0x170a, 0x2f52,
 0x222e, 0x3a1c, 0x0a40, 0x160a, 0x2e52,
 0x212e, 0x391c, 0x0940, 0x150a, 0x2d52,
 0x2130, 0x391e, 0x0942, 0x150c, 0x2d54,
 0x2230, 0x3a1e, 0x0a42, 0x160c, 0x2e54,
 0x2330, 0x3b1e, 0x0b42, 0x170c, 0x2f54,
 0x2332, 0x3b20, 0x0b44, 0x170e, 0x2f56,
 0x2232, 0x3a20, 0x0a44, 0x160e, 0x2e56,
 0x2132, 0x3920, 0x0944, 0x150e, 0x2d56,
 0x2134, 0x3922, 0x0946, 0x1510, 0x2d58,
 0x2234, 0x3a22, 0x0a46, 0x1610, 0x2e58,
 0x2334, 0x3b22, 0x0b46, 0x1710, 0x2f58,
 0x2724, 0x0312, 0x0f36, 0x1b00, 0x3348,
 0x2824, 0x0412, 0x1036, 0x1c00, 0x3448,
 0x2924, 0x0512, 0x1136, 0x1d00, 0x3548,
 0x2926, 0x0514, 0x1138, 0x1d02, 0x354a,
 0x2826, 0x0414, 0x1038, 0x1c02, 0x344a,
 0x2726, 0x0314, 0x0f38, 0x1b02, 0x334a,
 0x2728, 0x0316, 0x0f3a, 0x1b04, 0x334c,
 0x2828, 0x0416, 0x103a, 0x1c04, 0x344c,
 0x2928, 0x0516, 0x113a, 0x1d04, 0x354c,
 0x292a, 0x0518, 0x113c, 0x1d06, 0x354e,
 0x282a, 0x0418, 0x103c, 0x1c06, 0x344e,
 0x272a, 0x0318, 0x0f3c, 0x1b06, 0x334e,
 0x272c, 0x031a, 0x0f3e, 0x1b08, 0x3350,
 0x282c, 0x041a, 0x103e, 0x1c08, 0x3450,
 0x292c, 0x051a, 0x113e, 0x1d08, 0x3550,
 0x292e, 0x051c, 0x1140, 0x1d0a, 0x3552,
 0x282e, 0x041c, 0x1040, 0x1c0a, 0x3452,
 0x272e, 0x031c, 0x0f40, 0x1b0a, 0x3352,
 0x2730, 0x031e, 0x0f42, 0x1b0c, 0x3354,
 0x2830, 0x041e, 0x1042, 0x1c0c, 0x3454,
 0x2930, 0x051e, 0x1142, 0x1d0c, 0x3554,
 0x2932, 0x0520, 0x1144, 0x1d0e, 0x3556,
 0x2832, 0x0420, 0x1044, 0x1c0e, 0x3456,
 0x2732, 0x0320, 0x0f44, 0x1b0e, 0x3356,
 0x2734, 0x0322, 0x0f46, 0x1b10, 0x3358,
 0x2834, 0x0422, 0x1046, 0x1c10, 0x3458,
 0x2934, 0x0522, 0x1146, 0x1d10, 0x3558,
 0x2d24, 0x0912, 0x1536, 0x2100, 0x3948,
 0x2e24, 0x0a12, 0x1636, 0x2200, 0x3a48,
 0x2f24, 0x0b12, 0x1736, 0x2300, 0x3b48,
 0x2f26, 0x0b14, 0x1738, 0x2302, 0x3b4a,
 0x2e26, 0x0a14, 0x1638, 0x2202, 0x3a4a,
 0x2d26, 0x0914, 0x1538, 0x2102, 0x394a,
 0x2d28, 0x0916, 0x153a, 0x2104, 0x394c,
 0x2e28, 0x0a16, 0x163a, 0x2204, 0x3a4c,
 0x2f28, 0x0b16, 0x173a, 0x2304, 0x3b4c,
 0x2f2a, 0x0b18, 0x173c, 0x2306, 0x3b4e,
 0x2e2a, 0x0a18, 0x163c, 0x2206, 0x3a4e,
 0x2d2a, 0x0918, 0x153c, 0x2106, 0x394e,
 0x2d2c, 0x091a, 0x153e, 0x2108, 0x3950,
 0x2e2c, 0x0a1a, 0x163e, 0x2208, 0x3a50,
 0x2f2c, 0x0b1a, 0x173e, 0x2308, 0x3b50,
 0x2f2e, 0x0b1c, 0x1740, 0x230a, 0x3b52,
 0x2e2e, 0x0a1c, 0x1640, 0x220a, 0x3a52,
 0x2d2e, 0x091c, 0x1540, 0x210a, 0x3952,
 0x2d30, 0x091e, 0x1542, 0x210c, 0x3954,
 0x2e30, 0x0a1e, 0x1642, 0x220c, 0x3a54,
 0x2f30, 0x0b1e, 0x1742, 0x230c, 0x3b54,
 0x2f32, 0x0b20, 0x1744, 0x230e, 0x3b56,
 0x2e32, 0x0a20, 0x1644, 0x220e, 0x3a56,
 0x2d32, 0x0920, 0x1544, 0x210e, 0x3956,
 0x2d34, 0x0922, 0x1546, 0x2110, 0x3958,
 0x2e34, 0x0a22, 0x1646, 0x2210, 0x3a58,
 0x2f34, 0x0b22, 0x1746, 0x2310, 0x3b58,
 0x3324, 0x0f12, 0x1b36, 0x2700, 0x0348,
 0x3424, 0x1012, 0x1c36, 0x2800, 0x0448,
 0x3524, 0x1112, 0x1d36, 0x2900, 0x0548,
 0x3526, 0x1114, 0x1d38, 0x2902, 0x054a,
 0x3426, 0x1014, 0x1c38, 0x2802, 0x044a,
 0x3326, 0x0f14, 0x1b38, 0x2702, 0x034a,
 0x3328, 0x0f16, 0x1b3a, 0x2704, 0x034c,
 0x3428, 0x1016, 0x1c3a, 0x2804, 0x044c,
 0x3528, 0x1116, 0x1d3a, 0x2904, 0x054c,
 0x352a, 0x1118, 0x1d3c, 0x2906, 0x054e,
 0x342a, 0x1018, 0x1c3c, 0x2806, 0x044e,
 0x332a, 0x0f18, 0x1b3c, 0x2706, 0x034e,
 0x332c, 0x0f1a, 0x1b3e, 0x2708, 0x0350,
 0x342c, 0x101a, 0x1c3e, 0x2808, 0x0450,
 0x352c, 0x111a, 0x1d3e, 0x2908, 0x0550,
 0x352e, 0x111c, 0x1d40, 0x290a, 0x0552,
 0x342e, 0x101c, 0x1c40, 0x280a, 0x0452,
 0x332e, 0x0f1c, 0x1b40, 0x270a, 0x0352,
 0x3330, 0x0f1e, 0x1b42, 0x270c, 0x0354,
 0x3430, 0x101e, 0x1c42, 0x280c, 0x0454,
 0x3530, 0x111e, 0x1d42, 0x290c, 0x0554,
 0x3532, 0x1120, 0x1d44, 0x290e, 0x0556,
 0x3432, 0x1020, 0x1c44, 0x280e, 0x0456,
 0x3332, 0x0f20, 0x1b44, 0x270e, 0x0356,
 0x3334, 0x0f22, 0x1b46, 0x2710, 0x0358,
 0x3434, 0x1022, 0x1c46, 0x2810, 0x0458,
 0x3534, 0x1122, 0x1d46, 0x2910, 0x0558,
 0x3924, 0x1512, 0x2136, 0x2d00, 0x0948,
 0x3a24, 0x1612, 0x2236, 0x2e00, 0x0a48,
 0x3b24, 0x1712, 0x2336, 0x2f00, 0x0b48,
 0x3b26, 0x1714, 0x2338, 0x2f02, 0x0b4a,
 0x3a26, 0x1614, 0x2238, 0x2e02, 0x0a4a,
 0x3926, 0x1514, 0x2138, 0x2d02, 0x094a,
 0x3928, 0x1516, 0x213a, 0x2d04, 0x094c,
 0x3a28, 0x1616, 0x223a, 0x2e04, 0x0a4c,
 0x3b28, 0x1716, 0x233a, 0x2f04, 0x0b4c,
 0x3b2a, 0x1718, 0x233c, 0x2f06, 0x0b4e,
 0x3a2a, 0x1618, 0x223c, 0x2e06, 0x0a4e,
 0x392a, 0x1518, 0x213c, 0x2d06, 0x094e,
 0x392c, 0x151a, 0x213e, 0x2d08, 0x0950,
 0x3a2c, 0x161a, 0x223e, 0x2e08, 0x0a50,
 0x3b2c, 0x171a, 0x233e, 0x2f08, 0x0b50,
 0x3b2e, 0x171c, 0x2340, 0x2f0a, 0x0b52,
 0x3a2e, 0x161c, 0x2240, 0x2e0a, 0x0a52,
 0x392e, 0x151c, 0x2140, 0x2d0a, 0x0952,
 0x3930, 0x151e, 0x2142, 0x2d0c, 0x0954,
 0x3a30, 0x161e, 0x2242, 0x2e0c, 0x0a54,
 0x3b30, 0x171e, 0x2342, 0x2f0c, 0x0b54,
 0x3b32, 0x1720, 0x2344, 0x2f0e, 0x0b56,
 0x3a32, 0x1620, 0x2244, 0x2e0e, 0x0a56,
 0x3932, 0x1520, 0x2144, 0x2d0e, 0x0956,
 0x3934, 0x1522, 0x2146, 0x2d10, 0x0958,
 0x3a34, 0x1622, 0x2246, 0x2e10, 0x0a58,
 0x3b34, 0x1722, 0x2346, 0x2f10, 0x0b58,
 0x0324, 0x1b12, 0x2736, 0x3300, 0x0f48,
 0x0424, 0x1c12, 0x2836, 0x3400, 0x1048,
 0x0524, 0x1d12, 0x2936, 0x3500, 0x1148,
 0x0526, 0x1d14, 0x2938, 0x3502, 0x114a,
 0x0426, 0x1c14, 0x2838, 0x3402, 0x104a,
 0x0326, 0x1b14, 0x2738, 0x3302, 0x0f4a,
 0x0328, 0x1b16, 0x273a, 0x3304, 0x0f4c,
 0x0428, 0x1c16, 0x283a, 0x3404, 0x104c,
 0x0528, 0x1d16, 0x293a, 0x3504, 0x114c,
 0x052a, 0x1d18, 0x293c, 0x3506, 0x114e,
 0x042a, 0x1c18, 0x283c, 0x3406, 0x104e,
 0x032a, 0x1b18, 0x273c, 0x3306, 0x0f4e,
 0x032c, 0x1b1a, 0x273e, 0x3308, 0x0f50,
 0x042c, 0x1c1a, 0x283e, 0x3408, 0x1050,
 0x052c, 0x1d1a, 0x293e, 0x3508, 0x1150,
 0x052e, 0x1d1c, 0x2940, 0x350a, 0x1152,
 0x042e, 0x1c1c, 0x2840, 0x340a, 0x1052,
 0x032e, 0x1b1c, 0x2740, 0x330a, 0x0f52,
 0x0330, 0x1b1e, 0x2742, 0x330c, 0x0f54,
 0x0430, 0x1c1e, 0x2842, 0x340c, 0x1054,
 0x0530, 0x1d1e, 0x2942, 0x350c, 0x1154,
 0x0532, 0x1d20, 0x2944, 0x350e, 0x1156,
 0x0432, 0x1c20, 0x2844, 0x340e, 0x1056,
 0x0332, 0x1b20, 0x2744, 0x330e, 0x0f56,
 0x0334, 0x1b22, 0x2746, 0x3310, 0x0f58,
 0x0434, 0x1c22, 0x2846, 0x3410, 0x1058,
 0x0534, 0x1d22, 0x2946, 0x3510, 0x1158,
 0x0924, 0x2112, 0x2d36, 0x3900, 0x1548,
 0x0a24, 0x2212, 0x2e36, 0x3a00, 0x1648,
 0x0b24, 0x2312, 0x2f36, 0x3b00, 0x1748,
 0x0b26, 0x2314, 0x2f38, 0x3b02, 0x174a,
 0x0a26, 0x2214, 0x2e38, 0x3a02, 0x164a,
 0x0926, 0x2114, 0x2d38, 0x3902, 0x154a,
 0x0928, 0x2116, 0x2d3a, 0x3904, 0x154c,
 0x0a28, 0x2216, 0x2e3a, 0x3a04, 0x164c,
 0x0b28, 0x2316, 0x2f3a, 0x3b04, 0x174c,
 0x0b2a, 0x2318, 0x2f3c, 0x3b06, 0x174e,
 0x0a2a, 0x2218, 0x2e3c, 0x3a06, 0x164e,
 0x092a, 0x2118, 0x2d3c, 0x3906, 0x154e,
 0x092c, 0x211a, 0x2d3e, 0x3908, 0x1550,
 0x0a2c, 0x221a, 0x2e3e, 0x3a08, 0x1650,
 0x0b2c, 0x231a, 0x2f3e, 0x3b08, 0x1750,
 0x0b2e, 0x231c, 0x2f40, 0x3b0a, 0x1752,
 0x0a2e, 0x221c, 0x2e40, 0x3a0a, 0x1652,
 0x092e, 0x211c, 0x2d40, 0x390a, 0x1552,
 0x0930, 0x211e, 0x2d42, 0x390c, 0x1554,
 0x0a30, 0x221e, 0x2e42, 0x3a0c, 0x1654,
 0x0b30, 0x231e, 0x2f42, 0x3b0c, 0x1754,
 0x0b32, 0x2320, 0x2f44, 0x3b0e, 0x1756,
 0x0a32, 0x2220, 0x2e44, 0x3a0e, 0x1656,
 0x0932, 0x2120, 0x2d44, 0x390e, 0x1556,
 0x0934, 0x2122, 0x2d46, 0x3910, 0x1558,
 0x0a34, 0x2222, 0x2e46, 0x3a10, 0x1658,
 0x0b34, 0x2322, 0x2f46, 0x3b10, 0x1758,
};

/* 2 channels per frame, 12 DIF sequences per channel,
   27 video segments per DIF sequence, 5 macroblocks per video segment */
static const uint16_t dv_place_422_625[2*12*27*5] = {
 0x0c24, 0x2412, 0x3036, 0x0000, 0x1848,
 0x0d24, 0x2512, 0x3136, 0x0100, 0x1948,
 0x0e24, 0x2612, 0x3236, 0x0200, 0x1a48,
 0x0e26, 0x2614, 0x3238, 0x0202, 0x1a4a,
 0x0d26, 0x2514, 0x3138, 0x0102, 0x194a,
 0x0c26, 0x2414, 0x3038, 0x0002, 0x184a,
 0x0c28, 0x2416, 0x303a, 0x0004, 0x184c,
 0x0d28, 0x2516, 0x313a, 0x0104, 0x194c,
 0x0e28, 0x2616, 0x323a, 0x0204, 0x1a4c,
 0x0e2a, 0x2618, 0x323c, 0x0206, 0x1a4e,
 0x0d2a, 0x2518, 0x313c, 0x0106, 0x194e,
 0x0c2a, 0x2418, 0x303c, 0x0006, 0x184e,
 0x0c2c, 0x241a, 0x303e, 0x0008, 0x1850,
 0x0d2c, 0x251a, 0x313e, 0x0108, 0x1950,
 0x0e2c, 0x261a, 0x323e, 0x0208, 0x1a50,
 0x0e2e, 0x261c, 0x3240, 0x020a, 0x1a52,
 0x0d2e, 0x251c, 0x3140, 0x010a, 0x1952,
 0x0c2e, 0x241c, 0x3040, 0x000a, 0x1852,
 0x0c30, 0x241e, 0x3042, 0x000c, 0x1854,
 0x0d30, 0x251e, 0x3142, 0x010c, 0x1954,
 0x0e30, 0x261e, 0x3242, 0x020c, 0x1a54,
 0x0e32, 0x2620, 0x3244, 0x020e, 0x1a56,
 0x0d32, 0x2520, 0x3144, 0x010e, 0x1956,
 0x0c32, 0x2420, 0x3044, 0x000e, 0x1856,
 0x0c34, 0x2422, 0x3046, 0x0010, 0x1858,
 0x0d34, 0x2522, 0x3146, 0x0110, 0x1958,
 0x0e34, 0x2622, 0x3246, 0x0210, 0x1a58,
 0x1224, 0x2a12, 0x3636, 0x0600, 0x1e48,
 0x1324, 0x2b12, 0x3736, 0x0700, 0x1f48,
 0x1424, 0x2c12, 0x3836, 0x0800, 0x2048,
 0x1426, 0x2c14, 0x3838, 0x0802, 0x204a,
 0x1326, 0x2b14, 0x3738, 0x0702, 0x1f4a,
 0x1226, 0x2a14, 0x3638, 0x0602, 0x1e4a,
 0x1228, 0x2a16, 0x363a, 0x0604, 0x1e4c,
 0x1328, 0x2b16, 0x373a, 0x0704, 0x1f4c,
 0x1428, 0x2c16, 0x383a, 0x0804, 0x204c,
 0x142a, 0x2c18, 0x383c, 0x0806, 0x204e,
 0x132a, 0x2b18, 0x373c, 0x0706, 0x1f4e,
 0x122a, 0x2a18, 0x363c, 0x0606, 0x1e4e,
 0x122c, 0x2a1a, 0x363e, 0x0608, 0x1e50,
 0x132c, 0x2b1a, 0x373e, 0x0708, 0x1f50,
 0x142c, 0x2c1a, 0x383e, 0x0808, 0x2050,
 0x142e, 0x2c1c, 0x3840, 0x080a, 0x2052,
 0x132e, 0x2b1c, 0x3740, 0x070a, 0x1f52,
 0x122e, 0x2a1c, 0x3640, 0x060a, 0x1e52,
 0x1230, 0x2a1e, 0x3642, 0x060c, 0x1e54,
 0x1330, 0x2b1e, 0x3742, 0x070c, 0x1f54,
 0x1430, 0x2c1e, 0x3842, 0x080c, 0x2054,
 0x1432, 0x2c20, 0x3844, 0x080e, 0x2056,
 0x1332, 0x2b20, 0x3744, 0x070e, 0x1f56,
 0x1232, 0x2a20, 0x3644, 0x060e, 0x1e56,
 0x1234, 0x2a22, 0x3646, 0x0610, 0x1e58,
 0x1334, 0x2b22, 0x3746, 0x0710, 0x1f58,
 0x1434, 0x2c22, 0x3846, 0x0810, 0x2058,
 0x1824, 0x3012, 0x3c36, 0x0c00, 0x2448,
 0x1924, 0x3112, 0x3d36, 0x0d00, 0x2548,
 0x1a24, 0x3212, 0x3e36, 0x0e00, 0x2648,
 0x1a26, 0x3214, 0x3e38, 0x0e02, 0x264a,
 0x1926, 0x3114, 0x3d38, 0x0d02, 0x254a,
 0x1826, 0x3014, 0x3c38, 0x0c02, 0x244a,
 0x1828, 0x3016, 0x3c3a, 0x0c04, 0x244c,
 0x1928, 0x3116, 0x3d3a, 0x0d04, 0x254c,
 0x1a28, 0x3216, 0x3e3a, 0x0e04, 0x264c,
 0x1a2a, 0x3218, 0x3e3c, 0x0e06, 0x264e,
 0x192a, 0x3118, 0x3d3c, 0x0d06, 0x254e,
 0x182a, 0x3018, 0x3c3c, 0x0c06, 0x244e,
 0x182c, 0x301a, 0x3c3e, 0x0c08, 0x2450,
 0x192c, 0x311a, 0x3d3e, 0x0d08, 0x2550,
 0x1a2c, 0x321a, 0x3e3e, 0x0e08, 0x2650,
 0x1a2e, 0x321c, 0x3e40, 0x0e0a, 0x2652,
 0x192e, 0x311c, 0x3d40, 0x0d0a, 0x2552,
 0x182e, 0x301c, 0x3c40, 0x0c0a, 0x2452,
 0x1830, 0x301e, 0x3c42, 0x0c0c, 0x2454,
 0x1930, 0x311e, 0x3d42, 0x0d0c, 0x2554,
 0x1a30, 0x321e, 0x3e42, 0x0e0c, 0x2654,
 0x1a32, 0x3220, 0x3e44, 0x0e0e, 0x2656,
 0x1932, 0x3120, 0x3d44, 0x0d0e, 0x2556,
 0x1832, 0x3020, 0x3c44, 0x0c0e, 0x2456,
 0x1834, 0x3022, 0x3c46, 0x0c10, 0x2458,
 0x1934, 0x3122, 0x3d46, 0x0d10, 0x2558,
 0x1a34, 0x3222, 0x3e46, 0x0e10, 0x2658,
 0x1e24, 0x3612, 0x4236, 0x1200, 0x2a48,
 0x1f24, 0x3712, 0x4336, 0x1300, 0x2b48,
 0x2024, 0x3812, 0x4436, 0x1400, 0x2c48,
 0x2026, 0x3814, 0x4438, 0x1402, 0x2c4a,
 0x1f26, 0x3714, 0x4338, 0x1302, 0x2b4a,
 0x1e26, 0x3614, 0x4238, 0x1202, 0x2a4a,
 0x1e28, 0x3616, 0x423a, 0x1204, 0x2a4c,
 0x1f28, 0x3716, 0x433a, 0x1304, 0x2b4c,
 0x2028, 0x3816, 0x443a, 0x1404, 0x2c4c,
 0x202a, 0x3818, 0x443c, 0x1406, 0x2c4e,
 0x1f2a, 0x3718, 0x433c, 0x1306, 0x2b4e,
 0x1e2a, 0x3618, 0x423c, 0x1206, 0x2a4e,
 0x1e2c, 0x361a, 0x423e, 0x1208, 0x2a50,
 0x1f2c, 0x371a, 0x433e, 0x1308, 0x2b50,
 0x202c, 0x381a, 0x443e, 0x1408, 0x2c50,
 0x202e, 0x381c, 0x4440, 0x140a, 0x2c52,
 0x1f2e, 0x371c, 0x4340, 0x130a, 0x2b52,
 0x1e2e, 0x361c, 0x4240, 0x120a, 0x2a52,
 0x1e30, 0x361e, 0x4242, 0x120c, 0x2a54,
 0x1f30, 0x371e, 0x4342, 0x130c, 0x2b54,
 0x2030, 0x381e, 0x4442, 0x140c, 0x2c54,
 0x2032, 0x3820, 0x4444, 0x140e, 0x2c56,
 0x1f32, 0x3720, 0x4344, 0x130e, 0x2b56,
 0x1e32, 0x3620, 0x4244, 0x120e, 0x2a56,
 0x1e34, 0x3622, 0x4246, 0x1210, 0x2a58,
 0x1f34, 0x3722, 0x4346, 0x1310, 0x2b58,
 0x2034, 0x3822, 0x4446, 0x1410, 0x2c58,
 0x2424, 0x3c12, 0x0036, 0x1800, 0x3048,
 0x2524, 0x3d12, 0x0136, 0x1900, 0x3148,
 0x2624, 0x3e12, 0x0236, 0x1a00, 0x3248,
 0x2626, 0x3e14, 0x0238, 0x1a02, 0x324a,
 0x2526, 0x3d14, 0x0138, 0x1902, 0x314a,
 0x2426, 0x3c14, 0x0038, 0x1802, 0x304a,
 0x2428, 0x3c16, 0x003a, 0x1804, 0x304c,
 0x2528, 0x3d16, 0x013a, 0x1904, 0x314c,
 0x2628, 0x3e16, 0x023a, 0x1a04, 0x324c,
 0x262a, 0x3e18, 0x023c, 0x1a06, 0x324e,
 0x252a, 0x3d18, 0x013c, 0x1906, 0x314e,
 0x242a, 0x3c18, 0x003c, 0x1806, 0x304e,
 0x242c, 0x3c1a, 0x003e, 0x1808, 0x3050,
 0x252c, 0x3d1a, 0x013e, 0x1908, 0x3150,
 0x262c, 0x3e1a, 0x023e, 0x1a08, 0x3250,
 0x262e, 0x3e1c, 0x0240, 0x1a0a, 0x3252,
 0x252e, 0x3d1c, 0x0140, 0x190a, 0x3152,
 0x242e, 0x3c1c, 0x0040, 0x180a, 0x3052,
 0x2430, 0x3c1e, 0x0042, 0x180c, 0x3054,
 0x2530, 0x3d1e, 0x0142, 0x190c, 0x3154,
 0x2630, 0x3e1e, 0x0242, 0x1a0c, 0x3254,
 0x2632, 0x3e20, 0x0244, 0x1a0e, 0x3256,
 0x2532, 0x3d20, 0x0144, 0x190e, 0x3156,
 0x2432, 0x3c20, 0x0044, 0x180e, 0x3056,
 0x2434, 0x3c22, 0x0046, 0x1810, 0x3058,
 0x2534, 0x3d22, 0x0146, 0x1910, 0x3158,
 0x2634, 0x3e22, 0x0246, 0x1a10, 0x3258,
 0x2a24, 0x4212, 0x0636, 0x1e00, 0x3648,
 0x2b24, 0x4312, 0x0736, 0x1f00, 0x3748,
 0x2c24, 0x4412, 0x0836, 0x2000, 0x3848,
 0x2c26, 0x4414, 0x0838, 0x2002, 0x384a,
 0x2b26, 0x4314, 0x0738, 0x1f02, 0x374a,
 0x2a26, 0x4214, 0x0638, 0x1e02, 0x364a,
 0x2a28, 0x4216, 0x063a, 0x1e04, 0x364c,
 0x2b28, 0x4316, 0x073a, 0x1f04, 0x374c,
 0x2c28, 0x4416, 0x083a, 0x2004, 0x384c,
 0x2c2a, 0x4418, 0x083c, 0x2006, 0x384e,
 0x2b2a, 0x4318, 0x073c, 0x1f06, 0x374e,
 0x2a2a, 0x4218, 0x063c, 0x1e06, 0x364e,
 0x2a2c, 0x421a, 0x063e, 0x1e08, 0x3650,
 0x2b2c, 0x431a, 0x073e, 0x1f08, 0x3750,
 0x2c2c, 0x441a, 0x083e, 0x2008, 0x3850,
 0x2c2e, 0x441c, 0x0840, 0x200a, 0x3852,
 0x2b2e, 0x431c, 0x0740, 0x1f0a, 0x3752,
 0x2a2e, 0x421c, 0x0640, 0x1e0a, 0x3652,
 0x2a30, 0x421e, 0x0642, 0x1e0c, 0x3654,
 0x2b30, 0x431e, 0x0742, 0x1f0c, 0x3754,
 0x2c30, 0x441e, 0x0842, 0x200c, 0x3854,
 0x2c32, 0x4420, 0x0844, 0x200e, 0x3856,
 0x2b32, 0x4320, 0x0744, 0x1f0e, 0x3756,
 0x2a32, 0x4220, 0x0644, 0x1e0e, 0x3656,
 0x2a34, 0x4222, 0x0646, 0x1e10, 0x3658,
 0x2b34, 0x4322, 0x0746, 0x1f10, 0x3758,
 0x2c34, 0x4422, 0x0846, 0x2010, 0x3858,
 0x3024, 0x0012, 0x0c36, 0x2400, 0x3c48,
 0x3124, 0x0112, 0x0d36, 0x2500, 0x3d48,
 0x3224, 0x0212, 0x0e36, 0x2600, 0x3e48,
 0x3226, 0x0214, 0x0e38, 0x2602, 0x3e4a,
 0x3126, 0x0114, 0x0d38, 0x2502, 0x3d4a,
 0x3026, 0x0014, 0x0c38, 0x2402, 0x3c4a,
 0x3028, 0x0016, 0x0c3a, 0x2404, 0x3c4c,
 0x3128, 0x0116, 0x0d3a, 0x2504, 0x3d4c,
 0x3228, 0x0216, 0x0e3a, 0x2604, 0x3e4c,
 0x322a, 0x0218, 0x0e3c, 0x2606, 0x3e4e,
 0x312a, 0x0118, 0x0d3c, 0x2506, 0x3d4e,
 0x302a, 0x0018, 0x0c3c, 0x2406, 0x3c4e,
 0x302c, 0x001a, 0x0c3e, 0x2408, 0x3c50,
 0x312c, 0x011a, 0x0d3e, 0x2508, 0x3d50,
 0x322c, 0x021a, 0x0e3e, 0x2608, 0x3e50,
 0x322e, 0x021c, 0x0e40, 0x260a, 0x3e52,
 0x312e, 0x011c, 0x0d40, 0x250a, 0x3d52,
 0x302e, 0x001c, 0x0c40, 0x240a, 0x3c52,
 0x3030, 0x001e, 0x0c42, 0x240c, 0x3c54,
 0x3130, 0x011e, 0x0d42, 0x250c, 0x3d54,
 0x3230, 0x021e, 0x0e42, 0x260c, 0x3e54,
 0x3232, 0x0220, 0x0e44, 0x260e, 0x3e56,
 0x3132, 0x0120, 0x0d44, 0x250e, 0x3d56,
 0x3032, 0x0020, 0x0c44, 0x240e, 0x3c56,
 0x3034, 0x0022, 0x0c46, 0x2410, 0x3c58,
 0x3134, 0x0122, 0x0d46, 0x2510, 0x3d58,
 0x3234, 0x0222, 0x0e46, 0x2610, 0x3e58,
 0x3624, 0x0612, 0x1236, 0x2a00, 0x4248,
 0x3724, 0x0712, 0x1336, 0x2b00, 0x4348,
 0x3824, 0x0812, 0x1436, 0x2c00, 0x4448,
 0x3826, 0x0814, 0x1438, 0x2c02, 0x444a,
 0x3726, 0x0714, 0x1338, 0x2b02, 0x434a,
 0x3626, 0x0614, 0x1238, 0x2a02, 0x424a,
 0x3628, 0x0616, 0x123a, 0x2a04, 0x424c,
 0x3728, 0x0716, 0x133a, 0x2b04, 0x434c,
 0x3828, 0x0816, 0x143a, 0x2c04, 0x444c,
 0x382a, 0x0818, 0x143c, 0x2c06, 0x444e,
 0x372a, 0x0718, 0x133c, 0x2b06, 0x434e,
 0x362a, 0x0618, 0x123c, 0x2a06, 0x424e,
 0x362c, 0x061a, 0x123e, 0x2a08, 0x4250,
 0x372c, 0x071a, 0x133e, 0x2b08, 0x4350,
 0x382c, 0x081a, 0x143e, 0x2c08, 0x4450,
 0x382e, 0x081c, 0x1440, 0x2c0a, 0x4452,
 0x372e, 0x071c, 0x1340, 0x2b0a, 0x4352,
 0x362e, 0x061c, 0x1240, 0x2a0a, 0x4252,
 0x3630, 0x061e, 0x1242, 0x2a0c, 0x4254,
 0x3730, 0x071e, 0x1342, 0x2b0c, 0x4354,
 0x3830, 0x081e, 0x1442, 0x2c0c, 0x4454,
 0x3832, 0x0820, 0x1444, 0x2c0e, 0x4456,
 0x3732, 0x0720, 0x1344, 0x2b0e, 0x4356,
 0x3632, 0x0620, 0x1244, 0x2a0e, 0x4256,
 0x3634, 0x0622, 0x1246, 0x2a10, 0x4258,
 0x3734, 0x0722, 0x1346, 0x2b10, 0x4358,
 0x3834, 0x0822, 0x1446, 0x2c10, 0x4458,
 0x3c24, 0x0c12, 0x1836, 0x3000, 0x0048,
 0x3d24, 0x0d12, 0x1936, 0x3100, 0x0148,
 0x3e24, 0x0e12, 0x1a36, 0x3200, 0x0248,
 0x3e26, 0x0e14, 0x1a38, 0x3202, 0x024a,
 0x3d26, 0x0d14, 0x1938, 0x3102, 0x014a,
 0x3c26, 0x0c14, 0x1838, 0x3002, 0x004a,
 0x3c28, 0x0c16, 0x183a, 0x3004, 0x004c,
 0x3d28, 0x0d16, 0x193a, 0x3104, 0x014c,
 0x3e28, 0x0e16, 0x1a3a, 0x3204, 0x024c,
 0x3e2a, 0x0e18, 0x1a3c, 0x3206, 0x024e,
 0x3d2a, 0x0d18, 0x193c, 0x3106, 0x014e,
 0x3c2a, 0x0c18, 0x183c, 0x3006, 0x004e,
 0x3c2c, 0x0c1a, 0x183e, 0x3008, 0x0050,
 0x3d2c, 0x0d1a, 0x193e, 0x3108, 0x0150,
 0x3e2c, 0x0e1a, 0x1a3e, 0x3208, 0x0250,
 0x3e2e, 0x0e1c, 0x1a40, 0x320a, 0x0252,
 0x3d2e, 0x0d1c, 0x1940, 0x310a, 0x0152,
 0x3c2e, 0x0c1c, 0x1840, 0x300a, 0x0052,
 0x3c30, 0x0c1e, 0x1842, 0x300c, 0x0054,
 0x3d30, 0x0d1e, 0x1942, 0x310c, 0x0154,
 0x3e30, 0x0e1e, 0x1a42, 0x320c, 0x0254,
 0x3e32, 0x0e20, 0x1a44, 0x320e, 0x0256,
 0x3d32, 0x0d20, 0x1944, 0x310e, 0x0156,
 0x3c32, 0x0c20, 0x1844, 0x300e, 0x0056,
 0x3c34, 0x0c22, 0x1846, 0x3010, 0x0058,
 0x3d34, 0x0d22, 0x1946, 0x3110, 0x0158,
 0x3e34, 0x0e22, 0x1a46, 0x3210, 0x0258,
 0x4224, 0x1212, 0x1e36, 0x3600, 0x0648,
 0x4324, 0x1312, 0x1f36, 0x3700, 0x0748,
 0x4424, 0x1412, 0x2036, 0x3800, 0x0848,
 0x4426, 0x1414, 0x2038, 0x3802, 0x084a,
 0x4326, 0x1314, 0x1f38, 0x3702, 0x074a,
 0x4226, 0x1214, 0x1e38, 0x3602, 0x064a,
 0x4228, 0x1216, 0x1e3a, 0x3604, 0x064c,
 0x4328, 0x1316, 0x1f3a, 0x3704, 0x074c,
 0x4428, 0x1416, 0x203a, 0x3804, 0x084c,
 0x442a, 0x1418, 0x203c, 0x3806, 0x084e,
 0x432a, 0x1318, 0x1f3c, 0x3706, 0x074e,
 0x422a, 0x1218, 0x1e3c, 0x3606, 0x064e,
 0x422c, 0x121a, 0x1e3e, 0x3608, 0x0650,
 0x432c, 0x131a, 0x1f3e, 0x3708, 0x0750,
 0x442c, 0x141a, 0x203e, 0x3808, 0x0850,
 0x442e, 0x141c, 0x2040, 0x380a, 0x0852,
 0x432e, 0x131c, 0x1f40, 0x370a, 0x0752,
 0x422e, 0x121c, 0x1e40, 0x360a, 0x0652,
 0x4230, 0x121e, 0x1e42, 0x360c, 0x0654,
 0x4330, 0x131e, 0x1f42, 0x370c, 0x0754,
 0x4430, 0x141e, 0x2042, 0x380c, 0x0854,
 0x4432, 0x1420, 0x2044, 0x380e, 0x0856,
 0x4332, 0x1320, 0x1f44, 0x370e, 0x0756,
 0x4232, 0x1220, 0x1e44, 0x360e, 0x0656,
 0x4234, 0x1222, 0x1e46, 0x3610, 0x0658,
 0x4334, 0x1322, 0x1f46, 0x3710, 0x0758,
 0x4434, 0x1422, 0x2046, 0x3810, 0x0858,
 0x0024, 0x1812, 0x2436, 0x3c00, 0x0c48,
 0x0124, 0x1912, 0x2536, 0x3d00, 0x0d48,
 0x0224, 0x1a12, 0x2636, 0x3e00, 0x0e48,
 0x0226, 0x1a14, 0x2638, 0x3e02, 0x0e4a,
 0x0126, 0x1914, 0x2538, 0x3d02, 0x0d4a,
 0x0026, 0x1814, 0x2438, 0x3c02, 0x0c4a,
 0x0028, 0x1816, 0x243a, 0x3c04, 0x0c4c,
 0x0128, 0x1916, 0x253a, 0x3d04, 0x0d4c,
 0x0228, 0x1a16, 0x263a, 0x3e04, 0x0e4c,
 0x022a, 0x1a18, 0x263c, 0x3e06, 0x0e4e,
 0x012a, 0x1918, 0x253c, 0x3d06, 0x0d4e,
 0x002a, 0x1818, 0x243c, 0x3c06, 0x0c4e,
 0x002c, 0x181a, 0x243e, 0x3c08, 0x0c50,
 0x012c, 0x191a, 0x253e, 0x3d08, 0x0d50,
 0x022c, 0x1a1a, 0x263e, 0x3e08, 0x0e50,
 0x022e, 0x1a1c, 0x2640, 0x3e0a, 0x0e52,
 0x012e, 0x191c, 0x2540, 0x3d0a, 0x0d52,
 0x002e, 0x181c, 0x2440, 0x3c0a, 0x0c52,
 0x0030, 0x181e, 0x2442, 0x3c0c, 0x0c54,
 0x0130, 0x191e, 0x2542, 0x3d0c, 0x0d54,
 0x0230, 0x1a1e, 0x2642, 0x3e0c, 0x0e54,
 0x0232, 0x1a20, 0x2644, 0x3e0e, 0x0e56,
 0x0132, 0x1920, 0x2544, 0x3d0e, 0x0d56,
 0x0032, 0x1820, 0x2444, 0x3c0e, 0x0c56,
 0x0034, 0x1822, 0x2446, 0x3c10, 0x0c58,
 0x0134, 0x1922, 0x2546, 0x3d10, 0x0d58,
 0x0234, 0x1a22, 0x2646, 0x3e10, 0x0e58,
 0x0624, 0x1e12, 0x2a36, 0x4200, 0x1248,
 0x0724, 0x1f12, 0x2b36, 0x4300, 0x1348,
 0x0824, 0x2012, 0x2c36, 0x4400, 0x1448,
 0x0826, 0x2014, 0x2c38, 0x4402, 0x144a,
 0x0726, 0x1f14, 0x2b38, 0x4302, 0x134a,
 0x0626, 0x1e14, 0x2a38, 0x4202, 0x124a,
 0x0628, 0x1e16, 0x2a3a, 0x4204, 0x124c,
 0x0728, 0x1f16, 0x2b3a, 0x4304, 0x134c,
 0x0828, 0x2016, 0x2c3a, 0x4404, 0x144c,
 0x082a, 0x2018, 0x2c3c, 0x4406, 0x144e,
 0x072a, 0x1f18, 0x2b3c, 0x4306, 0x134e,
 0x062a, 0x1e18, 0x2a3c, 0x4206, 0x124e,
 0x062c, 0x1e1a, 0x2a3e, 0x4208, 0x1250,
 0x072c, 0x1f1a, 0x2b3e, 0x4308, 0x1350,
 0x082c, 0x201a, 0x2c3e, 0x4408, 0x1450,
 0x082e, 0x201c, 0x2c40, 0x440a, 0x1452,
 0x072e, 0x1f1c, 0x2b40, 0x430a, 0x1352,
 0x062e, 0x1e1c, 0x2a40, 0x420a, 0x1252,
 0x0630, 0x1e1e, 0x2a42, 0x420c, 0x1254,
 0x0730, 0x1f1e, 0x2b42, 0x430c, 0x1354,
 0x0830, 0x201e, 0x2c42, 0x440c, 0x1454,
 0x0832, 0x2020, 0x2c44, 0x440e, 0x1456,
 0x0732, 0x1f20, 0x2b44, 0x430e, 0x1356,
 0x0632, 0x1e20, 0x2a44, 0x420e, 0x1256,
 0x0634, 0x1e22, 0x2a46, 0x4210, 0x1258,
 0x0734, 0x1f22, 0x2b46, 0x4310, 0x1358,
 0x0834, 0x2022, 0x2c46, 0x4410, 0x1458,
 0x0f24, 0x2712, 0x3336, 0x0300, 0x1b48,
 0x1024, 0x2812, 0x3436, 0x0400, 0x1c48,
 0x1124, 0x2912, 0x3536, 0x0500, 0x1d48,
 0x1126, 0x2914, 0x3538, 0x0502, 0x1d4a,
 0x1026, 0x2814, 0x3438, 0x0402, 0x1c4a,
 0x0f26, 0x2714, 0x3338, 0x0302, 0x1b4a,
 0x0f28, 0x2716, 0x333a, 0x0304, 0x1b4c,
 0x1028, 0x2816, 0x343a, 0x0404, 0x1c4c,
 0x1128, 0x2916, 0x353a, 0x0504, 0x1d4c,
 0x112a, 0x2918, 0x353c, 0x0506, 0x1d4e,
 0x102a, 0x2818, 0x343c, 0x0406, 0x1c4e,
 0x0f2a, 0x2718, 0x333c, 0x0306, 0x1b4e,
 0x0f2c, 0x271a, 0x333e, 0x0308, 0x1b50,
 0x102c, 0x281a, 0x343e, 0x0408, 0x1c50,
 0x112c, 0x291a, 0x353e, 0x0508, 0x1d50,
 0x112e, 0x291c, 0x3540, 0x050a, 0x1d52,
 0x102e, 0x281c, 0x3440, 0x040a, 0x1c52,
 0x0f2e, 0x271c, 0x3340, 0x030a, 0x1b52,
 0x0f30, 0x271e, 0x3342, 0x030c, 0x1b54,
 0x1030, 0x281e, 0x3442, 0x040c, 0x1c54,
 0x1130, 0x291e, 0x3542, 0x050c, 0x1d54,
 0x1132, 0x2920, 0x3544, 0x050e, 0x1d56,
 0x1032, 0x2820, 0x3444, 0x040e, 0x1c56,
 0x0f32, 0x2720, 0x3344, 0x030e, 0x1b56,
 0x0f34, 0x2722, 0x3346, 0x0310, 0x1b58,
 0x1034, 0x2822, 0x3446, 0x0410, 0x1c58,
 0x1134, 0x2922, 0x3546, 0x0510, 0x1d58,
 0x1524, 0x2d12, 0x3936, 0x0900, 0x2148,
 0x1624, 0x2e12, 0x3a36, 0x0a00, 0x2248,
 0x1724, 0x2f12, 0x3b36, 0x0b00, 0x2348,
 0x1726, 0x2f14, 0x3b38, 0x0b02, 0x234a,
 0x1626, 0x2e14, 0x3a38, 0x0a02, 0x224a,
 0x1526, 0x2d14, 0x3938, 0x0902, 0x214a,
 0x1528, 0x2d16, 0x393a, 0x0904, 0x214c,
 0x1628, 0x2e16, 0x3a3a, 0x0a04, 0x224c,
 0x1728, 0x2f16, 0x3b3a, 0x0b04, 0x234c,
 0x172a, 0x2f18, 0x3b3c, 0x0b06, 0x234e,
 0x162a, 0x2e18, 0x3a3c, 0x0a06, 0x224e,
 0x152a, 0x2d18, 0x393c, 0x0906, 0x214e,
 0x152c, 0x2d1a, 0x393e, 0x0908, 0x2150,
 0x162c, 0x2e1a, 0x3a3e, 0x0a08, 0x2250,
 0x172c, 0x2f1a, 0x3b3e, 0x0b08, 0x2350,
 0x172e, 0x2f1c, 0x3b40, 0x0b0a, 0x2352,
 0x162e, 0x2e1c, 0x3a40, 0x0a0a, 0x2252,
 0x152e, 0x2d1c, 0x3940, 0x090a, 0x2152,
 0x1530, 0x2d1e, 0x3942, 0x090c, 0x2154,
 0x1630, 0x2e1e, 0x3a42, 0x0a0c, 0x2254,
 0x1730, 0x2f1e, 0x3b42, 0x0b0c, 0x2354,
 0x1732, 0x2f20, 0x3b44, 0x0b0e, 0x2356,
 0x1632, 0x2e20, 0x3a44, 0x0a0e, 0x2256,
 0x1532, 0x2d20, 0x3944, 0x090e, 0x2156,
 0x1534, 0x2d22, 0x3946, 0x0910, 0x2158,
 0x1634, 0x2e22, 0x3a46, 0x0a10, 0x2258,
 0x1734, 0x2f22, 0x3b46, 0x0b10, 0x2358,
 0x1b24, 0x3312, 0x3f36, 0x0f00, 0x2748,
 0x1c24, 0x3412, 0x4036, 0x1000, 0x2848,
 0x1d24, 0x3512, 0x4136, 0x1100, 0x2948,
 0x1d26, 0x3514, 0x4138, 0x1102, 0x294a,
 0x1c26, 0x3414, 0x4038, 0x1002, 0x284a,
 0x1b26, 0x3314, 0x3f38, 0x0f02, 0x274a,
 0x1b28, 0x3316, 0x3f3a, 0x0f04, 0x274c,
 0x1c28, 0x3416, 0x403a, 0x1004, 0x284c,
 0x1d28, 0x3516, 0x413a, 0x1104, 0x294c,
 0x1d2a, 0x3518, 0x413c, 0x1106, 0x294e,
 0x1c2a, 0x3418, 0x403c, 0x1006, 0x284e,
 0x1b2a, 0x3318, 0x3f3c, 0x0f06, 0x274e,
 0x1b2c, 0x331a, 0x3f3e, 0x0f08, 0x2750,
 0x1c2c, 0x341a, 0x403e, 0x1008, 0x2850,
 0x1d2c, 0x351a, 0x413e, 0x1108, 0x2950,
 0x1d2e, 0x351c, 0x4140, 0x110a, 0x2952,
 0x1c2e, 0x341c, 0x4040, 0x100a, 0x2852,
 0x1b2e, 0x331c, 0x3f40, 0x0f0a, 0x2752,
 0x1b30, 0x331e, 0x3f42, 0x0f0c, 0x2754,
 0x1c30, 0x341e, 0x4042, 0x100c, 0x2854,
 0x1d30, 0x351e, 0x4142, 0x110c, 0x2954,
 0x1d32, 0x3520, 0x4144, 0x110e, 0x2956,
 0x1c32, 0x3420, 0x4044, 0x100e, 0x2856,
 0x1b32, 0x3320, 0x3f44, 0x0f0e, 0x2756,
 0x1b34, 0x3322, 0x3f46, 0x0f10, 0x2758,
 0x1c34, 0x3422, 0x4046, 0x1010, 0x2858,
 0x1d34, 0x3522, 0x4146, 0x1110, 0x2958,
 0x2124, 0x3912, 0x4536, 0x1500, 0x2d48,
 0x2224, 0x3a12, 0x4636, 0x1600, 0x2e48,
 0x2324, 0x3b12, 0x4736, 0x1700, 0x2f48,
 0x2326, 0x3b14, 0x4738, 0x1702, 0x2f4a,
 0x2226, 0x3a14, 0x4638, 0x1602, 0x2e4a,
 0x2126, 0x3914, 0x4538, 0x1502, 0x2d4a,
 0x2128, 0x3916, 0x453a, 0x1504, 0x2d4c,
 0x2228, 0x3a16, 0x463a, 0x1604, 0x2e4c,
 0x2328, 0x3b16, 0x473a, 0x1704, 0x2f4c,
 0x232a, 0x3b18, 0x473c, 0x1706, 0x2f4e,
 0x222a, 0x3a18, 0x463c, 0x1606, 0x2e4e,
 0x212a, 0x3918, 0x453c, 0x1506, 0x2d4e,
 0x212c, 0x391a, 0x453e, 0x1508, 0x2d50,
 0x222c, 0x3a1a, 0x463e, 0x1608, 0x2e50,
 0x232c, 0x3b1a, 0x473e, 0x1708, 0x2f50,
 0x232e, 0x3b1c, 0x4740, 0x170a, 0x2f52,
 0x222e, 0x3a1c, 0x4640, 0x160a, 0x2e52,
 0x212e, 0x391c, 0x4540, 0x150a, 0x2d52,
 0x2130, 0x391e, 0x4542, 0x150c, 0x2d54,
 0x2230, 0x3a1e, 0x4642, 0x160c, 0x2e54,
 0x2330, 0x3b1e, 0x4742, 0x170c, 0x2f54,
 0x2332, 0x3b20, 0x4744, 0x170e, 0x2f56,
 0x2232, 0x3a20, 0x4644, 0x160e, 0x2e56,
 0x2132, 0x3920, 0x4544, 0x150e, 0x2d56,
 0x2134, 0x3922, 0x4546, 0x1510, 0x2d58,
 0x2234, 0x3a22, 0x4646, 0x1610, 0x2e58,
 0x2334, 0x3b22, 0x4746, 0x1710, 0x2f58,
 0x2724, 0x3f12, 0x0336, 0x1b00, 0x3348,
 0x2824, 0x4012, 0x0436, 0x1c00, 0x3448,
 0x2924, 0x4112, 0x0536, 0x1d00, 0x3548,
 0x2926, 0x4114, 0x0538, 0x1d02, 0x354a,
 0x2826, 0x4014, 0x0438, 0x1c02, 0x344a,
 0x2726, 0x3f14, 0x0338, 0x1b02, 0x334a,
 0x2728, 0x3f16, 0x033a, 0x1b04, 0x334c,
 0x2828, 0x4016, 0x043a, 0x1c04, 0x344c,
 0x2928, 0x4116, 0x053a, 0x1d04, 0x354c,
 0x292a, 0x4118, 0x053c, 0x1d06, 0x354e,
 0x282a, 0x4018, 0x043c, 0x1c06, 0x344e,
 0x272a, 0x3f18, 0x033c, 0x1b06, 0x334e,
 0x272c, 0x3f1a, 0x033e, 0x1b08, 0x3350,
 0x282c, 0x401a, 0x043e, 0x1c08, 0x3450,
 0x292c, 0x411a, 0x053e, 0x1d08, 0x3550,
 0x292e, 0x411c, 0x0540, 0x1d0a, 0x3552,
 0x282e, 0x401c, 0x0440, 0x1c0a, 0x3452,
 0x272e, 0x3f1c, 0x0340, 0x1b0a, 0x3352,
 0x2730, 0x3f1e, 0x0342, 0x1b0c, 0x3354,
 0x2830, 0x401e, 0x0442, 0x1c0c, 0x3454,
 0x2930, 0x411e, 0x0542, 0x1d0c, 0x3554,
 0x2932, 0x4120, 0x0544, 0x1d0e, 0x3556,
 0x2832, 0x4020, 0x0444, 0x1c0e, 0x3456,
 0x2732, 0x3f20, 0x0344, 0x1b0e, 0x3356,
 0x2734, 0x3f22, 0x0346, 0x1b10, 0x3358,
 0x2834, 0x4022, 0x0446, 0x1c10, 0x3458,
 0x2934, 0x4122, 0x0546, 0x1d10, 0x3558,
 0x2d24, 0x4512, 0x0936, 0x2100, 0x3948,
 0x2e24, 0x4612, 0x0a36, 0x2200, 0x3a48,
 0x2f24, 0x4712, 0x0b36, 0x2300, 0x3b48,
 0x2f26, 0x4714, 0x0b38, 0x2302, 0x3b4a,
 0x2e26, 0x4614, 0x0a38, 0x2202, 0x3a4a,
 0x2d26, 0x4514, 0x0938, 0x2102, 0x394a,
 0x2d28, 0x4516, 0x093a, 0x2104, 0x394c,
 0x2e28, 0x4616, 0x0a3a, 0x2204, 0x3a4c,
 0x2f28, 0x4716, 0x0b3a, 0x2304, 0x3b4c,
 0x2f2a, 0x4718, 0x0b3c, 0x2306, 0x3b4e,
 0x2e2a, 0x4618, 0x0a3c, 0x2206, 0x3a4e,
 0x2d2a, 0x4518, 0x093c, 0x2106, 0x394e,
 0x2d2c, 0x451a, 0x093e, 0x2108, 0x3950,
 0x2e2c, 0x461a, 0x0a3e, 0x2208, 0x3a50,
 0x2f2c, 0x471a, 0x0b3e, 0x2308, 0x3b50,
 0x2f2e, 0x471c, 0x0b40, 0x230a, 0x3b52,
 0x2e2e, 0x461c, 0x0a40, 0x220a, 0x3a52,
 0x2d2e, 0x451c, 0x0940, 0x210a, 0x3952,
 0x2d30, 0x451e, 0x0942, 0x210c, 0x3954,
 0x2e30, 0x461e, 0x0a42, 0x220c, 0x3a54,
 0x2f30, 0x471e, 0x0b42, 0x230c, 0x3b54,
 0x2f32, 0x4720, 0x0b44, 0x230e, 0x3b56,
 0x2e32, 0x4620, 0x0a44, 0x220e, 0x3a56,
 0x2d32, 0x4520, 0x0944, 0x210e, 0x3956,
 0x2d34, 0x4522, 0x0946, 0x2110, 0x3958,
 0x2e34, 0x4622, 0x0a46, 0x2210, 0x3a58,
 0x2f34, 0x4722, 0x0b46, 0x2310, 0x3b58,
 0x3324, 0x0312, 0x0f36, 0x2700, 0x3f48,
 0x3424, 0x0412, 0x1036, 0x2800, 0x4048,
 0x3524, 0x0512, 0x1136, 0x2900, 0x4148,
 0x3526, 0x0514, 0x1138, 0x2902, 0x414a,
 0x3426, 0x0414, 0x1038, 0x2802, 0x404a,
 0x3326, 0x0314, 0x0f38, 0x2702, 0x3f4a,
 0x3328, 0x0316, 0x0f3a, 0x2704, 0x3f4c,
 0x3428, 0x0416, 0x103a, 0x2804, 0x404c,
 0x3528, 0x0516, 0x113a, 0x2904, 0x414c,
 0x352a, 0x0518, 0x113c, 0x2906, 0x414e,
 0x342a, 0x0418, 0x103c, 0x2806, 0x404e,
 0x332a, 0x0318, 0x0f3c, 0x2706, 0x3f4e,
 0x332c, 0x031a, 0x0f3e, 0x2708, 0x3f50,
 0x342c, 0x041a, 0x103e, 0x2808, 0x4050,
 0x352c, 0x051a, 0x113e, 0x2908, 0x4150,
 0x352e, 0x051c, 0x1140, 0x290a, 0x4152,
 0x342e, 0x041c, 0x1040, 0x280a, 0x4052,
 0x332e, 0x031c, 0x0f40, 0x270a, 0x3f52,
 0x3330, 0x031e, 0x0f42, 0x270c, 0x3f54,
 0x3430, 0x041e, 0x1042, 0x280c, 0x4054,
 0x3530, 0x051e, 0x1142, 0x290c, 0x4154,
 0x3532, 0x0520, 0x1144, 0x290e, 0x4156,
 0x3432, 0x0420, 0x1044, 0x280e, 0x4056,
 0x3332, 0x0320, 0x0f44, 0x270e, 0x3f56,
 0x3334, 0x0322, 0x0f46, 0x2710, 0x3f58,
 0x3434, 0x0422, 0x1046, 0x2810, 0x4058,
 0x3534, 0x0522, 0x1146, 0x2910, 0x4158,
 0x3924, 0x0912, 0x1536, 0x2d00, 0x4548,
 0x3a24, 0x0a12, 0x1636, 0x2e00, 0x4648,
 0x3b24, 0x0b12, 0x1736, 0x2f00, 0x4748,
 0x3b26, 0x0b14, 0x1738, 0x2f02, 0x474a,
 0x3a26, 0x0a14, 0x1638, 0x2e02, 0x464a,
 0x3926, 0x0914, 0x1538, 0x2d02, 0x454a,
 0x3928, 0x0916, 0x153a, 0x2d04, 0x454c,
 0x3a28, 0x0a16, 0x163a, 0x2e04, 0x464c,
 0x3b28, 0x0b16, 0x173a, 0x2f04, 0x474c,
 0x3b2a, 0x0b18, 0x173c, 0x2f06, 0x474e,
 0x3a2a, 0x0a18, 0x163c, 0x2e06, 0x464e,
 0x392a, 0x0918, 0x153c, 0x2d06, 0x454e,
 0x392c, 0x091a, 0x153e, 0x2d08, 0x4550,
 0x3a2c, 0x0a1a, 0x163e, 0x2e08, 0x4650,
 0x3b2c, 0x0b1a, 0x173e, 0x2f08, 0x4750,
 0x3b2e, 0x0b1c, 0x1740, 0x2f0a, 0x4752,
 0x3a2e, 0x0a1c, 0x1640, 0x2e0a, 0x4652,
 0x392e, 0x091c, 0x1540, 0x2d0a, 0x4552,
 0x3930, 0x091e, 0x1542, 0x2d0c, 0x4554,
 0x3a30, 0x0a1e, 0x1642, 0x2e0c, 0x4654,
 0x3b30, 0x0b1e, 0x1742, 0x2f0c, 0x4754,
 0x3b32, 0x0b20, 0x1744, 0x2f0e, 0x4756,
 0x3a32, 0x0a20, 0x1644, 0x2e0e, 0x4656,
 0x3932, 0x0920, 0x1544, 0x2d0e, 0x4556,
 0x3934, 0x0922, 0x1546, 0x2d10, 0x4558,
 0x3a34, 0x0a22, 0x1646, 0x2e10, 0x4658,
 0x3b34, 0x0b22, 0x1746, 0x2f10, 0x4758,
 0x3f24, 0x0f12, 0x1b36, 0x3300, 0x0348,
 0x4024, 0x1012, 0x1c36, 0x3400, 0x0448,
 0x4124, 0x1112, 0x1d36, 0x3500, 0x0548,
 0x4126, 0x1114, 0x1d38, 0x3502, 0x054a,
 0x4026, 0x1014, 0x1c38, 0x3402, 0x044a,
 0x3f26, 0x0f14, 0x1b38, 0x3302, 0x034a,
 0x3f28, 0x0f16, 0x1b3a, 0x3304, 0x034c,
 0x4028, 0x1016, 0x1c3a, 0x3404, 0x044c,
 0x4128, 0x1116, 0x1d3a, 0x3504, 0x054c,
 0x412a, 0x1118, 0x1d3c, 0x3506, 0x054e,
 0x402a, 0x1018, 0x1c3c, 0x3406, 0x044e,
 0x3f2a, 0x0f18, 0x1b3c, 0x3306, 0x034e,
 0x3f2c, 0x0f1a, 0x1b3e, 0x3308, 0x0350,
 0x402c, 0x101a, 0x1c3e, 0x3408, 0x0450,
 0x412c, 0x111a, 0x1d3e, 0x3508, 0x0550,
 0x412e, 0x111c, 0x1d40, 0x350a, 0x0552,
 0x402e, 0x101c, 0x1c40, 0x340a, 0x0452,
 0x3f2e, 0x0f1c, 0x1b40, 0x330a, 0x0352,
 0x3f30, 0x0f1e, 0x1b42, 0x330c, 0x0354,
 0x4030, 0x101e, 0x1c42, 0x340c, 0x0454,
 0x4130, 0x111e, 0x1d42, 0x350c, 0x0554,
 0x4132, 0x1120, 0x1d44, 0x350e, 0x0556,
 0x4032, 0x1020, 0x1c44, 0x340e, 0x0456,
 0x3f32, 0x0f20, 0x1b44, 0x330e, 0x0356,
 0x3f34, 0x0f22, 0x1b46, 0x3310, 0x0358,
 0x4034, 0x1022, 0x1c46, 0x3410, 0x0458,
 0x4134, 0x1122, 0x1d46, 0x3510, 0x0558,
 0x4524, 0x1512, 0x2136, 0x3900, 0x0948,
 0x4624, 0x1612, 0x2236, 0x3a00, 0x0a48,
 0x4724, 0x1712, 0x2336, 0x3b00, 0x0b48,
 0x4726, 0x1714, 0x2338, 0x3b02, 0x0b4a,
 0x4626, 0x1614, 0x2238, 0x3a02, 0x0a4a,
 0x4526, 0x1514, 0x2138, 0x3902, 0x094a,
 0x4528, 0x1516, 0x213a, 0x3904, 0x094c,
 0x4628, 0x1616, 0x223a, 0x3a04, 0x0a4c,
 0x4728, 0x1716, 0x233a, 0x3b04, 0x0b4c,
 0x472a, 0x1718, 0x233c, 0x3b06, 0x0b4e,
 0x462a, 0x1618, 0x223c, 0x3a06, 0x0a4e,
 0x452a, 0x1518, 0x213c, 0x3906, 0x094e,
 0x452c, 0x151a, 0x213e, 0x3908, 0x0950,
 0x462c, 0x161a, 0x223e, 0x3a08, 0x0a50,
 0x472c, 0x171a, 0x233e, 0x3b08, 0x0b50,
 0x472e, 0x171c, 0x2340, 0x3b0a, 0x0b52,
 0x462e, 0x161c, 0x2240, 0x3a0a, 0x0a52,
 0x452e, 0x151c, 0x2140, 0x390a, 0x0952,
 0x4530, 0x151e, 0x2142, 0x390c, 0x0954,
 0x4630, 0x161e, 0x2242, 0x3a0c, 0x0a54,
 0x4730, 0x171e, 0x2342, 0x3b0c, 0x0b54,
 0x4732, 0x1720, 0x2344, 0x3b0e, 0x0b56,
 0x4632, 0x1620, 0x2244, 0x3a0e, 0x0a56,
 0x4532, 0x1520, 0x2144, 0x390e, 0x0956,
 0x4534, 0x1522, 0x2146, 0x3910, 0x0958,
 0x4634, 0x1622, 0x2246, 0x3a10, 0x0a58,
 0x4734, 0x1722, 0x2346, 0x3b10, 0x0b58,
 0x0324, 0x1b12, 0x2736, 0x3f00, 0x0f48,
 0x0424, 0x1c12, 0x2836, 0x4000, 0x1048,
 0x0524, 0x1d12, 0x2936, 0x4100, 0x1148,
 0x0526, 0x1d14, 0x2938, 0x4102, 0x114a,
 0x0426, 0x1c14, 0x2838, 0x4002, 0x104a,
 0x0326, 0x1b14, 0x2738, 0x3f02, 0x0f4a,
 0x0328, 0x1b16, 0x273a, 0x3f04, 0x0f4c,
 0x0428, 0x1c16, 0x283a, 0x4004, 0x104c,
 0x0528, 0x1d16, 0x293a, 0x4104, 0x114c,
 0x052a, 0x1d18, 0x293c, 0x4106, 0x114e,
 0x042a, 0x1c18, 0x283c, 0x4006, 0x104e,
 0x032a, 0x1b18, 0x273c, 0x3f06, 0x0f4e,
 0x032c, 0x1b1a, 0x273e, 0x3f08, 0x0f50,
 0x042c, 0x1c1a, 0x283e, 0x4008, 0x1050,
 0x052c, 0x1d1a, 0x293e, 0x4108, 0x1150,
 0x052e, 0x1d1c, 0x2940, 0x410a, 0x1152,
 0x042e, 0x1c1c, 0x2840, 0x400a, 0x1052,
 0x032e, 0x1b1c, 0x2740, 0x3f0a, 0x0f52,
 0x0330, 0x1b1e, 0x2742, 0x3f0c, 0x0f54,
 0x0430, 0x1c1e, 0x2842, 0x400c, 0x1054,
 0x0530, 0x1d1e, 0x2942, 0x410c, 0x1154,
 0x0532, 0x1d20, 0x2944, 0x410e, 0x1156,
 0x0432, 0x1c20, 0x2844, 0x400e, 0x1056,
 0x0332, 0x1b20, 0x2744, 0x3f0e, 0x0f56,
 0x0334, 0x1b22, 0x2746, 0x3f10, 0x0f58,
 0x0434, 0x1c22, 0x2846, 0x4010, 0x1058,
 0x0534, 0x1d22, 0x2946, 0x4110, 0x1158,
 0x0924, 0x2112, 0x2d36, 0x4500, 0x1548,
 0x0a24, 0x2212, 0x2e36, 0x4600, 0x1648,
 0x0b24, 0x2312, 0x2f36, 0x4700, 0x1748,
 0x0b26, 0x2314, 0x2f38, 0x4702, 0x174a,
 0x0a26, 0x2214, 0x2e38, 0x4602, 0x164a,
 0x0926, 0x2114, 0x2d38, 0x4502, 0x154a,
 0x0928, 0x2116, 0x2d3a, 0x4504, 0x154c,
 0x0a28, 0x2216, 0x2e3a, 0x4604, 0x164c,
 0x0b28, 0x2316, 0x2f3a, 0x4704, 0x174c,
 0x0b2a, 0x2318, 0x2f3c, 0x4706, 0x174e,
 0x0a2a, 0x2218, 0x2e3c, 0x4606, 0x164e,
 0x092a, 0x2118, 0x2d3c, 0x4506, 0x154e,
 0x092c, 0x211a, 0x2d3e, 0x4508, 0x1550,
 0x0a2c, 0x221a, 0x2e3e, 0x4608, 0x1650,
 0x0b2c, 0x231a, 0x2f3e, 0x4708, 0x1750,
 0x0b2e, 0x231c, 0x2f40, 0x470a, 0x1752,
 0x0a2e, 0x221c, 0x2e40, 0x460a, 0x1652,
 0x092e, 0x211c, 0x2d40, 0x450a, 0x1552,
 0x0930, 0x211e, 0x2d42, 0x450c, 0x1554,
 0x0a30, 0x221e, 0x2e42, 0x460c, 0x1654,
 0x0b30, 0x231e, 0x2f42, 0x470c, 0x1754,
 0x0b32, 0x2320, 0x2f44, 0x470e, 0x1756,
 0x0a32, 0x2220, 0x2e44, 0x460e, 0x1656,
 0x0932, 0x2120, 0x2d44, 0x450e, 0x1556,
 0x0934, 0x2122, 0x2d46, 0x4510, 0x1558,
 0x0a34, 0x2222, 0x2e46, 0x4610, 0x1658,
 0x0b34, 0x2322, 0x2f46, 0x4710, 0x1758,
};

static const uint16_t dv_place_1080i60[4*10*27*5] = {
 0x2048, 0x5024, 0x686c, 0x0800, 0x3890,
 0x3848, 0x6824, 0x086c, 0x2000, 0x5090,
 0x5048, 0x0824, 0x206c, 0x3800, 0x6890,
 0x6848, 0x2024, 0x386c, 0x5000, 0x0890,
 0x0848, 0x3824, 0x506c, 0x6800, 0x2090,
 0x204a, 0x5026, 0x686e, 0x0802, 0x3892,
 0x384a, 0x6826, 0x086e, 0x2002, 0x5092,
 0x504a, 0x0826, 0x206e, 0x3802, 0x6892,
 0x684a, 0x2026, 0x386e, 0x5002, 0x0892,
 0x084a, 0x3826, 0x506e, 0x6802, 0x2092,
 0x204c, 0x5028, 0x6870, 0x0804, 0x3894,
 0x384c, 0x6828, 0x0870, 0x2004, 0x5094,
 0x504c, 0x0828, 0x2070, 0x3804, 0x6894,
 0x684c, 0x2028, 0x3870, 0x5004, 0x0894,
 0x084c, 0x3828, 0x5070, 0x6804, 0x2094,
 0x204e, 0x502a, 0x6872, 0x0806, 0x3896,
 0x384e, 0x682a, 0x0872, 0x2006, 0x5096,
 0x504e, 0x082a, 0x2072, 0x3806, 0x6896,
 0x684e, 0x202a, 0x3872, 0x5006, 0x0896,
 0x084e, 0x382a, 0x5072, 0x6806, 0x2096,
 0x2050, 0x502c, 0x6874, 0x0808, 0x3898,
 0x3850, 0x682c, 0x0874, 0x2008, 0x5098,
 0x5050, 0x082c, 0x2074, 0x3808, 0x6898,
 0x6850, 0x202c, 0x3874, 0x5008, 0x0898,
 0x0850, 0x382c, 0x5074, 0x6808, 0x2098,
 0x2052, 0x502e, 0x6876, 0x080a, 0x389a,
 0x3852, 0x682e, 0x0876, 0x200a, 0x509a,
 0x5052, 0x082e, 0x2076, 0x380a, 0x689a,
 0x6852, 0x202e, 0x3876, 0x500a, 0x089a,
 0x0852, 0x382e, 0x5076, 0x680a, 0x209a,
 0x2054, 0x5030, 0x6878, 0x080c, 0x389c,
 0x3854, 0x6830, 0x0878, 0x200c, 0x509c,
 0x5054, 0x0830, 0x2078, 0x380c, 0x689c,
 0x6854, 0x2030, 0x3878, 0x500c, 0x089c,
 0x0854, 0x3830, 0x5078, 0x680c, 0x209c,
 0x2056, 0x5032, 0x687a, 0x080e, 0x389e,
 0x3856, 0x6832, 0x087a, 0x200e, 0x509e,
 0x5056, 0x0832, 0x207a, 0x380e, 0x689e,
 0x6856, 0x2032, 0x387a, 0x500e, 0x089e,
 0x0856, 0x3832, 0x507a, 0x680e, 0x209e,
 0x2058, 0x5034, 0x687c, 0x0810, 0x0078,
 0x3858, 0x6834, 0x087c, 0x2010, 0x8214,
 0x5058, 0x0834, 0x207c, 0x3810, 0x8264,
 0x6858, 0x2034, 0x387c, 0x5010, 0x0000,
 0x0858, 0x3834, 0x507c, 0x6810, 0x003c,
 0x2448, 0x5424, 0x6c6c, 0x0c00, 0x3c90,
 0x3c48, 0x6c24, 0x0c6c, 0x2400, 0x5490,
 0x5448, 0x0c24, 0x246c, 0x3c00, 0x6c90,
 0x6c48, 0x2424, 0x3c6c, 0x5400, 0x0c90,
 0x0c48, 0x3c24, 0x546c, 0x6c00, 0x2490,
 0x244a, 0x5426, 0x6c6e, 0x0c02, 0x3c92,
 0x3c4a, 0x6c26, 0x0c6e, 0x2402, 0x5492,
 0x544a, 0x0c26, 0x246e, 0x3c02, 0x6c92,
 0x6c4a, 0x2426, 0x3c6e, 0x5402, 0x0c92,
 0x0c4a, 0x3c26, 0x546e, 0x6c02, 0x2492,
 0x244c, 0x5428, 0x6c70, 0x0c04, 0x3c94,
 0x3c4c, 0x6c28, 0x0c70, 0x2404, 0x5494,
 0x544c, 0x0c28, 0x2470, 0x3c04, 0x6c94,
 0x6c4c, 0x2428, 0x3c70, 0x5404, 0x0c94,
 0x0c4c, 0x3c28, 0x5470, 0x6c04, 0x2494,
 0x244e, 0x542a, 0x6c72, 0x0c06, 0x3c96,
 0x3c4e, 0x6c2a, 0x0c72, 0x2406, 0x5496,
 0x544e, 0x0c2a, 0x2472, 0x3c06, 0x6c96,
 0x6c4e, 0x242a, 0x3c72, 0x5406, 0x0c96,
 0x0c4e, 0x3c2a, 0x5472, 0x6c06, 0x2496,
 0x2450, 0x542c, 0x6c74, 0x0c08, 0x3c98,
 0x3c50, 0x6c2c, 0x0c74, 0x2408, 0x5498,
 0x5450, 0x0c2c, 0x2474, 0x3c08, 0x6c98,
 0x6c50, 0x242c, 0x3c74, 0x5408, 0x0c98,
 0x0c50, 0x3c2c, 0x5474, 0x6c08, 0x2498,
 0x2452, 0x542e, 0x6c76, 0x0c0a, 0x3c9a,
 0x3c52, 0x6c2e, 0x0c76, 0x240a, 0x549a,
 0x5452, 0x0c2e, 0x2476, 0x3c0a, 0x6c9a,
 0x6c52, 0x242e, 0x3c76, 0x540a, 0x0c9a,
 0x0c52, 0x3c2e, 0x5476, 0x6c0a, 0x249a,
 0x2454, 0x5430, 0x6c78, 0x0c0c, 0x3c9c,
 0x3c54, 0x6c30, 0x0c78, 0x240c, 0x549c,
 0x5454, 0x0c30, 0x2478, 0x3c0c, 0x6c9c,
 0x6c54, 0x2430, 0x3c78, 0x540c, 0x0c9c,
 0x0c54, 0x3c30, 0x5478, 0x6c0c, 0x249c,
 0x2456, 0x5432, 0x6c7a, 0x0c0e, 0x3c9e,
 0x3c56, 0x6c32, 0x0c7a, 0x240e, 0x549e,
 0x5456, 0x0c32, 0x247a, 0x3c0e, 0x6c9e,
 0x6c56, 0x2432, 0x3c7a, 0x540e, 0x0c9e,
 0x0c56, 0x3c32, 0x547a, 0x6c0e, 0x249e,
 0x2458, 0x5434, 0x6c7c, 0x0c10, 0x0478,
 0x3c58, 0x6c34, 0x0c7c, 0x2410, 0x8028,
 0x5458, 0x0c34, 0x247c, 0x3c10, 0x8078,
 0x6c58, 0x2434, 0x3c7c, 0x5410, 0x0400,
 0x0c58, 0x3c34, 0x547c, 0x6c10, 0x043c,
 0x2848, 0x5824, 0x706c, 0x1000, 0x4090,
 0x4048, 0x7024, 0x106c, 0x2800, 0x5890,
 0x5848, 0x1024, 0x286c, 0x4000, 0x7090,
 0x7048, 0x2824, 0x406c, 0x5800, 0x1090,
 0x1048, 0x4024, 0x586c, 0x7000, 0x2890,
 0x284a, 0x5826, 0x706e, 0x1002, 0x4092,
 0x404a, 0x7026, 0x106e, 0x2802, 0x5892,
 0x584a, 0x1026, 0x286e, 0x4002, 0x7092,
 0x704a, 0x2826, 0x406e, 0x5802, 0x1092,
 0x104a, 0x4026, 0x586e, 0x7002, 0x2892,
 0x284c, 0x5828, 0x7070, 0x1004, 0x4094,
 0x404c, 0x7028, 0x1070, 0x2804, 0x5894,
 0x584c, 0x1028, 0x2870, 0x4004, 0x7094,
 0x704c, 0x2828, 0x4070, 0x5804, 0x1094,
 0x104c, 0x4028, 0x5870, 0x7004, 0x2894,
 0x284e, 0x582a, 0x7072, 0x1006, 0x4096,
 0x404e, 0x702a, 0x1072, 0x2806, 0x5896,
 0x584e, 0x102a, 0x2872, 0x4006, 0x7096,
 0x704e, 0x282a, 0x4072, 0x5806, 0x1096,
 0x104e, 0x402a, 0x5872, 0x7006, 0x2896,
 0x2850, 0x582c, 0x7074, 0x1008, 0x4098,
 0x4050, 0x702c, 0x1074, 0x2808, 0x5898,
 0x5850, 0x102c, 0x2874, 0x4008, 0x7098,
 0x7050, 0x282c, 0x4074, 0x5808, 0x1098,
 0x1050, 0x402c, 0x5874, 0x7008, 0x2898,
 0x2852, 0x582e, 0x7076, 0x100a, 0x409a,
 0x4052, 0x702e, 0x1076, 0x280a, 0x589a,
 0x5852, 0x102e, 0x2876, 0x400a, 0x709a,
 0x7052, 0x282e, 0x4076, 0x580a, 0x109a,
 0x1052, 0x402e, 0x5876, 0x700a, 0x289a,
 0x2854, 0x5830, 0x7078, 0x100c, 0x409c,
 0x4054, 0x7030, 0x1078, 0x280c, 0x589c,
 0x5854, 0x1030, 0x2878, 0x400c, 0x709c,
 0x7054, 0x2830, 0x4078, 0x580c, 0x109c,
 0x1054, 0x4030, 0x5878, 0x700c, 0x289c,
 0x2856, 0x5832, 0x707a, 0x100e, 0x409e,
 0x4056, 0x7032, 0x107a, 0x280e, 0x589e,
 0x5856, 0x1032, 0x287a, 0x400e, 0x709e,
 0x7056, 0x2832, 0x407a, 0x580e, 0x109e,
 0x1056, 0x4032, 0x587a, 0x700e, 0x289e,
 0x2858, 0x5834, 0x707c, 0x1010, 0x008c,
 0x4058, 0x7034, 0x107c, 0x2810, 0x8428,
 0x5858, 0x1034, 0x287c, 0x4010, 0x8478,
 0x7058, 0x2834, 0x407c, 0x5810, 0x0014,
 0x1058, 0x4034, 0x587c, 0x7010, 0x0050,
 0x2c48, 0x5c24, 0x746c, 0x1400, 0x4490,
 0x4448, 0x7424, 0x146c, 0x2c00, 0x5c90,
 0x5c48, 0x1424, 0x2c6c, 0x4400, 0x7490,
 0x7448, 0x2c24, 0x446c, 0x5c00, 0x1490,
 0x1448, 0x4424, 0x5c6c, 0x7400, 0x2c90,
 0x2c4a, 0x5c26, 0x746e, 0x1402, 0x4492,
 0x444a, 0x7426, 0x146e, 0x2c02, 0x5c92,
 0x5c4a, 0x1426, 0x2c6e, 0x4402, 0x7492,
 0x744a, 0x2c26, 0x446e, 0x5c02, 0x1492,
 0x144a, 0x4426, 0x5c6e, 0x7402, 0x2c92,
 0x2c4c, 0x5c28, 0x7470, 0x1404, 0x4494,
 0x444c, 0x7428, 0x1470, 0x2c04, 0x5c94,
 0x5c4c, 0x1428, 0x2c70, 0x4404, 0x7494,
 0x744c, 0x2c28, 0x4470, 0x5c04, 0x1494,
 0x144c, 0x4428, 0x5c70, 0x7404, 0x2c94,
 0x2c4e, 0x5c2a, 0x7472, 0x1406, 0x4496,
 0x444e, 0x742a, 0x1472, 0x2c06, 0x5c96,
 0x5c4e, 0x142a, 0x2c72, 0x4406, 0x7496,
 0x744e, 0x2c2a, 0x4472, 0x5c06, 0x1496,
 0x144e, 0x442a, 0x5c72, 0x7406, 0x2c96,
 0x2c50, 0x5c2c, 0x7474, 0x1408, 0x4498,
 0x4450, 0x742c, 0x1474, 0x2c08, 0x5c98,
 0x5c50, 0x142c, 0x2c74, 0x4408, 0x7498,
 0x7450, 0x2c2c, 0x4474, 0x5c08, 0x1498,
 0x1450, 0x442c, 0x5c74, 0x7408, 0x2c98,
 0x2c52, 0x5c2e, 0x7476, 0x140a, 0x449a,
 0x4452, 0x742e, 0x1476, 0x2c0a, 0x5c9a,
 0x5c52, 0x142e, 0x2c76, 0x440a, 0x749a,
 0x7452, 0x2c2e, 0x4476, 0x5c0a, 0x149a,
 0x1452, 0x442e, 0x5c76, 0x740a, 0x2c9a,
 0x2c54, 0x5c30, 0x7478, 0x140c, 0x449c,
 0x4454, 0x7430, 0x1478, 0x2c0c, 0x5c9c,
 0x5c54, 0x1430, 0x2c78, 0x440c, 0x749c,
 0x7454, 0x2c30, 0x4478, 0x5c0c, 0x149c,
 0x1454, 0x4430, 0x5c78, 0x740c, 0x2c9c,
 0x2c56, 0x5c32, 0x747a, 0x140e, 0x449e,
 0x4456, 0x7432, 0x147a, 0x2c0e, 0x5c9e,
 0x5c56, 0x1432, 0x2c7a, 0x440e, 0x749e,
 0x7456, 0x2c32, 0x447a, 0x5c0e, 0x149e,
 0x1456, 0x4432, 0x5c7a, 0x740e, 0x2c9e,
 0x2c58, 0x5c34, 0x747c, 0x1410, 0x048c,
 0x4458, 0x7434, 0x147c, 0x2c10, 0x823c,
 0x5c58, 0x1434, 0x2c7c, 0x4410, 0x828c,
 0x7458, 0x2c34, 0x447c, 0x5c10, 0x0414,
 0x1458, 0x4434, 0x5c7c, 0x7410, 0x0450,
 0x3048, 0x6024, 0x786c, 0x1800, 0x4890,
 0x4848, 0x7824, 0x186c, 0x3000, 0x6090,
 0x6048, 0x1824, 0x306c, 0x4800, 0x7890,
 0x7848, 0x3024, 0x486c, 0x6000, 0x1890,
 0x1848, 0x4824, 0x606c, 0x7800, 0x3090,
 0x304a, 0x6026, 0x786e, 0x1802, 0x4892,
 0x484a, 0x7826, 0x186e, 0x3002, 0x6092,
 0x604a, 0x1826, 0x306e, 0x4802, 0x7892,
 0x784a, 0x3026, 0x486e, 0x6002, 0x1892,
 0x184a, 0x4826, 0x606e, 0x7802, 0x3092,
 0x304c, 0x6028, 0x7870, 0x1804, 0x4894,
 0x484c, 0x7828, 0x1870, 0x3004, 0x6094,
 0x604c, 0x1828, 0x3070, 0x4804, 0x7894,
 0x784c, 0x3028, 0x4870, 0x6004, 0x1894,
 0x184c, 0x4828, 0x6070, 0x7804, 0x3094,
 0x304e, 0x602a, 0x7872, 0x1806, 0x4896,
 0x484e, 0x782a, 0x1872, 0x3006, 0x6096,
 0x604e, 0x182a, 0x3072, 0x4806, 0x7896,
 0x784e, 0x302a, 0x4872, 0x6006, 0x1896,
 0x184e, 0x482a, 0x6072, 0x7806, 0x3096,
 0x3050, 0x602c, 0x7874, 0x1808, 0x4898,
 0x4850, 0x782c, 0x1874, 0x3008, 0x6098,
 0x6050, 0x182c, 0x3074, 0x4808, 0x7898,
 0x7850, 0x302c, 0x4874, 0x6008, 0x1898,
 0x1850, 0x482c, 0x6074, 0x7808, 0x3098,
 0x3052, 0x602e, 0x7876, 0x180a, 0x489a,
 0x4852, 0x782e, 0x1876, 0x300a, 0x609a,
 0x6052, 0x182e, 0x3076, 0x480a, 0x789a,
 0x7852, 0x302e, 0x4876, 0x600a, 0x189a,
 0x1852, 0x482e, 0x6076, 0x780a, 0x309a,
 0x3054, 0x6030, 0x7878, 0x180c, 0x489c,
 0x4854, 0x7830, 0x1878, 0x300c, 0x609c,
 0x6054, 0x1830, 0x3078, 0x480c, 0x789c,
 0x7854, 0x3030, 0x4878, 0x600c, 0x189c,
 0x1854, 0x4830, 0x6078, 0x780c, 0x309c,
 0x3056, 0x6032, 0x787a, 0x180e, 0x489e,
 0x4856, 0x7832, 0x187a, 0x300e, 0x609e,
 0x6056, 0x1832, 0x307a, 0x480e, 0x789e,
 0x7856, 0x3032, 0x487a, 0x600e, 0x189e,
 0x1856, 0x4832, 0x607a, 0x780e, 0x309e,
 0x3058, 0x6034, 0x787c, 0x1810, 0x8000,
 0x4858, 0x7834, 0x187c, 0x3010, 0x8050,
 0x6058, 0x1834, 0x307c, 0x4810, 0x8600,
 0x7858, 0x3034, 0x487c, 0x6010, 0x0028,
 0x1858, 0x4834, 0x607c, 0x7810, 0x0064,
 0x3448, 0x6424, 0x7c6c, 0x1c00, 0x4c90,
 0x4c48, 0x7c24, 0x1c6c, 0x3400, 0x6490,
 0x6448, 0x1c24, 0x346c, 0x4c00, 0x7c90,
 0x7c48, 0x3424, 0x4c6c, 0x6400, 0x1c90,
 0x1c48, 0x4c24, 0x646c, 0x7c00, 0x3490,
 0x344a, 0x6426, 0x7c6e, 0x1c02, 0x4c92,
 0x4c4a, 0x7c26, 0x1c6e, 0x3402, 0x6492,
 0x644a, 0x1c26, 0x346e, 0x4c02, 0x7c92,
 0x7c4a, 0x3426, 0x4c6e, 0x6402, 0x1c92,
 0x1c4a, 0x4c26, 0x646e, 0x7c02, 0x3492,
 0x344c, 0x6428, 0x7c70, 0x1c04, 0x4c94,
 0x4c4c, 0x7c28, 0x1c70, 0x3404, 0x6494,
 0x644c, 0x1c28, 0x3470, 0x4c04, 0x7c94,
 0x7c4c, 0x3428, 0x4c70, 0x6404, 0x1c94,
 0x1c4c, 0x4c28, 0x6470, 0x7c04, 0x3494,
 0x344e, 0x642a, 0x7c72, 0x1c06, 0x4c96,
 0x4c4e, 0x7c2a, 0x1c72, 0x3406, 0x6496,
 0x644e, 0x1c2a, 0x3472, 0x4c06, 0x7c96,
 0x7c4e, 0x342a, 0x4c72, 0x6406, 0x1c96,
 0x1c4e, 0x4c2a, 0x6472, 0x7c06, 0x3496,
 0x3450, 0x642c, 0x7c74, 0x1c08, 0x4c98,
 0x4c50, 0x7c2c, 0x1c74, 0x3408, 0x6498,
 0x6450, 0x1c2c, 0x3474, 0x4c08, 0x7c98,
 0x7c50, 0x342c, 0x4c74, 0x6408, 0x1c98,
 0x1c50, 0x4c2c, 0x6474, 0x7c08, 0x3498,
 0x3452, 0x642e, 0x7c76, 0x1c0a, 0x4c9a,
 0x4c52, 0x7c2e, 0x1c76, 0x340a, 0x649a,
 0x6452, 0x1c2e, 0x3476, 0x4c0a, 0x7c9a,
 0x7c52, 0x342e, 0x4c76, 0x640a, 0x1c9a,
 0x1c52, 0x4c2e, 0x6476, 0x7c0a, 0x349a,
 0x3454, 0x6430, 0x7c78, 0x1c0c, 0x4c9c,
 0x4c54, 0x7c30, 0x1c78, 0x340c, 0x649c,
 0x6454, 0x1c30, 0x3478, 0x4c0c, 0x7c9c,
 0x7c54, 0x3430, 0x4c78, 0x640c, 0x1c9c,
 0x1c54, 0x4c30, 0x6478, 0x7c0c, 0x349c,
 0x3456, 0x6432, 0x7c7a, 0x1c0e, 0x4c9e,
 0x4c56, 0x7c32, 0x1c7a, 0x340e, 0x649e,
 0x6456, 0x1c32, 0x347a, 0x4c0e, 0x7c9e,
 0x7c56, 0x3432, 0x4c7a, 0x640e, 0x1c9e,
 0x1c56, 0x4c32, 0x647a, 0x7c0e, 0x349e,
 0x3458, 0x6434, 0x7c7c, 0x1c10, 0x8400,
 0x4c58, 0x7c34, 0x1c7c, 0x3410, 0x8450,
 0x6458, 0x1c34, 0x347c, 0x4c10, 0x8650,
 0x7c58, 0x3434, 0x4c7c, 0x6410, 0x0428,
 0x1c58, 0x4c34, 0x647c, 0x7c10, 0x0464,
 0x505a, 0x0836, 0x207e, 0x3812, 0x8266,
 0x685a, 0x2036, 0x387e, 0x5012, 0x0002,
 0x085a, 0x3836, 0x507e, 0x6812, 0x003e,
 0x205a, 0x5036, 0x687e, 0x0812, 0x007a,
 0x385a, 0x6836, 0x087e, 0x2012, 0x8216,
 0x505c, 0x0838, 0x2080, 0x3814, 0x8268,
 0x685c, 0x2038, 0x3880, 0x5014, 0x0004,
 0x085c, 0x3838, 0x5080, 0x6814, 0x0040,
 0x205c, 0x5038, 0x6880, 0x0814, 0x007c,
 0x385c, 0x6838, 0x0880, 0x2014, 0x8218,
 0x505e, 0x083a, 0x2082, 0x3816, 0x826a,
 0x685e, 0x203a, 0x3882, 0x5016, 0x0006,
 0x085e, 0x383a, 0x5082, 0x6816, 0x0042,
 0x205e, 0x503a, 0x6882, 0x0816, 0x007e,
 0x385e, 0x683a, 0x0882, 0x2016, 0x821a,
 0x5060, 0x083c, 0x2084, 0x3818, 0x826c,
 0x6860, 0x203c, 0x3884, 0x5018, 0x0008,
 0x0860, 0x383c, 0x5084, 0x6818, 0x0044,
 0x2060, 0x503c, 0x6884, 0x0818, 0x0080,
 0x3860, 0x683c, 0x0884, 0x2018, 0x821c,
 0x5062, 0x083e, 0x2086, 0x381a, 0x826e,
 0x6862, 0x203e, 0x3886, 0x501a, 0x000a,
 0x0862, 0x383e, 0x5086, 0x681a, 0x0046,
 0x2062, 0x503e, 0x6886, 0x081a, 0x0082,
 0x3862, 0x683e, 0x0886, 0x201a, 0x821e,
 0x5064, 0x0840, 0x2088, 0x381c, 0x8270,
 0x6864, 0x2040, 0x3888, 0x501c, 0x000c,
 0x0864, 0x3840, 0x5088, 0x681c, 0x0048,
 0x2064, 0x5040, 0x6888, 0x081c, 0x0084,
 0x3864, 0x6840, 0x0888, 0x201c, 0x8220,
 0x5066, 0x0842, 0x208a, 0x381e, 0x8272,
 0x6866, 0x2042, 0x388a, 0x501e, 0x000e,
 0x0866, 0x3842, 0x508a, 0x681e, 0x004a,
 0x2066, 0x5042, 0x688a, 0x081e, 0x0086,
 0x3866, 0x6842, 0x088a, 0x201e, 0x8222,
 0x5068, 0x0844, 0x208c, 0x3820, 0x8274,
 0x6868, 0x2044, 0x388c, 0x5020, 0x0010,
 0x0868, 0x3844, 0x508c, 0x6820, 0x004c,
 0x2068, 0x5044, 0x688c, 0x0820, 0x0088,
 0x3868, 0x6844, 0x088c, 0x2020, 0x8224,
 0x506a, 0x0846, 0x208e, 0x3822, 0x8276,
 0x686a, 0x2046, 0x388e, 0x5022, 0x0012,
 0x086a, 0x3846, 0x508e, 0x6822, 0x004e,
 0x206a, 0x5046, 0x688e, 0x0822, 0x008a,
 0x386a, 0x6846, 0x088e, 0x2022, 0x8226,
 0x545a, 0x0c36, 0x247e, 0x3c12, 0x807a,
 0x6c5a, 0x2436, 0x3c7e, 0x5412, 0x0402,
 0x0c5a, 0x3c36, 0x547e, 0x6c12, 0x043e,
 0x245a, 0x5436, 0x6c7e, 0x0c12, 0x047a,
 0x3c5a, 0x6c36, 0x0c7e, 0x2412, 0x802a,
 0x545c, 0x0c38, 0x2480, 0x3c14, 0x807c,
 0x6c5c, 0x2438, 0x3c80, 0x5414, 0x0404,
 0x0c5c, 0x3c38, 0x5480, 0x6c14, 0x0440,
 0x245c, 0x5438, 0x6c80, 0x0c14, 0x047c,
 0x3c5c, 0x6c38, 0x0c80, 0x2414, 0x802c,
 0x545e, 0x0c3a, 0x2482, 0x3c16, 0x807e,
 0x6c5e, 0x243a, 0x3c82, 0x5416, 0x0406,
 0x0c5e, 0x3c3a, 0x5482, 0x6c16, 0x0442,
 0x245e, 0x543a, 0x6c82, 0x0c16, 0x047e,
 0x3c5e, 0x6c3a, 0x0c82, 0x2416, 0x802e,
 0x5460, 0x0c3c, 0x2484, 0x3c18, 0x8080,
 0x6c60, 0x243c, 0x3c84, 0x5418, 0x0408,
 0x0c60, 0x3c3c, 0x5484, 0x6c18, 0x0444,
 0x2460, 0x543c, 0x6c84, 0x0c18, 0x0480,
 0x3c60, 0x6c3c, 0x0c84, 0x2418, 0x8030,
 0x5462, 0x0c3e, 0x2486, 0x3c1a, 0x8082,
 0x6c62, 0x243e, 0x3c86, 0x541a, 0x040a,
 0x0c62, 0x3c3e, 0x5486, 0x6c1a, 0x0446,
 0x2462, 0x543e, 0x6c86, 0x0c1a, 0x0482,
 0x3c62, 0x6c3e, 0x0c86, 0x241a, 0x8032,
 0x5464, 0x0c40, 0x2488, 0x3c1c, 0x8084,
 0x6c64, 0x2440, 0x3c88, 0x541c, 0x040c,
 0x0c64, 0x3c40, 0x5488, 0x6c1c, 0x0448,
 0x2464, 0x5440, 0x6c88, 0x0c1c, 0x0484,
 0x3c64, 0x6c40, 0x0c88, 0x241c, 0x8034,
 0x5466, 0x0c42, 0x248a, 0x3c1e, 0x8086,
 0x6c66, 0x2442, 0x3c8a, 0x541e, 0x040e,
 0x0c66, 0x3c42, 0x548a, 0x6c1e, 0x044a,
 0x2466, 0x5442, 0x6c8a, 0x0c1e, 0x0486,
 0x3c66, 0x6c42, 0x0c8a, 0x241e, 0x8036,
 0x5468, 0x0c44, 0x248c, 0x3c20, 0x8088,
 0x6c68, 0x2444, 0x3c8c, 0x5420, 0x0410,
 0x0c68, 0x3c44, 0x548c, 0x6c20, 0x044c,
 0x2468, 0x5444, 0x6c8c, 0x0c20, 0x0488,
 0x3c68, 0x6c44, 0x0c8c, 0x2420, 0x8038,
 0x546a, 0x0c46, 0x248e, 0x3c22, 0x808a,
 0x6c6a, 0x2446, 0x3c8e, 0x5422, 0x0412,
 0x0c6a, 0x3c46, 0x548e, 0x6c22, 0x044e,
 0x246a, 0x5446, 0x6c8e, 0x0c22, 0x048a,
 0x3c6a, 0x6c46, 0x0c8e, 0x2422, 0x803a,
 0x585a, 0x1036, 0x287e, 0x4012, 0x847a,
 0x705a, 0x2836, 0x407e, 0x5812, 0x0016,
 0x105a, 0x4036, 0x587e, 0x7012, 0x0052,
 0x285a, 0x5836, 0x707e, 0x1012, 0x008e,
 0x405a, 0x7036, 0x107e, 0x2812, 0x842a,
 0x585c, 0x1038, 0x2880, 0x4014, 0x847c,
 0x705c, 0x2838, 0x4080, 0x5814, 0x0018,
 0x105c, 0x4038, 0x5880, 0x7014, 0x0054,
 0x285c, 0x5838, 0x7080, 0x1014, 0x0090,
 0x405c, 0x7038, 0x1080, 0x2814, 0x842c,
 0x585e, 0x103a, 0x2882, 0x4016, 0x847e,
 0x705e, 0x283a, 0x4082, 0x5816, 0x001a,
 0x105e, 0x403a, 0x5882, 0x7016, 0x0056,
 0x285e, 0x583a, 0x7082, 0x1016, 0x0092,
 0x405e, 0x703a, 0x1082, 0x2816, 0x842e,
 0x5860, 0x103c, 0x2884, 0x4018, 0x8480,
 0x7060, 0x283c, 0x4084, 0x5818, 0x001c,
 0x1060, 0x403c, 0x5884, 0x7018, 0x0058,
 0x2860, 0x583c, 0x7084, 0x1018, 0x0094,
 0x4060, 0x703c, 0x1084, 0x2818, 0x8430,
 0x5862, 0x103e, 0x2886, 0x401a, 0x8482,
 0x7062, 0x283e, 0x4086, 0x581a, 0x001e,
 0x1062, 0x403e, 0x5886, 0x701a, 0x005a,
 0x2862, 0x583e, 0x7086, 0x101a, 0x0096,
 0x4062, 0x703e, 0x1086, 0x281a, 0x8432,
 0x5864, 0x1040, 0x2888, 0x401c, 0x8484,
 0x7064, 0x2840, 0x4088, 0x581c, 0x0020,
 0x1064, 0x4040, 0x5888, 0x701c, 0x005c,
 0x2864, 0x5840, 0x7088, 0x101c, 0x0098,
 0x4064, 0x7040, 0x1088, 0x281c, 0x8434,
 0x5866, 0x1042, 0x288a, 0x401e, 0x8486,
 0x7066, 0x2842, 0x408a, 0x581e, 0x0022,
 0x1066, 0x4042, 0x588a, 0x701e, 0x005e,
 0x2866, 0x5842, 0x708a, 0x101e, 0x009a,
 0x4066, 0x7042, 0x108a, 0x281e, 0x8436,
 0x5868, 0x1044, 0x288c, 0x4020, 0x8488,
 0x7068, 0x2844, 0x408c, 0x5820, 0x0024,
 0x1068, 0x4044, 0x588c, 0x7020, 0x0060,
 0x2868, 0x5844, 0x708c, 0x1020, 0x009c,
 0x4068, 0x7044, 0x108c, 0x2820, 0x8438,
 0x586a, 0x1046, 0x288e, 0x4022, 0x848a,
 0x706a, 0x2846, 0x408e, 0x5822, 0x0026,
 0x106a, 0x4046, 0x588e, 0x7022, 0x0062,
 0x286a, 0x5846, 0x708e, 0x1022, 0x009e,
 0x406a, 0x7046, 0x108e, 0x2822, 0x843a,
 0x5c5a, 0x1436, 0x2c7e, 0x4412, 0x828e,
 0x745a, 0x2c36, 0x447e, 0x5c12, 0x0416,
 0x145a, 0x4436, 0x5c7e, 0x7412, 0x0452,
 0x2c5a, 0x5c36, 0x747e, 0x1412, 0x048e,
 0x445a, 0x7436, 0x147e, 0x2c12, 0x823e,
 0x5c5c, 0x1438, 0x2c80, 0x4414, 0x8290,
 0x745c, 0x2c38, 0x4480, 0x5c14, 0x0418,
 0x145c, 0x4438, 0x5c80, 0x7414, 0x0454,
 0x2c5c, 0x5c38, 0x7480, 0x1414, 0x0490,
 0x445c, 0x7438, 0x1480, 0x2c14, 0x8240,
 0x5c5e, 0x143a, 0x2c82, 0x4416, 0x8292,
 0x745e, 0x2c3a, 0x4482, 0x5c16, 0x041a,
 0x145e, 0x443a, 0x5c82, 0x7416, 0x0456,
 0x2c5e, 0x5c3a, 0x7482, 0x1416, 0x0492,
 0x445e, 0x743a, 0x1482, 0x2c16, 0x8242,
 0x5c60, 0x143c, 0x2c84, 0x4418, 0x8294,
 0x7460, 0x2c3c, 0x4484, 0x5c18, 0x041c,
 0x1460, 0x443c, 0x5c84, 0x7418, 0x0458,
 0x2c60, 0x5c3c, 0x7484, 0x1418, 0x0494,
 0x4460, 0x743c, 0x1484, 0x2c18, 0x8244,
 0x5c62, 0x143e, 0x2c86, 0x441a, 0x8296,
 0x7462, 0x2c3e, 0x4486, 0x5c1a, 0x041e,
 0x1462, 0x443e, 0x5c86, 0x741a, 0x045a,
 0x2c62, 0x5c3e, 0x7486, 0x141a, 0x0496,
 0x4462, 0x743e, 0x1486, 0x2c1a, 0x8246,
 0x5c64, 0x1440, 0x2c88, 0x441c, 0x8298,
 0x7464, 0x2c40, 0x4488, 0x5c1c, 0x0420,
 0x1464, 0x4440, 0x5c88, 0x741c, 0x045c,
 0x2c64, 0x5c40, 0x7488, 0x141c, 0x0498,
 0x4464, 0x7440, 0x1488, 0x2c1c, 0x8248,
 0x5c66, 0x1442, 0x2c8a, 0x441e, 0x829a,
 0x7466, 0x2c42, 0x448a, 0x5c1e, 0x0422,
 0x1466, 0x4442, 0x5c8a, 0x741e, 0x045e,
 0x2c66, 0x5c42, 0x748a, 0x141e, 0x049a,
 0x4466, 0x7442, 0x148a, 0x2c1e, 0x824a,
 0x5c68, 0x1444, 0x2c8c, 0x4420, 0x829c,
 0x7468, 0x2c44, 0x448c, 0x5c20, 0x0424,
 0x1468, 0x4444, 0x5c8c, 0x7420, 0x0460,
 0x2c68, 0x5c44, 0x748c, 0x1420, 0x049c,
 0x4468, 0x7444, 0x148c, 0x2c20, 0x824c,
 0x5c6a, 0x1446, 0x2c8e, 0x4422, 0x829e,
 0x746a, 0x2c46, 0x448e, 0x5c22, 0x0426,
 0x146a, 0x4446, 0x5c8e, 0x7422, 0x0462,
 0x2c6a, 0x5c46, 0x748e, 0x1422, 0x049e,
 0x446a, 0x7446, 0x148e, 0x2c22, 0x824e,
 0x605a, 0x1836, 0x307e, 0x4812, 0x8604,
 0x785a, 0x3036, 0x487e, 0x6012, 0x002a,
 0x185a, 0x4836, 0x607e, 0x7812, 0x0066,
 0x305a, 0x6036, 0x787e, 0x1812, 0x8002,
 0x485a, 0x7836, 0x187e, 0x3012, 0x8052,
 0x605c, 0x1838, 0x3080, 0x4814, 0x8608,
 0x785c, 0x3038, 0x4880, 0x6014, 0x002c,
 0x185c, 0x4838, 0x6080, 0x7814, 0x0068,
 0x305c, 0x6038, 0x7880, 0x1814, 0x8004,
 0x485c, 0x7838, 0x1880, 0x3014, 0x8054,
 0x605e, 0x183a, 0x3082, 0x4816, 0x860c,
 0x785e, 0x303a, 0x4882, 0x6016, 0x002e,
 0x185e, 0x483a, 0x6082, 0x7816, 0x006a,
 0x305e, 0x603a, 0x7882, 0x1816, 0x8006,
 0x485e, 0x783a, 0x1882, 0x3016, 0x8056,
 0x6060, 0x183c, 0x3084, 0x4818, 0x8610,
 0x7860, 0x303c, 0x4884, 0x6018, 0x0030,
 0x1860, 0x483c, 0x6084, 0x7818, 0x006c,
 0x3060, 0x603c, 0x7884, 0x1818, 0x8008,
 0x4860, 0x783c, 0x1884, 0x3018, 0x8058,
 0x6062, 0x183e, 0x3086, 0x481a, 0x8614,
 0x7862, 0x303e, 0x4886, 0x601a, 0x0032,
 0x1862, 0x483e, 0x6086, 0x781a, 0x006e,
 0x3062, 0x603e, 0x7886, 0x181a, 0x800a,
 0x4862, 0x783e, 0x1886, 0x301a, 0x805a,
 0x6064, 0x1840, 0x3088, 0x481c, 0x8618,
 0x7864, 0x3040, 0x4888, 0x601c, 0x0034,
 0x1864, 0x4840, 0x6088, 0x781c, 0x0070,
 0x3064, 0x6040, 0x7888, 0x181c, 0x800c,
 0x4864, 0x7840, 0x1888, 0x301c, 0x805c,
 0x6066, 0x1842, 0x308a, 0x481e, 0x861c,
 0x7866, 0x3042, 0x488a, 0x601e, 0x0036,
 0x1866, 0x4842, 0x608a, 0x781e, 0x0072,
 0x3066, 0x6042, 0x788a, 0x181e, 0x800e,
 0x4866, 0x7842, 0x188a, 0x301e, 0x805e,
 0x6068, 0x1844, 0x308c, 0x4820, 0x8620,
 0x7868, 0x3044, 0x488c, 0x6020, 0x0038,
 0x1868, 0x4844, 0x608c, 0x7820, 0x0074,
 0x3068, 0x6044, 0x788c, 0x1820, 0x8010,
 0x4868, 0x7844, 0x188c, 0x3020, 0x8060,
 0x606a, 0x1846, 0x308e, 0x4822, 0x8624,
 0x786a, 0x3046, 0x488e, 0x6022, 0x003a,
 0x186a, 0x4846, 0x608e, 0x7822, 0x0076,
 0x306a, 0x6046, 0x788e, 0x1822, 0x8012,
 0x486a, 0x7846, 0x188e, 0x3022, 0x8062,
 0x645a, 0x1c36, 0x347e, 0x4c12, 0x8654,
 0x7c5a, 0x3436, 0x4c7e, 0x6412, 0x042a,
 0x1c5a, 0x4c36, 0x647e, 0x7c12, 0x0466,
 0x345a, 0x6436, 0x7c7e, 0x1c12, 0x8402,
 0x4c5a, 0x7c36, 0x1c7e, 0x3412, 0x8452,
 0x645c, 0x1c38, 0x3480, 0x4c14, 0x8658,
 0x7c5c, 0x3438, 0x4c80, 0x6414, 0x042c,
 0x1c5c, 0x4c38, 0x6480, 0x7c14, 0x0468,
 0x345c, 0x6438, 0x7c80, 0x1c14, 0x8404,
 0x4c5c, 0x7c38, 0x1c80, 0x3414, 0x8454,
 0x645e, 0x1c3a, 0x3482, 0x4c16, 0x865c,
 0x7c5e, 0x343a, 0x4c82, 0x6416, 0x042e,
 0x1c5e, 0x4c3a, 0x6482, 0x7c16, 0x046a,
 0x345e, 0x643a, 0x7c82, 0x1c16, 0x8406,
 0x4c5e, 0x7c3a, 0x1c82, 0x3416, 0x8456,
 0x6460, 0x1c3c, 0x3484, 0x4c18, 0x8660,
 0x7c60, 0x343c, 0x4c84, 0x6418, 0x0430,
 0x1c60, 0x4c3c, 0x6484, 0x7c18, 0x046c,
 0x3460, 0x643c, 0x7c84, 0x1c18, 0x8408,
 0x4c60, 0x7c3c, 0x1c84, 0x3418, 0x8458,
 0x6462, 0x1c3e, 0x3486, 0x4c1a, 0x8664,
 0x7c62, 0x343e, 0x4c86, 0x641a, 0x0432,
 0x1c62, 0x4c3e, 0x6486, 0x7c1a, 0x046e,
 0x3462, 0x643e, 0x7c86, 0x1c1a, 0x840a,
 0x4c62, 0x7c3e, 0x1c86, 0x341a, 0x845a,
 0x6464, 0x1c40, 0x3488, 0x4c1c, 0x8668,
 0x7c64, 0x3440, 0x4c88, 0x641c, 0x0434,
 0x1c64, 0x4c40, 0x6488, 0x7c1c, 0x0470,
 0x3464, 0x6440, 0x7c88, 0x1c1c, 0x840c,
 0x4c64, 0x7c40, 0x1c88, 0x341c, 0x845c,
 0x6466, 0x1c42, 0x348a, 0x4c1e, 0x866c,
 0x7c66, 0x3442, 0x4c8a, 0x641e, 0x0436,
 0x1c66, 0x4c42, 0x648a, 0x7c1e, 0x0472,
 0x3466, 0x6442, 0x7c8a, 0x1c1e, 0x840e,
 0x4c66, 0x7c42, 0x1c8a, 0x341e, 0x845e,
 0x6468, 0x1c44, 0x348c, 0x4c20, 0x8670,
 0x7c68, 0x3444, 0x4c8c, 0x6420, 0x0438,
 0x1c68, 0x4c44, 0x648c, 0x7c20, 0x0474,
 0x3468, 0x6444, 0x7c8c, 0x1c20, 0x8410,
 0x4c68, 0x7c44, 0x1c8c, 0x3420, 0x8460,
 0x646a, 0x1c46, 0x348e, 0x4c22, 0x8674,
 0x7c6a, 0x3446, 0x4c8e, 0x6422, 0x043a,
 0x1c6a, 0x4c46, 0x648e, 0x7c22, 0x0476,
 0x346a, 0x6446, 0x7c8e, 0x1c22, 0x8412,
 0x4c6a, 0x7c46, 0x1c8e, 0x3422, 0x8462,
 0x0a48, 0x3a24, 0x526c, 0x6a00, 0x2290,
 0x2248, 0x5224, 0x6a6c, 0x0a00, 0x3a90,
 0x3a48, 0x6a24, 0x0a6c, 0x2200, 0x5290,
 0x5248, 0x0a24, 0x226c, 0x3a00, 0x6a90,
 0x6a48, 0x2224, 0x3a6c, 0x5200, 0x0a90,
 0x0a4a, 0x3a26, 0x526e, 0x6a02, 0x2292,
 0x224a, 0x5226, 0x6a6e, 0x0a02, 0x3a92,
 0x3a4a, 0x6a26, 0x0a6e, 0x2202, 0x5292,
 0x524a, 0x0a26, 0x226e, 0x3a02, 0x6a92,
 0x6a4a, 0x2226, 0x3a6e, 0x5202, 0x0a92,
 0x0a4c, 0x3a28, 0x5270, 0x6a04, 0x2294,
 0x224c, 0x5228, 0x6a70, 0x0a04, 0x3a94,
 0x3a4c, 0x6a28, 0x0a70, 0x2204, 0x5294,
 0x524c, 0x0a28, 0x2270, 0x3a04, 0x6a94,
 0x6a4c, 0x2228, 0x3a70, 0x5204, 0x0a94,
 0x0a4e, 0x3a2a, 0x5272, 0x6a06, 0x2296,
 0x224e, 0x522a, 0x6a72, 0x0a06, 0x3a96,
 0x3a4e, 0x6a2a, 0x0a72, 0x2206, 0x5296,
 0x524e, 0x0a2a, 0x2272, 0x3a06, 0x6a96,
 0x6a4e, 0x222a, 0x3a72, 0x5206, 0x0a96,
 0x0a50, 0x3a2c, 0x5274, 0x6a08, 0x2298,
 0x2250, 0x522c, 0x6a74, 0x0a08, 0x3a98,
 0x3a50, 0x6a2c, 0x0a74, 0x2208, 0x5298,
 0x5250, 0x0a2c, 0x2274, 0x3a08, 0x6a98,
 0x6a50, 0x222c, 0x3a74, 0x5208, 0x0a98,
 0x0a52, 0x3a2e, 0x5276, 0x6a0a, 0x229a,
 0x2252, 0x522e, 0x6a76, 0x0a0a, 0x3a9a,
 0x3a52, 0x6a2e, 0x0a76, 0x220a, 0x529a,
 0x5252, 0x0a2e, 0x2276, 0x3a0a, 0x6a9a,
 0x6a52, 0x222e, 0x3a76, 0x520a, 0x0a9a,
 0x0a54, 0x3a30, 0x5278, 0x6a0c, 0x229c,
 0x2254, 0x5230, 0x6a78, 0x0a0c, 0x3a9c,
 0x3a54, 0x6a30, 0x0a78, 0x220c, 0x529c,
 0x5254, 0x0a30, 0x2278, 0x3a0c, 0x6a9c,
 0x6a54, 0x2230, 0x3a78, 0x520c, 0x0a9c,
 0x0a56, 0x3a32, 0x527a, 0x6a0e, 0x229e,
 0x2256, 0x5232, 0x6a7a, 0x0a0e, 0x3a9e,
 0x3a56, 0x6a32, 0x0a7a, 0x220e, 0x529e,
 0x5256, 0x0a32, 0x227a, 0x3a0e, 0x6a9e,
 0x6a56, 0x2232, 0x3a7a, 0x520e, 0x0a9e,
 0x0a58, 0x3a34, 0x527c, 0x6a10, 0x023c,
 0x2258, 0x5234, 0x6a7c, 0x0a10, 0x0278,
 0x3a58, 0x6a34, 0x0a7c, 0x2210, 0x8414,
 0x5258, 0x0a34, 0x227c, 0x3a10, 0x8464,
 0x6a58, 0x2234, 0x3a7c, 0x5210, 0x0200,
 0x0e48, 0x3e24, 0x566c, 0x6e00, 0x2690,
 0x2648, 0x5624, 0x6e6c, 0x0e00, 0x3e90,
 0x3e48, 0x6e24, 0x0e6c, 0x2600, 0x5690,
 0x5648, 0x0e24, 0x266c, 0x3e00, 0x6e90,
 0x6e48, 0x2624, 0x3e6c, 0x5600, 0x0e90,
 0x0e4a, 0x3e26, 0x566e, 0x6e02, 0x2692,
 0x264a, 0x5626, 0x6e6e, 0x0e02, 0x3e92,
 0x3e4a, 0x6e26, 0x0e6e, 0x2602, 0x5692,
 0x564a, 0x0e26, 0x266e, 0x3e02, 0x6e92,
 0x6e4a, 0x2626, 0x3e6e, 0x5602, 0x0e92,
 0x0e4c, 0x3e28, 0x5670, 0x6e04, 0x2694,
 0x264c, 0x5628, 0x6e70, 0x0e04, 0x3e94,
 0x3e4c, 0x6e28, 0x0e70, 0x2604, 0x5694,
 0x564c, 0x0e28, 0x2670, 0x3e04, 0x6e94,
 0x6e4c, 0x2628, 0x3e70, 0x5604, 0x0e94,
 0x0e4e, 0x3e2a, 0x5672, 0x6e06, 0x2696,
 0x264e, 0x562a, 0x6e72, 0x0e06, 0x3e96,
 0x3e4e, 0x6e2a, 0x0e72, 0x2606, 0x5696,
 0x564e, 0x0e2a, 0x2672, 0x3e06, 0x6e96,
 0x6e4e, 0x262a, 0x3e72, 0x5606, 0x0e96,
 0x0e50, 0x3e2c, 0x5674, 0x6e08, 0x2698,
 0x2650, 0x562c, 0x6e74, 0x0e08, 0x3e98,
 0x3e50, 0x6e2c, 0x0e74, 0x2608, 0x5698,
 0x5650, 0x0e2c, 0x2674, 0x3e08, 0x6e98,
 0x6e50, 0x262c, 0x3e74, 0x5608, 0x0e98,
 0x0e52, 0x3e2e, 0x5676, 0x6e0a, 0x269a,
 0x2652, 0x562e, 0x6e76, 0x0e0a, 0x3e9a,
 0x3e52, 0x6e2e, 0x0e76, 0x260a, 0x569a,
 0x5652, 0x0e2e, 0x2676, 0x3e0a, 0x6e9a,
 0x6e52, 0x262e, 0x3e76, 0x560a, 0x0e9a,
 0x0e54, 0x3e30, 0x5678, 0x6e0c, 0x269c,
 0x2654, 0x5630, 0x6e78, 0x0e0c, 0x3e9c,
 0x3e54, 0x6e30, 0x0e78, 0x260c, 0x569c,
 0x5654, 0x0e30, 0x2678, 0x3e0c, 0x6e9c,
 0x6e54, 0x2630, 0x3e78, 0x560c, 0x0e9c,
 0x0e56, 0x3e32, 0x567a, 0x6e0e, 0x269e,
 0x2656, 0x5632, 0x6e7a, 0x0e0e, 0x3e9e,
 0x3e56, 0x6e32, 0x0e7a, 0x260e, 0x569e,
 0x5656, 0x0e32, 0x267a, 0x3e0e, 0x6e9e,
 0x6e56, 0x2632, 0x3e7a, 0x560e, 0x0e9e,
 0x0e58, 0x3e34, 0x567c, 0x6e10, 0x063c,
 0x2658, 0x5634, 0x6e7c, 0x0e10, 0x0678,
 0x3e58, 0x6e34, 0x0e7c, 0x2610, 0x8228,
 0x5658, 0x0e34, 0x267c, 0x3e10, 0x8278,
 0x6e58, 0x2634, 0x3e7c, 0x5610, 0x0600,
 0x1248, 0x4224, 0x5a6c, 0x7200, 0x2a90,
 0x2a48, 0x5a24, 0x726c, 0x1200, 0x4290,
 0x4248, 0x7224, 0x126c, 0x2a00, 0x5a90,
 0x5a48, 0x1224, 0x2a6c, 0x4200, 0x7290,
 0x7248, 0x2a24, 0x426c, 0x5a00, 0x1290,
 0x124a, 0x4226, 0x5a6e, 0x7202, 0x2a92,
 0x2a4a, 0x5a26, 0x726e, 0x1202, 0x4292,
 0x424a, 0x7226, 0x126e, 0x2a02, 0x5a92,
 0x5a4a, 0x1226, 0x2a6e, 0x4202, 0x7292,
 0x724a, 0x2a26, 0x426e, 0x5a02, 0x1292,
 0x124c, 0x4228, 0x5a70, 0x7204, 0x2a94,
 0x2a4c, 0x5a28, 0x7270, 0x1204, 0x4294,
 0x424c, 0x7228, 0x1270, 0x2a04, 0x5a94,
 0x5a4c, 0x1228, 0x2a70, 0x4204, 0x7294,
 0x724c, 0x2a28, 0x4270, 0x5a04, 0x1294,
 0x124e, 0x422a, 0x5a72, 0x7206, 0x2a96,
 0x2a4e, 0x5a2a, 0x7272, 0x1206, 0x4296,
 0x424e, 0x722a, 0x1272, 0x2a06, 0x5a96,
 0x5a4e, 0x122a, 0x2a72, 0x4206, 0x7296,
 0x724e, 0x2a2a, 0x4272, 0x5a06, 0x1296,
 0x1250, 0x422c, 0x5a74, 0x7208, 0x2a98,
 0x2a50, 0x5a2c, 0x7274, 0x1208, 0x4298,
 0x4250, 0x722c, 0x1274, 0x2a08, 0x5a98,
 0x5a50, 0x122c, 0x2a74, 0x4208, 0x7298,
 0x7250, 0x2a2c, 0x4274, 0x5a08, 0x1298,
 0x1252, 0x422e, 0x5a76, 0x720a, 0x2a9a,
 0x2a52, 0x5a2e, 0x7276, 0x120a, 0x429a,
 0x4252, 0x722e, 0x1276, 0x2a0a, 0x5a9a,
 0x5a52, 0x122e, 0x2a76, 0x420a, 0x729a,
 0x7252, 0x2a2e, 0x4276, 0x5a0a, 0x129a,
 0x1254, 0x4230, 0x5a78, 0x720c, 0x2a9c,
 0x2a54, 0x5a30, 0x7278, 0x120c, 0x429c,
 0x4254, 0x7230, 0x1278, 0x2a0c, 0x5a9c,
 0x5a54, 0x1230, 0x2a78, 0x420c, 0x729c,
 0x7254, 0x2a30, 0x4278, 0x5a0c, 0x129c,
 0x1256, 0x4232, 0x5a7a, 0x720e, 0x2a9e,
 0x2a56, 0x5a32, 0x727a, 0x120e, 0x429e,
 0x4256, 0x7232, 0x127a, 0x2a0e, 0x5a9e,
 0x5a56, 0x1232, 0x2a7a, 0x420e, 0x729e,
 0x7256, 0x2a32, 0x427a, 0x5a0e, 0x129e,
 0x1258, 0x4234, 0x5a7c, 0x7210, 0x0250,
 0x2a58, 0x5a34, 0x727c, 0x1210, 0x028c,
 0x4258, 0x7234, 0x127c, 0x2a10, 0x803c,
 0x5a58, 0x1234, 0x2a7c, 0x4210, 0x808c,
 0x7258, 0x2a34, 0x427c, 0x5a10, 0x0214,
 0x1648, 0x4624, 0x5e6c, 0x7600, 0x2e90,
 0x2e48, 0x5e24, 0x766c, 0x1600, 0x4690,
 0x4648, 0x7624, 0x166c, 0x2e00, 0x5e90,
 0x5e48, 0x1624, 0x2e6c, 0x4600, 0x7690,
 0x7648, 0x2e24, 0x466c, 0x5e00, 0x1690,
 0x164a, 0x4626, 0x5e6e, 0x7602, 0x2e92,
 0x2e4a, 0x5e26, 0x766e, 0x1602, 0x4692,
 0x464a, 0x7626, 0x166e, 0x2e02, 0x5e92,
 0x5e4a, 0x1626, 0x2e6e, 0x4602, 0x7692,
 0x764a, 0x2e26, 0x466e, 0x5e02, 0x1692,
 0x164c, 0x4628, 0x5e70, 0x7604, 0x2e94,
 0x2e4c, 0x5e28, 0x7670, 0x1604, 0x4694,
 0x464c, 0x7628, 0x1670, 0x2e04, 0x5e94,
 0x5e4c, 0x1628, 0x2e70, 0x4604, 0x7694,
 0x764c, 0x2e28, 0x4670, 0x5e04, 0x1694,
 0x164e, 0x462a, 0x5e72, 0x7606, 0x2e96,
 0x2e4e, 0x5e2a, 0x7672, 0x1606, 0x4696,
 0x464e, 0x762a, 0x1672, 0x2e06, 0x5e96,
 0x5e4e, 0x162a, 0x2e72, 0x4606, 0x7696,
 0x764e, 0x2e2a, 0x4672, 0x5e06, 0x1696,
 0x1650, 0x462c, 0x5e74, 0x7608, 0x2e98,
 0x2e50, 0x5e2c, 0x7674, 0x1608, 0x4698,
 0x4650, 0x762c, 0x1674, 0x2e08, 0x5e98,
 0x5e50, 0x162c, 0x2e74, 0x4608, 0x7698,
 0x7650, 0x2e2c, 0x4674, 0x5e08, 0x1698,
 0x1652, 0x462e, 0x5e76, 0x760a, 0x2e9a,
 0x2e52, 0x5e2e, 0x7676, 0x160a, 0x469a,
 0x4652, 0x762e, 0x1676, 0x2e0a, 0x5e9a,
 0x5e52, 0x162e, 0x2e76, 0x460a, 0x769a,
 0x7652, 0x2e2e, 0x4676, 0x5e0a, 0x169a,
 0x1654, 0x4630, 0x5e78, 0x760c, 0x2e9c,
 0x2e54, 0x5e30, 0x7678, 0x160c, 0x469c,
 0x4654, 0x7630, 0x1678, 0x2e0c, 0x5e9c,
 0x5e54, 0x1630, 0x2e78, 0x460c, 0x769c,
 0x7654, 0x2e30, 0x4678, 0x5e0c, 0x169c,
 0x1656, 0x4632, 0x5e7a, 0x760e, 0x2e9e,
 0x2e56, 0x5e32, 0x767a, 0x160e, 0x469e,
 0x4656, 0x7632, 0x167a, 0x2e0e, 0x5e9e,
 0x5e56, 0x1632, 0x2e7a, 0x460e, 0x769e,
 0x7656, 0x2e32, 0x467a, 0x5e0e, 0x169e,
 0x1658, 0x4634, 0x5e7c, 0x7610, 0x0650,
 0x2e58, 0x5e34, 0x767c, 0x1610, 0x068c,
 0x4658, 0x7634, 0x167c, 0x2e10, 0x843c,
 0x5e58, 0x1634, 0x2e7c, 0x4610, 0x848c,
 0x7658, 0x2e34, 0x467c, 0x5e10, 0x0614,
 0x1a48, 0x4a24, 0x626c, 0x7a00, 0x3290,
 0x3248, 0x6224, 0x7a6c, 0x1a00, 0x4a90,
 0x4a48, 0x7a24, 0x1a6c, 0x3200, 0x6290,
 0x6248, 0x1a24, 0x326c, 0x4a00, 0x7a90,
 0x7a48, 0x3224, 0x4a6c, 0x6200, 0x1a90,
 0x1a4a, 0x4a26, 0x626e, 0x7a02, 0x3292,
 0x324a, 0x6226, 0x7a6e, 0x1a02, 0x4a92,
 0x4a4a, 0x7a26, 0x1a6e, 0x3202, 0x6292,
 0x624a, 0x1a26, 0x326e, 0x4a02, 0x7a92,
 0x7a4a, 0x3226, 0x4a6e, 0x6202, 0x1a92,
 0x1a4c, 0x4a28, 0x6270, 0x7a04, 0x3294,
 0x324c, 0x6228, 0x7a70, 0x1a04, 0x4a94,
 0x4a4c, 0x7a28, 0x1a70, 0x3204, 0x6294,
 0x624c, 0x1a28, 0x3270, 0x4a04, 0x7a94,
 0x7a4c, 0x3228, 0x4a70, 0x6204, 0x1a94,
 0x1a4e, 0x4a2a, 0x6272, 0x7a06, 0x3296,
 0x324e, 0x622a, 0x7a72, 0x1a06, 0x4a96,
 0x4a4e, 0x7a2a, 0x1a72, 0x3206, 0x6296,
 0x624e, 0x1a2a, 0x3272, 0x4a06, 0x7a96,
 0x7a4e, 0x322a, 0x4a72, 0x6206, 0x1a96,
 0x1a50, 0x4a2c, 0x6274, 0x7a08, 0x3298,
 0x3250, 0x622c, 0x7a74, 0x1a08, 0x4a98,
 0x4a50, 0x7a2c, 0x1a74, 0x3208, 0x6298,
 0x6250, 0x1a2c, 0x3274, 0x4a08, 0x7a98,
 0x7a50, 0x322c, 0x4a74, 0x6208, 0x1a98,
 0x1a52, 0x4a2e, 0x6276, 0x7a0a, 0x329a,
 0x3252, 0x622e, 0x7a76, 0x1a0a, 0x4a9a,
 0x4a52, 0x7a2e, 0x1a76, 0x320a, 0x629a,
 0x6252, 0x1a2e, 0x3276, 0x4a0a, 0x7a9a,
 0x7a52, 0x322e, 0x4a76, 0x620a, 0x1a9a,
 0x1a54, 0x4a30, 0x6278, 0x7a0c, 0x329c,
 0x3254, 0x6230, 0x7a78, 0x1a0c, 0x4a9c,
 0x4a54, 0x7a30, 0x1a78, 0x320c, 0x629c,
 0x6254, 0x1a30, 0x3278, 0x4a0c, 0x7a9c,
 0x7a54, 0x3230, 0x4a78, 0x620c, 0x1a9c,
 0x1a56, 0x4a32, 0x627a, 0x7a0e, 0x329e,
 0x3256, 0x6232, 0x7a7a, 0x1a0e, 0x4a9e,
 0x4a56, 0x7a32, 0x1a7a, 0x320e, 0x629e,
 0x6256, 0x1a32, 0x327a, 0x4a0e, 0x7a9e,
 0x7a56, 0x3232, 0x4a7a, 0x620e, 0x1a9e,
 0x1a58, 0x4a34, 0x627c, 0x7a10, 0x0264,
 0x3258, 0x6234, 0x7a7c, 0x1a10, 0x8200,
 0x4a58, 0x7a34, 0x1a7c, 0x3210, 0x8250,
 0x6258, 0x1a34, 0x327c, 0x4a10, 0x8628,
 0x7a58, 0x3234, 0x4a7c, 0x6210, 0x0228,
 0x1e48, 0x4e24, 0x666c, 0x7e00, 0x3690,
 0x3648, 0x6624, 0x7e6c, 0x1e00, 0x4e90,
 0x4e48, 0x7e24, 0x1e6c, 0x3600, 0x6690,
 0x6648, 0x1e24, 0x366c, 0x4e00, 0x7e90,
 0x7e48, 0x3624, 0x4e6c, 0x6600, 0x1e90,
 0x1e4a, 0x4e26, 0x666e, 0x7e02, 0x3692,
 0x364a, 0x6626, 0x7e6e, 0x1e02, 0x4e92,
 0x4e4a, 0x7e26, 0x1e6e, 0x3602, 0x6692,
 0x664a, 0x1e26, 0x366e, 0x4e02, 0x7e92,
 0x7e4a, 0x3626, 0x4e6e, 0x6602, 0x1e92,
 0x1e4c, 0x4e28, 0x6670, 0x7e04, 0x3694,
 0x364c, 0x6628, 0x7e70, 0x1e04, 0x4e94,
 0x4e4c, 0x7e28, 0x1e70, 0x3604, 0x6694,
 0x664c, 0x1e28, 0x3670, 0x4e04, 0x7e94,
 0x7e4c, 0x3628, 0x4e70, 0x6604, 0x1e94,
 0x1e4e, 0x4e2a, 0x6672, 0x7e06, 0x3696,
 0x364e, 0x662a, 0x7e72, 0x1e06, 0x4e96,
 0x4e4e, 0x7e2a, 0x1e72, 0x3606, 0x6696,
 0x664e, 0x1e2a, 0x3672, 0x4e06, 0x7e96,
 0x7e4e, 0x362a, 0x4e72, 0x6606, 0x1e96,
 0x1e50, 0x4e2c, 0x6674, 0x7e08, 0x3698,
 0x3650, 0x662c, 0x7e74, 0x1e08, 0x4e98,
 0x4e50, 0x7e2c, 0x1e74, 0x3608, 0x6698,
 0x6650, 0x1e2c, 0x3674, 0x4e08, 0x7e98,
 0x7e50, 0x362c, 0x4e74, 0x6608, 0x1e98,
 0x1e52, 0x4e2e, 0x6676, 0x7e0a, 0x369a,
 0x3652, 0x662e, 0x7e76, 0x1e0a, 0x4e9a,
 0x4e52, 0x7e2e, 0x1e76, 0x360a, 0x669a,
 0x6652, 0x1e2e, 0x3676, 0x4e0a, 0x7e9a,
 0x7e52, 0x362e, 0x4e76, 0x660a, 0x1e9a,
 0x1e54, 0x4e30, 0x6678, 0x7e0c, 0x369c,
 0x3654, 0x6630, 0x7e78, 0x1e0c, 0x4e9c,
 0x4e54, 0x7e30, 0x1e78, 0x360c, 0x669c,
 0x6654, 0x1e30, 0x3678, 0x4e0c, 0x7e9c,
 0x7e54, 0x3630, 0x4e78, 0x660c, 0x1e9c,
 0x1e56, 0x4e32, 0x667a, 0x7e0e, 0x369e,
 0x3656, 0x6632, 0x7e7a, 0x1e0e, 0x4e9e,
 0x4e56, 0x7e32, 0x1e7a, 0x360e, 0x669e,
 0x6656, 0x1e32, 0x367a, 0x4e0e, 0x7e9e,
 0x7e56, 0x3632, 0x4e7a, 0x660e, 0x1e9e,
 0x1e58, 0x4e34, 0x667c, 0x7e10, 0x0664,
 0x3658, 0x6634, 0x7e7c, 0x1e10, 0x8014,
 0x4e58, 0x7e34, 0x1e7c, 0x3610, 0x8064,
 0x6658, 0x1e34, 0x367c, 0x4e10, 0x8678,
 0x7e58, 0x3634, 0x4e7c, 0x6610, 0x0628,
 0x3a5a, 0x6a36, 0x0a7e, 0x2212, 0x8416,
 0x525a, 0x0a36, 0x227e, 0x3a12, 0x8466,
 0x6a5a, 0x2236, 0x3a7e, 0x5212, 0x0202,
 0x0a5a, 0x3a36, 0x527e, 0x6a12, 0x023e,
 0x225a, 0x5236, 0x6a7e, 0x0a12, 0x027a,
 0x3a5c, 0x6a38, 0x0a80, 0x2214, 0x8418,
 0x525c, 0x0a38, 0x2280, 0x3a14, 0x8468,
 0x6a5c, 0x2238, 0x3a80, 0x5214, 0x0204,
 0x0a5c, 0x3a38, 0x5280, 0x6a14, 0x0240,
 0x225c, 0x5238, 0x6a80, 0x0a14, 0x027c,
 0x3a5e, 0x6a3a, 0x0a82, 0x2216, 0x841a,
 0x525e, 0x0a3a, 0x2282, 0x3a16, 0x846a,
 0x6a5e, 0x223a, 0x3a82, 0x5216, 0x0206,
 0x0a5e, 0x3a3a, 0x5282, 0x6a16, 0x0242,
 0x225e, 0x523a, 0x6a82, 0x0a16, 0x027e,
 0x3a60, 0x6a3c, 0x0a84, 0x2218, 0x841c,
 0x5260, 0x0a3c, 0x2284, 0x3a18, 0x846c,
 0x6a60, 0x223c, 0x3a84, 0x5218, 0x0208,
 0x0a60, 0x3a3c, 0x5284, 0x6a18, 0x0244,
 0x2260, 0x523c, 0x6a84, 0x0a18, 0x0280,
 0x3a62, 0x6a3e, 0x0a86, 0x221a, 0x841e,
 0x5262, 0x0a3e, 0x2286, 0x3a1a, 0x846e,
 0x6a62, 0x223e, 0x3a86, 0x521a, 0x020a,
 0x0a62, 0x3a3e, 0x5286, 0x6a1a, 0x0246,
 0x2262, 0x523e, 0x6a86, 0x0a1a, 0x0282,
 0x3a64, 0x6a40, 0x0a88, 0x221c, 0x8420,
 0x5264, 0x0a40, 0x2288, 0x3a1c, 0x8470,
 0x6a64, 0x2240, 0x3a88, 0x521c, 0x020c,
 0x0a64, 0x3a40, 0x5288, 0x6a1c, 0x0248,
 0x2264, 0x5240, 0x6a88, 0x0a1c, 0x0284,
 0x3a66, 0x6a42, 0x0a8a, 0x221e, 0x8422,
 0x5266, 0x0a42, 0x228a, 0x3a1e, 0x8472,
 0x6a66, 0x2242, 0x3a8a, 0x521e, 0x020e,
 0x0a66, 0x3a42, 0x528a, 0x6a1e, 0x024a,
 0x2266, 0x5242, 0x6a8a, 0x0a1e, 0x0286,
 0x3a68, 0x6a44, 0x0a8c, 0x2220, 0x8424,
 0x5268, 0x0a44, 0x228c, 0x3a20, 0x8474,
 0x6a68, 0x2244, 0x3a8c, 0x5220, 0x0210,
 0x0a68, 0x3a44, 0x528c, 0x6a20, 0x024c,
 0x2268, 0x5244, 0x6a8c, 0x0a20, 0x0288,
 0x3a6a, 0x6a46, 0x0a8e, 0x2222, 0x8426,
 0x526a, 0x0a46, 0x228e, 0x3a22, 0x8476,
 0x6a6a, 0x2246, 0x3a8e, 0x5222, 0x0212,
 0x0a6a, 0x3a46, 0x528e, 0x6a22, 0x024e,
 0x226a, 0x5246, 0x6a8e, 0x0a22, 0x028a,
 0x3e5a, 0x6e36, 0x0e7e, 0x2612, 0x822a,
 0x565a, 0x0e36, 0x267e, 0x3e12, 0x827a,
 0x6e5a, 0x2636, 0x3e7e, 0x5612, 0x0602,
 0x0e5a, 0x3e36, 0x567e, 0x6e12, 0x063e,
 0x265a, 0x5636, 0x6e7e, 0x0e12, 0x067a,
 0x3e5c, 0x6e38, 0x0e80, 0x2614, 0x822c,
 0x565c, 0x0e38, 0x2680, 0x3e14, 0x827c,
 0x6e5c, 0x2638, 0x3e80, 0x5614, 0x0604,
 0x0e5c, 0x3e38, 0x5680, 0x6e14, 0x0640,
 0x265c, 0x5638, 0x6e80, 0x0e14, 0x067c,
 0x3e5e, 0x6e3a, 0x0e82, 0x2616, 0x822e,
 0x565e, 0x0e3a, 0x2682, 0x3e16, 0x827e,
 0x6e5e, 0x263a, 0x3e82, 0x5616, 0x0606,
 0x0e5e, 0x3e3a, 0x5682, 0x6e16, 0x0642,
 0x265e, 0x563a, 0x6e82, 0x0e16, 0x067e,
 0x3e60, 0x6e3c, 0x0e84, 0x2618, 0x8230,
 0x5660, 0x0e3c, 0x2684, 0x3e18, 0x8280,
 0x6e60, 0x263c, 0x3e84, 0x5618, 0x0608,
 0x0e60, 0x3e3c, 0x5684, 0x6e18, 0x0644,
 0x2660, 0x563c, 0x6e84, 0x0e18, 0x0680,
 0x3e62, 0x6e3e, 0x0e86, 0x261a, 0x8232,
 0x5662, 0x0e3e, 0x2686, 0x3e1a, 0x8282,
 0x6e62, 0x263e, 0x3e86, 0x561a, 0x060a,
 0x0e62, 0x3e3e, 0x5686, 0x6e1a, 0x0646,
 0x2662, 0x563e, 0x6e86, 0x0e1a, 0x0682,
 0x3e64, 0x6e40, 0x0e88, 0x261c, 0x8234,
 0x5664, 0x0e40, 0x2688, 0x3e1c, 0x8284,
 0x6e64, 0x2640, 0x3e88, 0x561c, 0x060c,
 0x0e64, 0x3e40, 0x5688, 0x6e1c, 0x0648,
 0x2664, 0x5640, 0x6e88, 0x0e1c, 0x0684,
 0x3e66, 0x6e42, 0x0e8a, 0x261e, 0x8236,
 0x5666, 0x0e42, 0x268a, 0x3e1e, 0x8286,
 0x6e66, 0x2642, 0x3e8a, 0x561e, 0x060e,
 0x0e66, 0x3e42, 0x568a, 0x6e1e, 0x064a,
 0x2666, 0x5642, 0x6e8a, 0x0e1e, 0x0686,
 0x3e68, 0x6e44, 0x0e8c, 0x2620, 0x8238,
 0x5668, 0x0e44, 0x268c, 0x3e20, 0x8288,
 0x6e68, 0x2644, 0x3e8c, 0x5620, 0x0610,
 0x0e68, 0x3e44, 0x568c, 0x6e20, 0x064c,
 0x2668, 0x5644, 0x6e8c, 0x0e20, 0x0688,
 0x3e6a, 0x6e46, 0x0e8e, 0x2622, 0x823a,
 0x566a, 0x0e46, 0x268e, 0x3e22, 0x828a,
 0x6e6a, 0x2646, 0x3e8e, 0x5622, 0x0612,
 0x0e6a, 0x3e46, 0x568e, 0x6e22, 0x064e,
 0x266a, 0x5646, 0x6e8e, 0x0e22, 0x068a,
 0x425a, 0x7236, 0x127e, 0x2a12, 0x803e,
 0x5a5a, 0x1236, 0x2a7e, 0x4212, 0x808e,
 0x725a, 0x2a36, 0x427e, 0x5a12, 0x0216,
 0x125a, 0x4236, 0x5a7e, 0x7212, 0x0252,
 0x2a5a, 0x5a36, 0x727e, 0x1212, 0x028e,
 0x425c, 0x7238, 0x1280, 0x2a14, 0x8040,
 0x5a5c, 0x1238, 0x2a80, 0x4214, 0x8090,
 0x725c, 0x2a38, 0x4280, 0x5a14, 0x0218,
 0x125c, 0x4238, 0x5a80, 0x7214, 0x0254,
 0x2a5c, 0x5a38, 0x7280, 0x1214, 0x0290,
 0x425e, 0x723a, 0x1282, 0x2a16, 0x8042,
 0x5a5e, 0x123a, 0x2a82, 0x4216, 0x8092,
 0x725e, 0x2a3a, 0x4282, 0x5a16, 0x021a,
 0x125e, 0x423a, 0x5a82, 0x7216, 0x0256,
 0x2a5e, 0x5a3a, 0x7282, 0x1216, 0x0292,
 0x4260, 0x723c, 0x1284, 0x2a18, 0x8044,
 0x5a60, 0x123c, 0x2a84, 0x4218, 0x8094,
 0x7260, 0x2a3c, 0x4284, 0x5a18, 0x021c,
 0x1260, 0x423c, 0x5a84, 0x7218, 0x0258,
 0x2a60, 0x5a3c, 0x7284, 0x1218, 0x0294,
 0x4262, 0x723e, 0x1286, 0x2a1a, 0x8046,
 0x5a62, 0x123e, 0x2a86, 0x421a, 0x8096,
 0x7262, 0x2a3e, 0x4286, 0x5a1a, 0x021e,
 0x1262, 0x423e, 0x5a86, 0x721a, 0x025a,
 0x2a62, 0x5a3e, 0x7286, 0x121a, 0x0296,
 0x4264, 0x7240, 0x1288, 0x2a1c, 0x8048,
 0x5a64, 0x1240, 0x2a88, 0x421c, 0x8098,
 0x7264, 0x2a40, 0x4288, 0x5a1c, 0x0220,
 0x1264, 0x4240, 0x5a88, 0x721c, 0x025c,
 0x2a64, 0x5a40, 0x7288, 0x121c, 0x0298,
 0x4266, 0x7242, 0x128a, 0x2a1e, 0x804a,
 0x5a66, 0x1242, 0x2a8a, 0x421e, 0x809a,
 0x7266, 0x2a42, 0x428a, 0x5a1e, 0x0222,
 0x1266, 0x4242, 0x5a8a, 0x721e, 0x025e,
 0x2a66, 0x5a42, 0x728a, 0x121e, 0x029a,
 0x4268, 0x7244, 0x128c, 0x2a20, 0x804c,
 0x5a68, 0x1244, 0x2a8c, 0x4220, 0x809c,
 0x7268, 0x2a44, 0x428c, 0x5a20, 0x0224,
 0x1268, 0x4244, 0x5a8c, 0x7220, 0x0260,
 0x2a68, 0x5a44, 0x728c, 0x1220, 0x029c,
 0x426a, 0x7246, 0x128e, 0x2a22, 0x804e,
 0x5a6a, 0x1246, 0x2a8e, 0x4222, 0x809e,
 0x726a, 0x2a46, 0x428e, 0x5a22, 0x0226,
 0x126a, 0x4246, 0x5a8e, 0x7222, 0x0262,
 0x2a6a, 0x5a46, 0x728e, 0x1222, 0x029e,
 0x465a, 0x7636, 0x167e, 0x2e12, 0x843e,
 0x5e5a, 0x1636, 0x2e7e, 0x4612, 0x848e,
 0x765a, 0x2e36, 0x467e, 0x5e12, 0x0616,
 0x165a, 0x4636, 0x5e7e, 0x7612, 0x0652,
 0x2e5a, 0x5e36, 0x767e, 0x1612, 0x068e,
 0x465c, 0x7638, 0x1680, 0x2e14, 0x8440,
 0x5e5c, 0x1638, 0x2e80, 0x4614, 0x8490,
 0x765c, 0x2e38, 0x4680, 0x5e14, 0x0618,
 0x165c, 0x4638, 0x5e80, 0x7614, 0x0654,
 0x2e5c, 0x5e38, 0x7680, 0x1614, 0x0690,
 0x465e, 0x763a, 0x1682, 0x2e16, 0x8442,
 0x5e5e, 0x163a, 0x2e82, 0x4616, 0x8492,
 0x765e, 0x2e3a, 0x4682, 0x5e16, 0x061a,
 0x165e, 0x463a, 0x5e82, 0x7616, 0x0656,
 0x2e5e, 0x5e3a, 0x7682, 0x1616, 0x0692,
 0x4660, 0x763c, 0x1684, 0x2e18, 0x8444,
 0x5e60, 0x163c, 0x2e84, 0x4618, 0x8494,
 0x7660, 0x2e3c, 0x4684, 0x5e18, 0x061c,
 0x1660, 0x463c, 0x5e84, 0x7618, 0x0658,
 0x2e60, 0x5e3c, 0x7684, 0x1618, 0x0694,
 0x4662, 0x763e, 0x1686, 0x2e1a, 0x8446,
 0x5e62, 0x163e, 0x2e86, 0x461a, 0x8496,
 0x7662, 0x2e3e, 0x4686, 0x5e1a, 0x061e,
 0x1662, 0x463e, 0x5e86, 0x761a, 0x065a,
 0x2e62, 0x5e3e, 0x7686, 0x161a, 0x0696,
 0x4664, 0x7640, 0x1688, 0x2e1c, 0x8448,
 0x5e64, 0x1640, 0x2e88, 0x461c, 0x8498,
 0x7664, 0x2e40, 0x4688, 0x5e1c, 0x0620,
 0x1664, 0x4640, 0x5e88, 0x761c, 0x065c,
 0x2e64, 0x5e40, 0x7688, 0x161c, 0x0698,
 0x4666, 0x7642, 0x168a, 0x2e1e, 0x844a,
 0x5e66, 0x1642, 0x2e8a, 0x461e, 0x849a,
 0x7666, 0x2e42, 0x468a, 0x5e1e, 0x0622,
 0x1666, 0x4642, 0x5e8a, 0x761e, 0x065e,
 0x2e66, 0x5e42, 0x768a, 0x161e, 0x069a,
 0x4668, 0x7644, 0x168c, 0x2e20, 0x844c,
 0x5e68, 0x1644, 0x2e8c, 0x4620, 0x849c,
 0x7668, 0x2e44, 0x468c, 0x5e20, 0x0624,
 0x1668, 0x4644, 0x5e8c, 0x7620, 0x0660,
 0x2e68, 0x5e44, 0x768c, 0x1620, 0x069c,
 0x466a, 0x7646, 0x168e, 0x2e22, 0x844e,
 0x5e6a, 0x1646, 0x2e8e, 0x4622, 0x849e,
 0x766a, 0x2e46, 0x468e, 0x5e22, 0x0626,
 0x166a, 0x4646, 0x5e8e, 0x7622, 0x0662,
 0x2e6a, 0x5e46, 0x768e, 0x1622, 0x069e,
 0x4a5a, 0x7a36, 0x1a7e, 0x3212, 0x8252,
 0x625a, 0x1a36, 0x327e, 0x4a12, 0x862c,
 0x7a5a, 0x3236, 0x4a7e, 0x6212, 0x022a,
 0x1a5a, 0x4a36, 0x627e, 0x7a12, 0x0266,
 0x325a, 0x6236, 0x7a7e, 0x1a12, 0x8202,
 0x4a5c, 0x7a38, 0x1a80, 0x3214, 0x8254,
 0x625c, 0x1a38, 0x3280, 0x4a14, 0x8630,
 0x7a5c, 0x3238, 0x4a80, 0x6214, 0x022c,
 0x1a5c, 0x4a38, 0x6280, 0x7a14, 0x0268,
 0x325c, 0x6238, 0x7a80, 0x1a14, 0x8204,
 0x4a5e, 0x7a3a, 0x1a82, 0x3216, 0x8256,
 0x625e, 0x1a3a, 0x3282, 0x4a16, 0x8634,
 0x7a5e, 0x323a, 0x4a82, 0x6216, 0x022e,
 0x1a5e, 0x4a3a, 0x6282, 0x7a16, 0x026a,
 0x325e, 0x623a, 0x7a82, 0x1a16, 0x8206,
 0x4a60, 0x7a3c, 0x1a84, 0x3218, 0x8258,
 0x6260, 0x1a3c, 0x3284, 0x4a18, 0x8638,
 0x7a60, 0x323c, 0x4a84, 0x6218, 0x0230,
 0x1a60, 0x4a3c, 0x6284, 0x7a18, 0x026c,
 0x3260, 0x623c, 0x7a84, 0x1a18, 0x8208,
 0x4a62, 0x7a3e, 0x1a86, 0x321a, 0x825a,
 0x6262, 0x1a3e, 0x3286, 0x4a1a, 0x863c,
 0x7a62, 0x323e, 0x4a86, 0x621a, 0x0232,
 0x1a62, 0x4a3e, 0x6286, 0x7a1a, 0x026e,
 0x3262, 0x623e, 0x7a86, 0x1a1a, 0x820a,
 0x4a64, 0x7a40, 0x1a88, 0x321c, 0x825c,
 0x6264, 0x1a40, 0x3288, 0x4a1c, 0x8640,
 0x7a64, 0x3240, 0x4a88, 0x621c, 0x0234,
 0x1a64, 0x4a40, 0x6288, 0x7a1c, 0x0270,
 0x3264, 0x6240, 0x7a88, 0x1a1c, 0x820c,
 0x4a66, 0x7a42, 0x1a8a, 0x321e, 0x825e,
 0x6266, 0x1a42, 0x328a, 0x4a1e, 0x8644,
 0x7a66, 0x3242, 0x4a8a, 0x621e, 0x0236,
 0x1a66, 0x4a42, 0x628a, 0x7a1e, 0x0272,
 0x3266, 0x6242, 0x7a8a, 0x1a1e, 0x820e,
 0x4a68, 0x7a44, 0x1a8c, 0x3220, 0x8260,
 0x6268, 0x1a44, 0x328c, 0x4a20, 0x8648,
 0x7a68, 0x3244, 0x4a8c, 0x6220, 0x0238,
 0x1a68, 0x4a44, 0x628c, 0x7a20, 0x0274,
 0x3268, 0x6244, 0x7a8c, 0x1a20, 0x8210,
 0x4a6a, 0x7a46, 0x1a8e, 0x3222, 0x8262,
 0x626a, 0x1a46, 0x328e, 0x4a22, 0x864c,
 0x7a6a, 0x3246, 0x4a8e, 0x6222, 0x023a,
 0x1a6a, 0x4a46, 0x628e, 0x7a22, 0x0276,
 0x326a, 0x6246, 0x7a8e, 0x1a22, 0x8212,
 0x4e5a, 0x7e36, 0x1e7e, 0x3612, 0x8066,
 0x665a, 0x1e36, 0x367e, 0x4e12, 0x867c,
 0x7e5a, 0x3636, 0x4e7e, 0x6612, 0x062a,
 0x1e5a, 0x4e36, 0x667e, 0x7e12, 0x0666,
 0x365a, 0x6636, 0x7e7e, 0x1e12, 0x8016,
 0x4e5c, 0x7e38, 0x1e80, 0x3614, 0x8068,
 0x665c, 0x1e38, 0x3680, 0x4e14, 0x8680,
 0x7e5c, 0x3638, 0x4e80, 0x6614, 0x062c,
 0x1e5c, 0x4e38, 0x6680, 0x7e14, 0x0668,
 0x365c, 0x6638, 0x7e80, 0x1e14, 0x8018,
 0x4e5e, 0x7e3a, 0x1e82, 0x3616, 0x806a,
 0x665e, 0x1e3a, 0x3682, 0x4e16, 0x8684,
 0x7e5e, 0x363a, 0x4e82, 0x6616, 0x062e,
 0x1e5e, 0x4e3a, 0x6682, 0x7e16, 0x066a,
 0x365e, 0x663a, 0x7e82, 0x1e16, 0x801a,
 0x4e60, 0x7e3c, 0x1e84, 0x3618, 0x806c,
 0x6660, 0x1e3c, 0x3684, 0x4e18, 0x8688,
 0x7e60, 0x363c, 0x4e84, 0x6618, 0x0630,
 0x1e60, 0x4e3c, 0x6684, 0x7e18, 0x066c,
 0x3660, 0x663c, 0x7e84, 0x1e18, 0x801c,
 0x4e62, 0x7e3e, 0x1e86, 0x361a, 0x806e,
 0x6662, 0x1e3e, 0x3686, 0x4e1a, 0x868c,
 0x7e62, 0x363e, 0x4e86, 0x661a, 0x0632,
 0x1e62, 0x4e3e, 0x6686, 0x7e1a, 0x066e,
 0x3662, 0x663e, 0x7e86, 0x1e1a, 0x801e,
 0x4e64, 0x7e40, 0x1e88, 0x361c, 0x8070,
 0x6664, 0x1e40, 0x3688, 0x4e1c, 0x8690,
 0x7e64, 0x3640, 0x4e88, 0x661c, 0x0634,
 0x1e64, 0x4e40, 0x6688, 0x7e1c, 0x0670,
 0x3664, 0x6640, 0x7e88, 0x1e1c, 0x8020,
 0x4e66, 0x7e42, 0x1e8a, 0x361e, 0x8072,
 0x6666, 0x1e42, 0x368a, 0x4e1e, 0x8694,
 0x7e66, 0x3642, 0x4e8a, 0x661e, 0x0636,
 0x1e66, 0x4e42, 0x668a, 0x7e1e, 0x0672,
 0x3666, 0x6642, 0x7e8a, 0x1e1e, 0x8022,
 0x4e68, 0x7e44, 0x1e8c, 0x3620, 0x8074,
 0x6668, 0x1e44, 0x368c, 0x4e20, 0x8698,
 0x7e68, 0x3644, 0x4e8c, 0x6620, 0x0638,
 0x1e68, 0x4e44, 0x668c, 0x7e20, 0x0674,
 0x3668, 0x6644, 0x7e8c, 0x1e20, 0x8024,
 0x4e6a, 0x7e46, 0x1e8e, 0x3622, 0x8076,
 0x666a, 0x1e46, 0x368e, 0x4e22, 0x869c,
 0x7e6a, 0x3646, 0x4e8e, 0x6622, 0x063a,
 0x1e6a, 0x4e46, 0x668e, 0x7e22, 0x0676,
 0x366a, 0x6646, 0x7e8e, 0x1e22, 0x8026,
};

static const uint16_t dv_place_1080i50[4*12*27*5] = {
 0x1a48, 0x4a24, 0x626c, 0x0200, 0x3290,
 0x2648, 0x5624, 0x6e6c, 0x0e00, 0x3e90,
 0x3248, 0x6224, 0x7a6c, 0x1a00, 0x4a90,
 0x3e48, 0x6e24, 0x026c, 0x2600, 0x5690,
 0x4a48, 0x7a24, 0x0e6c, 0x3200, 0x6290,
 0x5648, 0x0224, 0x1a6c, 0x3e00, 0x6e90,
 0x6248, 0x0e24, 0x266c, 0x4a00, 0x7a90,
 0x6e48, 0x1a24, 0x326c, 0x5600, 0x0290,
 0x7a48, 0x2624, 0x3e6c, 0x6200, 0x0e90,
 0x0248, 0x3224, 0x4a6c, 0x6e00, 0x1a90,
 0x0e48, 0x3e24, 0x566c, 0x7a00, 0x2690,
 0x1a4a, 0x4a26, 0x626e, 0x0202, 0x3292,
 0x264a, 0x5626, 0x6e6e, 0x0e02, 0x3e92,
 0x324a, 0x6226, 0x7a6e, 0x1a02, 0x4a92,
 0x3e4a, 0x6e26, 0x026e, 0x2602, 0x5692,
 0x4a4a, 0x7a26, 0x0e6e, 0x3202, 0x6292,
 0x564a, 0x0226, 0x1a6e, 0x3e02, 0x6e92,
 0x624a, 0x0e26, 0x266e, 0x4a02, 0x7a92,
 0x6e4a, 0x1a26, 0x326e, 0x5602, 0x0292,
 0x7a4a, 0x2626, 0x3e6e, 0x6202, 0x0e92,
 0x024a, 0x3226, 0x4a6e, 0x6e02, 0x1a92,
 0x0e4a, 0x3e26, 0x566e, 0x7a02, 0x2692,
 0x1a4c, 0x4a28, 0x6270, 0x0204, 0x3294,
 0x264c, 0x5628, 0x6e70, 0x0e04, 0x3e94,
 0x324c, 0x6228, 0x7a70, 0x1a04, 0x4a94,
 0x3e4c, 0x6e28, 0x0270, 0x2604, 0x5694,
 0x4a4c, 0x7a28, 0x0e70, 0x3204, 0x6294,
 0x564c, 0x0228, 0x1a70, 0x3e04, 0x6e94,
 0x624c, 0x0e28, 0x2670, 0x4a04, 0x7a94,
 0x6e4c, 0x1a28, 0x3270, 0x5604, 0x0294,
 0x7a4c, 0x2628, 0x3e70, 0x6204, 0x0e94,
 0x024c, 0x3228, 0x4a70, 0x6e04, 0x1a94,
 0x0e4c, 0x3e28, 0x5670, 0x7a04, 0x2694,
 0x1a4e, 0x4a2a, 0x6272, 0x0206, 0x3296,
 0x264e, 0x562a, 0x6e72, 0x0e06, 0x3e96,
 0x324e, 0x622a, 0x7a72, 0x1a06, 0x4a96,
 0x3e4e, 0x6e2a, 0x0272, 0x2606, 0x5696,
 0x4a4e, 0x7a2a, 0x0e72, 0x3206, 0x6296,
 0x564e, 0x022a, 0x1a72, 0x3e06, 0x6e96,
 0x624e, 0x0e2a, 0x2672, 0x4a06, 0x7a96,
 0x6e4e, 0x1a2a, 0x3272, 0x5606, 0x0296,
 0x7a4e, 0x262a, 0x3e72, 0x6206, 0x0e96,
 0x024e, 0x322a, 0x4a72, 0x6e06, 0x1a96,
 0x0e4e, 0x3e2a, 0x5672, 0x7a06, 0x2696,
 0x1a50, 0x4a2c, 0x6274, 0x0208, 0x3298,
 0x2650, 0x562c, 0x6e74, 0x0e08, 0x3e98,
 0x3250, 0x622c, 0x7a74, 0x1a08, 0x4a98,
 0x3e50, 0x6e2c, 0x0274, 0x2608, 0x5698,
 0x4a50, 0x7a2c, 0x0e74, 0x3208, 0x6298,
 0x5650, 0x022c, 0x1a74, 0x3e08, 0x6e98,
 0x6250, 0x0e2c, 0x2674, 0x4a08, 0x7a98,
 0x6e50, 0x1a2c, 0x3274, 0x5608, 0x0298,
 0x7a50, 0x262c, 0x3e74, 0x6208, 0x0e98,
 0x0250, 0x322c, 0x4a74, 0x6e08, 0x1a98,
 0x0e50, 0x3e2c, 0x5674, 0x7a08, 0x2698,
 0x1a52, 0x4a2e, 0x6276, 0x020a, 0x329a,
 0x2652, 0x562e, 0x6e76, 0x0e0a, 0x3e9a,
 0x3252, 0x622e, 0x7a76, 0x1a0a, 0x4a9a,
 0x3e52, 0x6e2e, 0x0276, 0x260a, 0x569a,
 0x4a52, 0x7a2e, 0x0e76, 0x320a, 0x629a,
 0x5652, 0x022e, 0x1a76, 0x3e0a, 0x6e9a,
 0x6252, 0x0e2e, 0x2676, 0x4a0a, 0x7a9a,
 0x6e52, 0x1a2e, 0x3276, 0x560a, 0x029a,
 0x7a52, 0x262e, 0x3e76, 0x620a, 0x0e9a,
 0x0252, 0x322e, 0x4a76, 0x6e0a, 0x1a9a,
 0x0e52, 0x3e2e, 0x5676, 0x7a0a, 0x269a,
 0x1a54, 0x4a30, 0x6278, 0x020c, 0x329c,
 0x2654, 0x5630, 0x6e78, 0x0e0c, 0x3e9c,
 0x3254, 0x6230, 0x7a78, 0x1a0c, 0x4a9c,
 0x3e54, 0x6e30, 0x0278, 0x260c, 0x569c,
 0x4a54, 0x7a30, 0x0e78, 0x320c, 0x629c,
 0x5654, 0x0230, 0x1a78, 0x3e0c, 0x6e9c,
 0x6254, 0x0e30, 0x2678, 0x4a0c, 0x7a9c,
 0x6e54, 0x1a30, 0x3278, 0x560c, 0x029c,
 0x7a54, 0x2630, 0x3e78, 0x620c, 0x0e9c,
 0x0254, 0x3230, 0x4a78, 0x6e0c, 0x1a9c,
 0x0e54, 0x3e30, 0x5678, 0x7a0c, 0x269c,
 0x1a56, 0x4a32, 0x627a, 0x020e, 0x329e,
 0x2656, 0x5632, 0x6e7a, 0x0e0e, 0x3e9e,
 0x3256, 0x6232, 0x7a7a, 0x1a0e, 0x4a9e,
 0x3e56, 0x6e32, 0x027a, 0x260e, 0x569e,
 0x4a56, 0x7a32, 0x0e7a, 0x320e, 0x629e,
 0x5656, 0x0232, 0x1a7a, 0x3e0e, 0x6e9e,
 0x6256, 0x0e32, 0x267a, 0x4a0e, 0x7a9e,
 0x6e56, 0x1a32, 0x327a, 0x560e, 0x029e,
 0x7a56, 0x2632, 0x3e7a, 0x620e, 0x0e9e,
 0x0256, 0x3232, 0x4a7a, 0x6e0e, 0x1a9e,
 0x0e56, 0x3e32, 0x567a, 0x7a0e, 0x269e,
 0x1a58, 0x4a34, 0x627c, 0x0210, 0x32a0,
 0x2658, 0x5634, 0x6e7c, 0x0e10, 0x3ea0,
 0x3258, 0x6234, 0x7a7c, 0x1a10, 0x4aa0,
 0x3e58, 0x6e34, 0x027c, 0x2610, 0x56a0,
 0x4a58, 0x7a34, 0x0e7c, 0x3210, 0x62a0,
 0x5658, 0x0234, 0x1a7c, 0x3e10, 0x6ea0,
 0x6258, 0x0e34, 0x267c, 0x4a10, 0x7aa0,
 0x6e58, 0x1a34, 0x327c, 0x5610, 0x02a0,
 0x7a58, 0x2634, 0x3e7c, 0x6210, 0x0ea0,
 0x0258, 0x3234, 0x4a7c, 0x6e10, 0x1aa0,
 0x0e58, 0x3e34, 0x567c, 0x7a10, 0x26a0,
 0x1e48, 0x4e24, 0x666c, 0x0600, 0x3690,
 0x2a48, 0x5a24, 0x726c, 0x1200, 0x4290,
 0x3648, 0x6624, 0x7e6c, 0x1e00, 0x4e90,
 0x4248, 0x7224, 0x066c, 0x2a00, 0x5a90,
 0x4e48, 0x7e24, 0x126c, 0x3600, 0x6690,
 0x5a48, 0x0624, 0x1e6c, 0x4200, 0x7290,
 0x6648, 0x1224, 0x2a6c, 0x4e00, 0x7e90,
 0x7248, 0x1e24, 0x366c, 0x5a00, 0x0690,
 0x7e48, 0x2a24, 0x426c, 0x6600, 0x1290,
 0x0648, 0x3624, 0x4e6c, 0x7200, 0x1e90,
 0x1248, 0x4224, 0x5a6c, 0x7e00, 0x2a90,
 0x1e4a, 0x4e26, 0x666e, 0x0602, 0x3692,
 0x2a4a, 0x5a26, 0x726e, 0x1202, 0x4292,
 0x364a, 0x6626, 0x7e6e, 0x1e02, 0x4e92,
 0x424a, 0x7226, 0x066e, 0x2a02, 0x5a92,
 0x4e4a, 0x7e26, 0x126e, 0x3602, 0x6692,
 0x5a4a, 0x0626, 0x1e6e, 0x4202, 0x7292,
 0x664a, 0x1226, 0x2a6e, 0x4e02, 0x7e92,
 0x724a, 0x1e26, 0x366e, 0x5a02, 0x0692,
 0x7e4a, 0x2a26, 0x426e, 0x6602, 0x1292,
 0x064a, 0x3626, 0x4e6e, 0x7202, 0x1e92,
 0x124a, 0x4226, 0x5a6e, 0x7e02, 0x2a92,
 0x1e4c, 0x4e28, 0x6670, 0x0604, 0x3694,
 0x2a4c, 0x5a28, 0x7270, 0x1204, 0x4294,
 0x364c, 0x6628, 0x7e70, 0x1e04, 0x4e94,
 0x424c, 0x7228, 0x0670, 0x2a04, 0x5a94,
 0x4e4c, 0x7e28, 0x1270, 0x3604, 0x6694,
 0x5a4c, 0x0628, 0x1e70, 0x4204, 0x7294,
 0x664c, 0x1228, 0x2a70, 0x4e04, 0x7e94,
 0x724c, 0x1e28, 0x3670, 0x5a04, 0x0694,
 0x7e4c, 0x2a28, 0x4270, 0x6604, 0x1294,
 0x064c, 0x3628, 0x4e70, 0x7204, 0x1e94,
 0x124c, 0x4228, 0x5a70, 0x7e04, 0x2a94,
 0x1e4e, 0x4e2a, 0x6672, 0x0606, 0x3696,
 0x2a4e, 0x5a2a, 0x7272, 0x1206, 0x4296,
 0x364e, 0x662a, 0x7e72, 0x1e06, 0x4e96,
 0x424e, 0x722a, 0x0672, 0x2a06, 0x5a96,
 0x4e4e, 0x7e2a, 0x1272, 0x3606, 0x6696,
 0x5a4e, 0x062a, 0x1e72, 0x4206, 0x7296,
 0x664e, 0x122a, 0x2a72, 0x4e06, 0x7e96,
 0x724e, 0x1e2a, 0x3672, 0x5a06, 0x0696,
 0x7e4e, 0x2a2a, 0x4272, 0x6606, 0x1296,
 0x064e, 0x362a, 0x4e72, 0x7206, 0x1e96,
 0x124e, 0x422a, 0x5a72, 0x7e06, 0x2a96,
 0x1e50, 0x4e2c, 0x6674, 0x0608, 0x3698,
 0x2a50, 0x5a2c, 0x7274, 0x1208, 0x4298,
 0x3650, 0x662c, 0x7e74, 0x1e08, 0x4e98,
 0x4250, 0x722c, 0x0674, 0x2a08, 0x5a98,
 0x4e50, 0x7e2c, 0x1274, 0x3608, 0x6698,
 0x5a50, 0x062c, 0x1e74, 0x4208, 0x7298,
 0x6650, 0x122c, 0x2a74, 0x4e08, 0x7e98,
 0x7250, 0x1e2c, 0x3674, 0x5a08, 0x0698,
 0x7e50, 0x2a2c, 0x4274, 0x6608, 0x1298,
 0x0650, 0x362c, 0x4e74, 0x7208, 0x1e98,
 0x1250, 0x422c, 0x5a74, 0x7e08, 0x2a98,
 0x1e52, 0x4e2e, 0x6676, 0x060a, 0x369a,
 0x2a52, 0x5a2e, 0x7276, 0x120a, 0x429a,
 0x3652, 0x662e, 0x7e76, 0x1e0a, 0x4e9a,
 0x4252, 0x722e, 0x0676, 0x2a0a, 0x5a9a,
 0x4e52, 0x7e2e, 0x1276, 0x360a, 0x669a,
 0x5a52, 0x062e, 0x1e76, 0x420a, 0x729a,
 0x6652, 0x122e, 0x2a76, 0x4e0a, 0x7e9a,
 0x7252, 0x1e2e, 0x3676, 0x5a0a, 0x069a,
 0x7e52, 0x2a2e, 0x4276, 0x660a, 0x129a,
 0x0652, 0x362e, 0x4e76, 0x720a, 0x1e9a,
 0x1252, 0x422e, 0x5a76, 0x7e0a, 0x2a9a,
 0x1e54, 0x4e30, 0x6678, 0x060c, 0x369c,
 0x2a54, 0x5a30, 0x7278, 0x120c, 0x429c,
 0x3654, 0x6630, 0x7e78, 0x1e0c, 0x4e9c,
 0x4254, 0x7230, 0x0678, 0x2a0c, 0x5a9c,
 0x4e54, 0x7e30, 0x1278, 0x360c, 0x669c,
 0x5a54, 0x0630, 0x1e78, 0x420c, 0x729c,
 0x6654, 0x1230, 0x2a78, 0x4e0c, 0x7e9c,
 0x7254, 0x1e30, 0x3678, 0x5a0c, 0x069c,
 0x7e54, 0x2a30, 0x4278, 0x660c, 0x129c,
 0x0654, 0x3630, 0x4e78, 0x720c, 0x1e9c,
 0x1254, 0x4230, 0x5a78, 0x7e0c, 0x2a9c,
 0x1e56, 0x4e32, 0x667a, 0x060e, 0x369e,
 0x2a56, 0x5a32, 0x727a, 0x120e, 0x429e,
 0x3656, 0x6632, 0x7e7a, 0x1e0e, 0x4e9e,
 0x4256, 0x7232, 0x067a, 0x2a0e, 0x5a9e,
 0x4e56, 0x7e32, 0x127a, 0x360e, 0x669e,
 0x5a56, 0x0632, 0x1e7a, 0x420e, 0x729e,
 0x6656, 0x1232, 0x2a7a, 0x4e0e, 0x7e9e,
 0x7256, 0x1e32, 0x367a, 0x5a0e, 0x069e,
 0x7e56, 0x2a32, 0x427a, 0x660e, 0x129e,
 0x0656, 0x3632, 0x4e7a, 0x720e, 0x1e9e,
 0x1256, 0x4232, 0x5a7a, 0x7e0e, 0x2a9e,
 0x1e58, 0x4e34, 0x667c, 0x0610, 0x36a0,
 0x2a58, 0x5a34, 0x727c, 0x1210, 0x42a0,
 0x3658, 0x6634, 0x7e7c, 0x1e10, 0x4ea0,
 0x4258, 0x7234, 0x067c, 0x2a10, 0x5aa0,
 0x4e58, 0x7e34, 0x127c, 0x3610, 0x66a0,
 0x5a58, 0x0634, 0x1e7c, 0x4210, 0x72a0,
 0x6658, 0x1234, 0x2a7c, 0x4e10, 0x7ea0,
 0x7258, 0x1e34, 0x367c, 0x5a10, 0x06a0,
 0x7e58, 0x2a34, 0x427c, 0x6610, 0x12a0,
 0x0658, 0x3634, 0x4e7c, 0x7210, 0x1ea0,
 0x1258, 0x4234, 0x5a7c, 0x7e10, 0x2aa0,
 0x2248, 0x5224, 0x6a6c, 0x0a00, 0x3a90,
 0x2e48, 0x5e24, 0x766c, 0x1600, 0x4690,
 0x3a48, 0x6a24, 0x826c, 0x2200, 0x5290,
 0x4648, 0x7624, 0x0a6c, 0x2e00, 0x5e90,
 0x5248, 0x8224, 0x166c, 0x3a00, 0x6a90,
 0x5e48, 0x0a24, 0x226c, 0x4600, 0x7690,
 0x6a48, 0x1624, 0x2e6c, 0x5200, 0x8290,
 0x7648, 0x2224, 0x3a6c, 0x5e00, 0x0a90,
 0x8248, 0x2e24, 0x466c, 0x6a00, 0x1690,
 0x0a48, 0x3a24, 0x526c, 0x7600, 0x2290,
 0x1648, 0x4624, 0x5e6c, 0x8200, 0x2e90,
 0x224a, 0x5226, 0x6a6e, 0x0a02, 0x3a92,
 0x2e4a, 0x5e26, 0x766e, 0x1602, 0x4692,
 0x3a4a, 0x6a26, 0x826e, 0x2202, 0x5292,
 0x464a, 0x7626, 0x0a6e, 0x2e02, 0x5e92,
 0x524a, 0x8226, 0x166e, 0x3a02, 0x6a92,
 0x5e4a, 0x0a26, 0x226e, 0x4602, 0x7692,
 0x6a4a, 0x1626, 0x2e6e, 0x5202, 0x8292,
 0x764a, 0x2226, 0x3a6e, 0x5e02, 0x0a92,
 0x824a, 0x2e26, 0x466e, 0x6a02, 0x1692,
 0x0a4a, 0x3a26, 0x526e, 0x7602, 0x2292,
 0x164a, 0x4626, 0x5e6e, 0x8202, 0x2e92,
 0x224c, 0x5228, 0x6a70, 0x0a04, 0x3a94,
 0x2e4c, 0x5e28, 0x7670, 0x1604, 0x4694,
 0x3a4c, 0x6a28, 0x8270, 0x2204, 0x5294,
 0x464c, 0x7628, 0x0a70, 0x2e04, 0x5e94,
 0x524c, 0x8228, 0x1670, 0x3a04, 0x6a94,
 0x5e4c, 0x0a28, 0x2270, 0x4604, 0x7694,
 0x6a4c, 0x1628, 0x2e70, 0x5204, 0x8294,
 0x764c, 0x2228, 0x3a70, 0x5e04, 0x0a94,
 0x824c, 0x2e28, 0x4670, 0x6a04, 0x1694,
 0x0a4c, 0x3a28, 0x5270, 0x7604, 0x2294,
 0x164c, 0x4628, 0x5e70, 0x8204, 0x2e94,
 0x224e, 0x522a, 0x6a72, 0x0a06, 0x3a96,
 0x2e4e, 0x5e2a, 0x7672, 0x1606, 0x4696,
 0x3a4e, 0x6a2a, 0x8272, 0x2206, 0x5296,
 0x464e, 0x762a, 0x0a72, 0x2e06, 0x5e96,
 0x524e, 0x822a, 0x1672, 0x3a06, 0x6a96,
 0x5e4e, 0x0a2a, 0x2272, 0x4606, 0x7696,
 0x6a4e, 0x162a, 0x2e72, 0x5206, 0x8296,
 0x764e, 0x222a, 0x3a72, 0x5e06, 0x0a96,
 0x824e, 0x2e2a, 0x4672, 0x6a06, 0x1696,
 0x0a4e, 0x3a2a, 0x5272, 0x7606, 0x2296,
 0x164e, 0x462a, 0x5e72, 0x8206, 0x2e96,
 0x2250, 0x522c, 0x6a74, 0x0a08, 0x3a98,
 0x2e50, 0x5e2c, 0x7674, 0x1608, 0x4698,
 0x3a50, 0x6a2c, 0x8274, 0x2208, 0x5298,
 0x4650, 0x762c, 0x0a74, 0x2e08, 0x5e98,
 0x5250, 0x822c, 0x1674, 0x3a08, 0x6a98,
 0x5e50, 0x0a2c, 0x2274, 0x4608, 0x7698,
 0x6a50, 0x162c, 0x2e74, 0x5208, 0x8298,
 0x7650, 0x222c, 0x3a74, 0x5e08, 0x0a98,
 0x8250, 0x2e2c, 0x4674, 0x6a08, 0x1698,
 0x0a50, 0x3a2c, 0x5274, 0x7608, 0x2298,
 0x1650, 0x462c, 0x5e74, 0x8208, 0x2e98,
 0x2252, 0x522e, 0x6a76, 0x0a0a, 0x3a9a,
 0x2e52, 0x5e2e, 0x7676, 0x160a, 0x469a,
 0x3a52, 0x6a2e, 0x8276, 0x220a, 0x529a,
 0x4652, 0x762e, 0x0a76, 0x2e0a, 0x5e9a,
 0x5252, 0x822e, 0x1676, 0x3a0a, 0x6a9a,
 0x5e52, 0x0a2e, 0x2276, 0x460a, 0x769a,
 0x6a52, 0x162e, 0x2e76, 0x520a, 0x829a,
 0x7652, 0x222e, 0x3a76, 0x5e0a, 0x0a9a,
 0x8252, 0x2e2e, 0x4676, 0x6a0a, 0x169a,
 0x0a52, 0x3a2e, 0x5276, 0x760a, 0x229a,
 0x1652, 0x462e, 0x5e76, 0x820a, 0x2e9a,
 0x2254, 0x5230, 0x6a78, 0x0a0c, 0x3a9c,
 0x2e54, 0x5e30, 0x7678, 0x160c, 0x469c,
 0x3a54, 0x6a30, 0x8278, 0x220c, 0x529c,
 0x4654, 0x7630, 0x0a78, 0x2e0c, 0x5e9c,
 0x5254, 0x8230, 0x1678, 0x3a0c, 0x6a9c,
 0x5e54, 0x0a30, 0x2278, 0x460c, 0x769c,
 0x6a54, 0x1630, 0x2e78, 0x520c, 0x829c,
 0x7654, 0x2230, 0x3a78, 0x5e0c, 0x0a9c,
 0x8254, 0x2e30, 0x4678, 0x6a0c, 0x169c,
 0x0a54, 0x3a30, 0x5278, 0x760c, 0x229c,
 0x1654, 0x4630, 0x5e78, 0x820c, 0x2e9c,
 0x2256, 0x5232, 0x6a7a, 0x0a0e, 0x3a9e,
 0x2e56, 0x5e32, 0x767a, 0x160e, 0x469e,
 0x3a56, 0x6a32, 0x827a, 0x220e, 0x529e,
 0x4656, 0x7632, 0x0a7a, 0x2e0e, 0x5e9e,
 0x5256, 0x8232, 0x167a, 0x3a0e, 0x6a9e,
 0x5e56, 0x0a32, 0x227a, 0x460e, 0x769e,
 0x6a56, 0x1632, 0x2e7a, 0x520e, 0x829e,
 0x7656, 0x2232, 0x3a7a, 0x5e0e, 0x0a9e,
 0x8256, 0x2e32, 0x467a, 0x6a0e, 0x169e,
 0x0a56, 0x3a32, 0x527a, 0x760e, 0x229e,
 0x1656, 0x4632, 0x5e7a, 0x820e, 0x2e9e,
 0x2258, 0x5234, 0x6a7c, 0x0a10, 0x3aa0,
 0x2e58, 0x5e34, 0x767c, 0x1610, 0x46a0,
 0x3a58, 0x6a34, 0x827c, 0x2210, 0x52a0,
 0x4658, 0x7634, 0x0a7c, 0x2e10, 0x5ea0,
 0x5258, 0x8234, 0x167c, 0x3a10, 0x6aa0,
 0x5e58, 0x0a34, 0x227c, 0x4610, 0x76a0,
 0x6a58, 0x1634, 0x2e7c, 0x5210, 0x82a0,
 0x7658, 0x2234, 0x3a7c, 0x5e10, 0x0aa0,
 0x8258, 0x2e34, 0x467c, 0x6a10, 0x16a0,
 0x0a58, 0x3a34, 0x527c, 0x7610, 0x22a0,
 0x1658, 0x4634, 0x5e7c, 0x8210, 0x2ea0,
 0x0000, 0x0036, 0x006c, 0x00a2, 0x8648,
 0x0002, 0x0038, 0x006e, 0x00a4, 0x864c,
 0x0004, 0x003a, 0x0070, 0x00a6, 0x8650,
 0x0006, 0x003c, 0x0072, 0x00a8, 0x8654,
 0x0008, 0x003e, 0x0074, 0x00aa, 0x8658,
 0x000a, 0x0040, 0x0076, 0x00ac, 0x865c,
 0x000c, 0x0042, 0x0078, 0x00ae, 0x8660,
 0x000e, 0x0044, 0x007a, 0x00b0, 0x8664,
 0x0010, 0x0046, 0x007c, 0x00b2, 0x8668,
 0x0012, 0x0048, 0x007e, 0x8600, 0x866c,
 0x0014, 0x004a, 0x0080, 0x8604, 0x8670,
 0x0016, 0x004c, 0x0082, 0x8608, 0x8674,
 0x0018, 0x004e, 0x0084, 0x860c, 0x8678,
 0x001a, 0x0050, 0x0086, 0x8610, 0x867c,
 0x001c, 0x0052, 0x0088, 0x8614, 0x8680,
 0x001e, 0x0054, 0x008a, 0x8618, 0x8684,
 0x0020, 0x0056, 0x008c, 0x861c, 0x8688,
 0x0022, 0x0058, 0x008e, 0x8620, 0x868c,
 0x0024, 0x005a, 0x0090, 0x8624, 0x8690,
 0x0026, 0x005c, 0x0092, 0x8628, 0x8694,
 0x0028, 0x005e, 0x0094, 0x862c, 0x8698,
 0x002a, 0x0060, 0x0096, 0x8630, 0x869c,
 0x002c, 0x0062, 0x0098, 0x8634, 0x86a0,
 0x002e, 0x0064, 0x009a, 0x8638, 0x86a4,
 0x0030, 0x0066, 0x009c, 0x863c, 0x86a8,
 0x0032, 0x0068, 0x009e, 0x8640, 0x86ac,
 0x0034, 0x006a, 0x00a0, 0x8644, 0x86b0,
 0x4a5a, 0x7a36, 0x0e7e, 0x3212, 0x62a2,
 0x565a, 0x0236, 0x1a7e, 0x3e12, 0x6ea2,
 0x625a, 0x0e36, 0x267e, 0x4a12, 0x7aa2,
 0x6e5a, 0x1a36, 0x327e, 0x5612, 0x02a2,
 0x7a5a, 0x2636, 0x3e7e, 0x6212, 0x0ea2,
 0x025a, 0x3236, 0x4a7e, 0x6e12, 0x1aa2,
 0x0e5a, 0x3e36, 0x567e, 0x7a12, 0x26a2,
 0x1a5a, 0x4a36, 0x627e, 0x0212, 0x32a2,
 0x265a, 0x5636, 0x6e7e, 0x0e12, 0x3ea2,
 0x325a, 0x6236, 0x7a7e, 0x1a12, 0x4aa2,
 0x3e5a, 0x6e36, 0x027e, 0x2612, 0x56a2,
 0x4a5c, 0x7a38, 0x0e80, 0x3214, 0x62a4,
 0x565c, 0x0238, 0x1a80, 0x3e14, 0x6ea4,
 0x625c, 0x0e38, 0x2680, 0x4a14, 0x7aa4,
 0x6e5c, 0x1a38, 0x3280, 0x5614, 0x02a4,
 0x7a5c, 0x2638, 0x3e80, 0x6214, 0x0ea4,
 0x025c, 0x3238, 0x4a80, 0x6e14, 0x1aa4,
 0x0e5c, 0x3e38, 0x5680, 0x7a14, 0x26a4,
 0x1a5c, 0x4a38, 0x6280, 0x0214, 0x32a4,
 0x265c, 0x5638, 0x6e80, 0x0e14, 0x3ea4,
 0x325c, 0x6238, 0x7a80, 0x1a14, 0x4aa4,
 0x3e5c, 0x6e38, 0x0280, 0x2614, 0x56a4,
 0x4a5e, 0x7a3a, 0x0e82, 0x3216, 0x62a6,
 0x565e, 0x023a, 0x1a82, 0x3e16, 0x6ea6,
 0x625e, 0x0e3a, 0x2682, 0x4a16, 0x7aa6,
 0x6e5e, 0x1a3a, 0x3282, 0x5616, 0x02a6,
 0x7a5e, 0x263a, 0x3e82, 0x6216, 0x0ea6,
 0x025e, 0x323a, 0x4a82, 0x6e16, 0x1aa6,
 0x0e5e, 0x3e3a, 0x5682, 0x7a16, 0x26a6,
 0x1a5e, 0x4a3a, 0x6282, 0x0216, 0x32a6,
 0x265e, 0x563a, 0x6e82, 0x0e16, 0x3ea6,
 0x325e, 0x623a, 0x7a82, 0x1a16, 0x4aa6,
 0x3e5e, 0x6e3a, 0x0282, 0x2616, 0x56a6,
 0x4a60, 0x7a3c, 0x0e84, 0x3218, 0x62a8,
 0x5660, 0x023c, 0x1a84, 0x3e18, 0x6ea8,
 0x6260, 0x0e3c, 0x2684, 0x4a18, 0x7aa8,
 0x6e60, 0x1a3c, 0x3284, 0x5618, 0x02a8,
 0x7a60, 0x263c, 0x3e84, 0x6218, 0x0ea8,
 0x0260, 0x323c, 0x4a84, 0x6e18, 0x1aa8,
 0x0e60, 0x3e3c, 0x5684, 0x7a18, 0x26a8,
 0x1a60, 0x4a3c, 0x6284, 0x0218, 0x32a8,
 0x2660, 0x563c, 0x6e84, 0x0e18, 0x3ea8,
 0x3260, 0x623c, 0x7a84, 0x1a18, 0x4aa8,
 0x3e60, 0x6e3c, 0x0284, 0x2618, 0x56a8,
 0x4a62, 0x7a3e, 0x0e86, 0x321a, 0x62aa,
 0x5662, 0x023e, 0x1a86, 0x3e1a, 0x6eaa,
 0x6262, 0x0e3e, 0x2686, 0x4a1a, 0x7aaa,
 0x6e62, 0x1a3e, 0x3286, 0x561a, 0x02aa,
 0x7a62, 0x263e, 0x3e86, 0x621a, 0x0eaa,
 0x0262, 0x323e, 0x4a86, 0x6e1a, 0x1aaa,
 0x0e62, 0x3e3e, 0x5686, 0x7a1a, 0x26aa,
 0x1a62, 0x4a3e, 0x6286, 0x021a, 0x32aa,
 0x2662, 0x563e, 0x6e86, 0x0e1a, 0x3eaa,
 0x3262, 0x623e, 0x7a86, 0x1a1a, 0x4aaa,
 0x3e62, 0x6e3e, 0x0286, 0x261a, 0x56aa,
 0x4a64, 0x7a40, 0x0e88, 0x321c, 0x62ac,
 0x5664, 0x0240, 0x1a88, 0x3e1c, 0x6eac,
 0x6264, 0x0e40, 0x2688, 0x4a1c, 0x7aac,
 0x6e64, 0x1a40, 0x3288, 0x561c, 0x02ac,
 0x7a64, 0x2640, 0x3e88, 0x621c, 0x0eac,
 0x0264, 0x3240, 0x4a88, 0x6e1c, 0x1aac,
 0x0e64, 0x3e40, 0x5688, 0x7a1c, 0x26ac,
 0x1a64, 0x4a40, 0x6288, 0x021c, 0x32ac,
 0x2664, 0x5640, 0x6e88, 0x0e1c, 0x3eac,
 0x3264, 0x6240, 0x7a88, 0x1a1c, 0x4aac,
 0x3e64, 0x6e40, 0x0288, 0x261c, 0x56ac,
 0x4a66, 0x7a42, 0x0e8a, 0x321e, 0x62ae,
 0x5666, 0x0242, 0x1a8a, 0x3e1e, 0x6eae,
 0x6266, 0x0e42, 0x268a, 0x4a1e, 0x7aae,
 0x6e66, 0x1a42, 0x328a, 0x561e, 0x02ae,
 0x7a66, 0x2642, 0x3e8a, 0x621e, 0x0eae,
 0x0266, 0x3242, 0x4a8a, 0x6e1e, 0x1aae,
 0x0e66, 0x3e42, 0x568a, 0x7a1e, 0x26ae,
 0x1a66, 0x4a42, 0x628a, 0x021e, 0x32ae,
 0x2666, 0x5642, 0x6e8a, 0x0e1e, 0x3eae,
 0x3266, 0x6242, 0x7a8a, 0x1a1e, 0x4aae,
 0x3e66, 0x6e42, 0x028a, 0x261e, 0x56ae,
 0x4a68, 0x7a44, 0x0e8c, 0x3220, 0x62b0,
 0x5668, 0x0244, 0x1a8c, 0x3e20, 0x6eb0,
 0x6268, 0x0e44, 0x268c, 0x4a20, 0x7ab0,
 0x6e68, 0x1a44, 0x328c, 0x5620, 0x02b0,
 0x7a68, 0x2644, 0x3e8c, 0x6220, 0x0eb0,
 0x0268, 0x3244, 0x4a8c, 0x6e20, 0x1ab0,
 0x0e68, 0x3e44, 0x568c, 0x7a20, 0x26b0,
 0x1a68, 0x4a44, 0x628c, 0x0220, 0x32b0,
 0x2668, 0x5644, 0x6e8c, 0x0e20, 0x3eb0,
 0x3268, 0x6244, 0x7a8c, 0x1a20, 0x4ab0,
 0x3e68, 0x6e44, 0x028c, 0x2620, 0x56b0,
 0x4a6a, 0x7a46, 0x0e8e, 0x3222, 0x62b2,
 0x566a, 0x0246, 0x1a8e, 0x3e22, 0x6eb2,
 0x626a, 0x0e46, 0x268e, 0x4a22, 0x7ab2,
 0x6e6a, 0x1a46, 0x328e, 0x5622, 0x02b2,
 0x7a6a, 0x2646, 0x3e8e, 0x6222, 0x0eb2,
 0x026a, 0x3246, 0x4a8e, 0x6e22, 0x1ab2,
 0x0e6a, 0x3e46, 0x568e, 0x7a22, 0x26b2,
 0x1a6a, 0x4a46, 0x628e, 0x0222, 0x32b2,
 0x266a, 0x5646, 0x6e8e, 0x0e22, 0x3eb2,
 0x326a, 0x6246, 0x7a8e, 0x1a22, 0x4ab2,
 0x3e6a, 0x6e46, 0x028e, 0x2622, 0x56b2,
 0x4e5a, 0x7e36, 0x127e, 0x3612, 0x66a2,
 0x5a5a, 0x0636, 0x1e7e, 0x4212, 0x72a2,
 0x665a, 0x1236, 0x2a7e, 0x4e12, 0x7ea2,
 0x725a, 0x1e36, 0x367e, 0x5a12, 0x06a2,
 0x7e5a, 0x2a36, 0x427e, 0x6612, 0x12a2,
 0x065a, 0x3636, 0x4e7e, 0x7212, 0x1ea2,
 0x125a, 0x4236, 0x5a7e, 0x7e12, 0x2aa2,
 0x1e5a, 0x4e36, 0x667e, 0x0612, 0x36a2,
 0x2a5a, 0x5a36, 0x727e, 0x1212, 0x42a2,
 0x365a, 0x6636, 0x7e7e, 0x1e12, 0x4ea2,
 0x425a, 0x7236, 0x067e, 0x2a12, 0x5aa2,
 0x4e5c, 0x7e38, 0x1280, 0x3614, 0x66a4,
 0x5a5c, 0x0638, 0x1e80, 0x4214, 0x72a4,
 0x665c, 0x1238, 0x2a80, 0x4e14, 0x7ea4,
 0x725c, 0x1e38, 0x3680, 0x5a14, 0x06a4,
 0x7e5c, 0x2a38, 0x4280, 0x6614, 0x12a4,
 0x065c, 0x3638, 0x4e80, 0x7214, 0x1ea4,
 0x125c, 0x4238, 0x5a80, 0x7e14, 0x2aa4,
 0x1e5c, 0x4e38, 0x6680, 0x0614, 0x36a4,
 0x2a5c, 0x5a38, 0x7280, 0x1214, 0x42a4,
 0x365c, 0x6638, 0x7e80, 0x1e14, 0x4ea4,
 0x425c, 0x7238, 0x0680, 0x2a14, 0x5aa4,
 0x4e5e, 0x7e3a, 0x1282, 0x3616, 0x66a6,
 0x5a5e, 0x063a, 0x1e82, 0x4216, 0x72a6,
 0x665e, 0x123a, 0x2a82, 0x4e16, 0x7ea6,
 0x725e, 0x1e3a, 0x3682, 0x5a16, 0x06a6,
 0x7e5e, 0x2a3a, 0x4282, 0x6616, 0x12a6,
 0x065e, 0x363a, 0x4e82, 0x7216, 0x1ea6,
 0x125e, 0x423a, 0x5a82, 0x7e16, 0x2aa6,
 0x1e5e, 0x4e3a, 0x6682, 0x0616, 0x36a6,
 0x2a5e, 0x5a3a, 0x7282, 0x1216, 0x42a6,
 0x365e, 0x663a, 0x7e82, 0x1e16, 0x4ea6,
 0x425e, 0x723a, 0x0682, 0x2a16, 0x5aa6,
 0x4e60, 0x7e3c, 0x1284, 0x3618, 0x66a8,
 0x5a60, 0x063c, 0x1e84, 0x4218, 0x72a8,
 0x6660, 0x123c, 0x2a84, 0x4e18, 0x7ea8,
 0x7260, 0x1e3c, 0x3684, 0x5a18, 0x06a8,
 0x7e60, 0x2a3c, 0x4284, 0x6618, 0x12a8,
 0x0660, 0x363c, 0x4e84, 0x7218, 0x1ea8,
 0x1260, 0x423c, 0x5a84, 0x7e18, 0x2aa8,
 0x1e60, 0x4e3c, 0x6684, 0x0618, 0x36a8,
 0x2a60, 0x5a3c, 0x7284, 0x1218, 0x42a8,
 0x3660, 0x663c, 0x7e84, 0x1e18, 0x4ea8,
 0x4260, 0x723c, 0x0684, 0x2a18, 0x5aa8,
 0x4e62, 0x7e3e, 0x1286, 0x361a, 0x66aa,
 0x5a62, 0x063e, 0x1e86, 0x421a, 0x72aa,
 0x6662, 0x123e, 0x2a86, 0x4e1a, 0x7eaa,
 0x7262, 0x1e3e, 0x3686, 0x5a1a, 0x06aa,
 0x7e62, 0x2a3e, 0x4286, 0x661a, 0x12aa,
 0x0662, 0x363e, 0x4e86, 0x721a, 0x1eaa,
 0x1262, 0x423e, 0x5a86, 0x7e1a, 0x2aaa,
 0x1e62, 0x4e3e, 0x6686, 0x061a, 0x36aa,
 0x2a62, 0x5a3e, 0x7286, 0x121a, 0x42aa,
 0x3662, 0x663e, 0x7e86, 0x1e1a, 0x4eaa,
 0x4262, 0x723e, 0x0686, 0x2a1a, 0x5aaa,
 0x4e64, 0x7e40, 0x1288, 0x361c, 0x66ac,
 0x5a64, 0x0640, 0x1e88, 0x421c, 0x72ac,
 0x6664, 0x1240, 0x2a88, 0x4e1c, 0x7eac,
 0x7264, 0x1e40, 0x3688, 0x5a1c, 0x06ac,
 0x7e64, 0x2a40, 0x4288, 0x661c, 0x12ac,
 0x0664, 0x3640, 0x4e88, 0x721c, 0x1eac,
 0x1264, 0x4240, 0x5a88, 0x7e1c, 0x2aac,
 0x1e64, 0x4e40, 0x6688, 0x061c, 0x36ac,
 0x2a64, 0x5a40, 0x7288, 0x121c, 0x42ac,
 0x3664, 0x6640, 0x7e88, 0x1e1c, 0x4eac,
 0x4264, 0x7240, 0x0688, 0x2a1c, 0x5aac,
 0x4e66, 0x7e42, 0x128a, 0x361e, 0x66ae,
 0x5a66, 0x0642, 0x1e8a, 0x421e, 0x72ae,
 0x6666, 0x1242, 0x2a8a, 0x4e1e, 0x7eae,
 0x7266, 0x1e42, 0x368a, 0x5a1e, 0x06ae,
 0x7e66, 0x2a42, 0x428a, 0x661e, 0x12ae,
 0x0666, 0x3642, 0x4e8a, 0x721e, 0x1eae,
 0x1266, 0x4242, 0x5a8a, 0x7e1e, 0x2aae,
 0x1e66, 0x4e42, 0x668a, 0x061e, 0x36ae,
 0x2a66, 0x5a42, 0x728a, 0x121e, 0x42ae,
 0x3666, 0x6642, 0x7e8a, 0x1e1e, 0x4eae,
 0x4266, 0x7242, 0x068a, 0x2a1e, 0x5aae,
 0x4e68, 0x7e44, 0x128c, 0x3620, 0x66b0,
 0x5a68, 0x0644, 0x1e8c, 0x4220, 0x72b0,
 0x6668, 0x1244, 0x2a8c, 0x4e20, 0x7eb0,
 0x7268, 0x1e44, 0x368c, 0x5a20, 0x06b0,
 0x7e68, 0x2a44, 0x428c, 0x6620, 0x12b0,
 0x0668, 0x3644, 0x4e8c, 0x7220, 0x1eb0,
 0x1268, 0x4244, 0x5a8c, 0x7e20, 0x2ab0,
 0x1e68, 0x4e44, 0x668c, 0x0620, 0x36b0,
 0x2a68, 0x5a44, 0x728c, 0x1220, 0x42b0,
 0x3668, 0x6644, 0x7e8c, 0x1e20, 0x4eb0,
 0x4268, 0x7244, 0x068c, 0x2a20, 0x5ab0,
 0x4e6a, 0x7e46, 0x128e, 0x3622, 0x66b2,
 0x5a6a, 0x0646, 0x1e8e, 0x4222, 0x72b2,
 0x666a, 0x1246, 0x2a8e, 0x4e22, 0x7eb2,
 0x726a, 0x1e46, 0x368e, 0x5a22, 0x06b2,
 0x7e6a, 0x2a46, 0x428e, 0x6622, 0x12b2,
 0x066a, 0x3646, 0x4e8e, 0x7222, 0x1eb2,
 0x126a, 0x4246, 0x5a8e, 0x7e22, 0x2ab2,
 0x1e6a, 0x4e46, 0x668e, 0x0622, 0x36b2,
 0x2a6a, 0x5a46, 0x728e, 0x1222, 0x42b2,
 0x366a, 0x6646, 0x7e8e, 0x1e22, 0x4eb2,
 0x426a, 0x7246, 0x068e, 0x2a22, 0x5ab2,
 0x525a, 0x8236, 0x167e, 0x3a12, 0x6aa2,
 0x5e5a, 0x0a36, 0x227e, 0x4612, 0x76a2,
 0x6a5a, 0x1636, 0x2e7e, 0x5212, 0x82a2,
 0x765a, 0x2236, 0x3a7e, 0x5e12, 0x0aa2,
 0x825a, 0x2e36, 0x467e, 0x6a12, 0x16a2,
 0x0a5a, 0x3a36, 0x527e, 0x7612, 0x22a2,
 0x165a, 0x4636, 0x5e7e, 0x8212, 0x2ea2,
 0x225a, 0x5236, 0x6a7e, 0x0a12, 0x3aa2,
 0x2e5a, 0x5e36, 0x767e, 0x1612, 0x46a2,
 0x3a5a, 0x6a36, 0x827e, 0x2212, 0x52a2,
 0x465a, 0x7636, 0x0a7e, 0x2e12, 0x5ea2,
 0x525c, 0x8238, 0x1680, 0x3a14, 0x6aa4,
 0x5e5c, 0x0a38, 0x2280, 0x4614, 0x76a4,
 0x6a5c, 0x1638, 0x2e80, 0x5214, 0x82a4,
 0x765c, 0x2238, 0x3a80, 0x5e14, 0x0aa4,
 0x825c, 0x2e38, 0x4680, 0x6a14, 0x16a4,
 0x0a5c, 0x3a38, 0x5280, 0x7614, 0x22a4,
 0x165c, 0x4638, 0x5e80, 0x8214, 0x2ea4,
 0x225c, 0x5238, 0x6a80, 0x0a14, 0x3aa4,
 0x2e5c, 0x5e38, 0x7680, 0x1614, 0x46a4,
 0x3a5c, 0x6a38, 0x8280, 0x2214, 0x52a4,
 0x465c, 0x7638, 0x0a80, 0x2e14, 0x5ea4,
 0x525e, 0x823a, 0x1682, 0x3a16, 0x6aa6,
 0x5e5e, 0x0a3a, 0x2282, 0x4616, 0x76a6,
 0x6a5e, 0x163a, 0x2e82, 0x5216, 0x82a6,
 0x765e, 0x223a, 0x3a82, 0x5e16, 0x0aa6,
 0x825e, 0x2e3a, 0x4682, 0x6a16, 0x16a6,
 0x0a5e, 0x3a3a, 0x5282, 0x7616, 0x22a6,
 0x165e, 0x463a, 0x5e82, 0x8216, 0x2ea6,
 0x225e, 0x523a, 0x6a82, 0x0a16, 0x3aa6,
 0x2e5e, 0x5e3a, 0x7682, 0x1616, 0x46a6,
 0x3a5e, 0x6a3a, 0x8282, 0x2216, 0x52a6,
 0x465e, 0x763a, 0x0a82, 0x2e16, 0x5ea6,
 0x5260, 0x823c, 0x1684, 0x3a18, 0x6aa8,
 0x5e60, 0x0a3c, 0x2284, 0x4618, 0x76a8,
 0x6a60, 0x163c, 0x2e84, 0x5218, 0x82a8,
 0x7660, 0x223c, 0x3a84, 0x5e18, 0x0aa8,
 0x8260, 0x2e3c, 0x4684, 0x6a18, 0x16a8,
 0x0a60, 0x3a3c, 0x5284, 0x7618, 0x22a8,
 0x1660, 0x463c, 0x5e84, 0x8218, 0x2ea8,
 0x2260, 0x523c, 0x6a84, 0x0a18, 0x3aa8,
 0x2e60, 0x5e3c, 0x7684, 0x1618, 0x46a8,
 0x3a60, 0x6a3c, 0x8284, 0x2218, 0x52a8,
 0x4660, 0x763c, 0x0a84, 0x2e18, 0x5ea8,
 0x5262, 0x823e, 0x1686, 0x3a1a, 0x6aaa,
 0x5e62, 0x0a3e, 0x2286, 0x461a, 0x76aa,
 0x6a62, 0x163e, 0x2e86, 0x521a, 0x82aa,
 0x7662, 0x223e, 0x3a86, 0x5e1a, 0x0aaa,
 0x8262, 0x2e3e, 0x4686, 0x6a1a, 0x16aa,
 0x0a62, 0x3a3e, 0x5286, 0x761a, 0x22aa,
 0x1662, 0x463e, 0x5e86, 0x821a, 0x2eaa,
 0x2262, 0x523e, 0x6a86, 0x0a1a, 0x3aaa,
 0x2e62, 0x5e3e, 0x7686, 0x161a, 0x46aa,
 0x3a62, 0x6a3e, 0x8286, 0x221a, 0x52aa,
 0x4662, 0x763e, 0x0a86, 0x2e1a, 0x5eaa,
 0x5264, 0x8240, 0x1688, 0x3a1c, 0x6aac,
 0x5e64, 0x0a40, 0x2288, 0x461c, 0x76ac,
 0x6a64, 0x1640, 0x2e88, 0x521c, 0x82ac,
 0x7664, 0x2240, 0x3a88, 0x5e1c, 0x0aac,
 0x8264, 0x2e40, 0x4688, 0x6a1c, 0x16ac,
 0x0a64, 0x3a40, 0x5288, 0x761c, 0x22ac,
 0x1664, 0x4640, 0x5e88, 0x821c, 0x2eac,
 0x2264, 0x5240, 0x6a88, 0x0a1c, 0x3aac,
 0x2e64, 0x5e40, 0x7688, 0x161c, 0x46ac,
 0x3a64, 0x6a40, 0x8288, 0x221c, 0x52ac,
 0x4664, 0x7640, 0x0a88, 0x2e1c, 0x5eac,
 0x5266, 0x8242, 0x168a, 0x3a1e, 0x6aae,
 0x5e66, 0x0a42, 0x228a, 0x461e, 0x76ae,
 0x6a66, 0x1642, 0x2e8a, 0x521e, 0x82ae,
 0x7666, 0x2242, 0x3a8a, 0x5e1e, 0x0aae,
 0x8266, 0x2e42, 0x468a, 0x6a1e, 0x16ae,
 0x0a66, 0x3a42, 0x528a, 0x761e, 0x22ae,
 0x1666, 0x4642, 0x5e8a, 0x821e, 0x2eae,
 0x2266, 0x5242, 0x6a8a, 0x0a1e, 0x3aae,
 0x2e66, 0x5e42, 0x768a, 0x161e, 0x46ae,
 0x3a66, 0x6a42, 0x828a, 0x221e, 0x52ae,
 0x4666, 0x7642, 0x0a8a, 0x2e1e, 0x5eae,
 0x5268, 0x8244, 0x168c, 0x3a20, 0x6ab0,
 0x5e68, 0x0a44, 0x228c, 0x4620, 0x76b0,
 0x6a68, 0x1644, 0x2e8c, 0x5220, 0x82b0,
 0x7668, 0x2244, 0x3a8c, 0x5e20, 0x0ab0,
 0x8268, 0x2e44, 0x468c, 0x6a20, 0x16b0,
 0x0a68, 0x3a44, 0x528c, 0x7620, 0x22b0,
 0x1668, 0x4644, 0x5e8c, 0x8220, 0x2eb0,
 0x2268, 0x5244, 0x6a8c, 0x0a20, 0x3ab0,
 0x2e68, 0x5e44, 0x768c, 0x1620, 0x46b0,
 0x3a68, 0x6a44, 0x828c, 0x2220, 0x52b0,
 0x4668, 0x7644, 0x0a8c, 0x2e20, 0x5eb0,
 0x526a, 0x8246, 0x168e, 0x3a22, 0x6ab2,
 0x5e6a, 0x0a46, 0x228e, 0x4622, 0x76b2,
 0x6a6a, 0x1646, 0x2e8e, 0x5222, 0x82b2,
 0x766a, 0x2246, 0x3a8e, 0x5e22, 0x0ab2,
 0x826a, 0x2e46, 0x468e, 0x6a22, 0x16b2,
 0x0a6a, 0x3a46, 0x528e, 0x7622, 0x22b2,
 0x166a, 0x4646, 0x5e8e, 0x8222, 0x2eb2,
 0x226a, 0x5246, 0x6a8e, 0x0a22, 0x3ab2,
 0x2e6a, 0x5e46, 0x768e, 0x1622, 0x46b2,
 0x3a6a, 0x6a46, 0x828e, 0x2222, 0x52b2,
 0x466a, 0x7646, 0x0a8e, 0x2e22, 0x5eb2,
 0x4a5a, 0x7a36, 0x0e7e, 0x3212, 0x62a2,
 0x565a, 0x0236, 0x1a7e, 0x3e12, 0x6ea2,
 0x625a, 0x0e36, 0x267e, 0x4a12, 0x7aa2,
 0x6e5a, 0x1a36, 0x327e, 0x5612, 0x02a2,
 0x7a5a, 0x2636, 0x3e7e, 0x6212, 0x0ea2,
 0x025a, 0x3236, 0x4a7e, 0x6e12, 0x1aa2,
 0x0e5a, 0x3e36, 0x567e, 0x7a12, 0x26a2,
 0x1a5a, 0x4a36, 0x627e, 0x0212, 0x32a2,
 0x265a, 0x5636, 0x6e7e, 0x0e12, 0x3ea2,
 0x325a, 0x6236, 0x7a7e, 0x1a12, 0x4aa2,
 0x3e5a, 0x6e36, 0x027e, 0x2612, 0x56a2,
 0x4a5c, 0x7a38, 0x0e80, 0x3214, 0x62a4,
 0x565c, 0x0238, 0x1a80, 0x3e14, 0x6ea4,
 0x625c, 0x0e38, 0x2680, 0x4a14, 0x7aa4,
 0x6e5c, 0x1a38, 0x3280, 0x5614, 0x02a4,
 0x7a5c, 0x2638, 0x3e80, 0x6214, 0x0ea4,
 0x025c, 0x3238, 0x4a80, 0x6e14, 0x1aa4,
 0x0e5c, 0x3e38, 0x5680, 0x7a14, 0x26a4,
 0x1a5c, 0x4a38, 0x6280, 0x0214, 0x32a4,
 0x265c, 0x5638, 0x6e80, 0x0e14, 0x3ea4,
 0x325c, 0x6238, 0x7a80, 0x1a14, 0x4aa4,
 0x3e5c, 0x6e38, 0x0280, 0x2614, 0x56a4,
 0x4a5e, 0x7a3a, 0x0e82, 0x3216, 0x62a6,
 0x565e, 0x023a, 0x1a82, 0x3e16, 0x6ea6,
 0x625e, 0x0e3a, 0x2682, 0x4a16, 0x7aa6,
 0x6e5e, 0x1a3a, 0x3282, 0x5616, 0x02a6,
 0x7a5e, 0x263a, 0x3e82, 0x6216, 0x0ea6,
 0x7c48, 0x2824, 0x406c, 0x6400, 0x1090,
 0x0448, 0x3424, 0x4c6c, 0x7000, 0x1c90,
 0x1048, 0x4024, 0x586c, 0x7c00, 0x2890,
 0x1c48, 0x4c24, 0x646c, 0x0400, 0x3490,
 0x2848, 0x5824, 0x706c, 0x1000, 0x4090,
 0x3448, 0x6424, 0x7c6c, 0x1c00, 0x4c90,
 0x4048, 0x7024, 0x046c, 0x2800, 0x5890,
 0x4c48, 0x7c24, 0x106c, 0x3400, 0x6490,
 0x5848, 0x0424, 0x1c6c, 0x4000, 0x7090,
 0x6448, 0x1024, 0x286c, 0x4c00, 0x7c90,
 0x7048, 0x1c24, 0x346c, 0x5800, 0x0490,
 0x7c4a, 0x2826, 0x406e, 0x6402, 0x1092,
 0x044a, 0x3426, 0x4c6e, 0x7002, 0x1c92,
 0x104a, 0x4026, 0x586e, 0x7c02, 0x2892,
 0x1c4a, 0x4c26, 0x646e, 0x0402, 0x3492,
 0x284a, 0x5826, 0x706e, 0x1002, 0x4092,
 0x344a, 0x6426, 0x7c6e, 0x1c02, 0x4c92,
 0x404a, 0x7026, 0x046e, 0x2802, 0x5892,
 0x4c4a, 0x7c26, 0x106e, 0x3402, 0x6492,
 0x584a, 0x0426, 0x1c6e, 0x4002, 0x7092,
 0x644a, 0x1026, 0x286e, 0x4c02, 0x7c92,
 0x704a, 0x1c26, 0x346e, 0x5802, 0x0492,
 0x7c4c, 0x2828, 0x4070, 0x6404, 0x1094,
 0x044c, 0x3428, 0x4c70, 0x7004, 0x1c94,
 0x104c, 0x4028, 0x5870, 0x7c04, 0x2894,
 0x1c4c, 0x4c28, 0x6470, 0x0404, 0x3494,
 0x284c, 0x5828, 0x7070, 0x1004, 0x4094,
 0x344c, 0x6428, 0x7c70, 0x1c04, 0x4c94,
 0x404c, 0x7028, 0x0470, 0x2804, 0x5894,
 0x4c4c, 0x7c28, 0x1070, 0x3404, 0x6494,
 0x584c, 0x0428, 0x1c70, 0x4004, 0x7094,
 0x644c, 0x1028, 0x2870, 0x4c04, 0x7c94,
 0x704c, 0x1c28, 0x3470, 0x5804, 0x0494,
 0x7c4e, 0x282a, 0x4072, 0x6406, 0x1096,
 0x044e, 0x342a, 0x4c72, 0x7006, 0x1c96,
 0x104e, 0x402a, 0x5872, 0x7c06, 0x2896,
 0x1c4e, 0x4c2a, 0x6472, 0x0406, 0x3496,
 0x284e, 0x582a, 0x7072, 0x1006, 0x4096,
 0x344e, 0x642a, 0x7c72, 0x1c06, 0x4c96,
 0x404e, 0x702a, 0x0472, 0x2806, 0x5896,
 0x4c4e, 0x7c2a, 0x1072, 0x3406, 0x6496,
 0x584e, 0x042a, 0x1c72, 0x4006, 0x7096,
 0x644e, 0x102a, 0x2872, 0x4c06, 0x7c96,
 0x704e, 0x1c2a, 0x3472, 0x5806, 0x0496,
 0x7c50, 0x282c, 0x4074, 0x6408, 0x1098,
 0x0450, 0x342c, 0x4c74, 0x7008, 0x1c98,
 0x1050, 0x402c, 0x5874, 0x7c08, 0x2898,
 0x1c50, 0x4c2c, 0x6474, 0x0408, 0x3498,
 0x2850, 0x582c, 0x7074, 0x1008, 0x4098,
 0x3450, 0x642c, 0x7c74, 0x1c08, 0x4c98,
 0x4050, 0x702c, 0x0474, 0x2808, 0x5898,
 0x4c50, 0x7c2c, 0x1074, 0x3408, 0x6498,
 0x5850, 0x042c, 0x1c74, 0x4008, 0x7098,
 0x6450, 0x102c, 0x2874, 0x4c08, 0x7c98,
 0x7050, 0x1c2c, 0x3474, 0x5808, 0x0498,
 0x7c52, 0x282e, 0x4076, 0x640a, 0x109a,
 0x0452, 0x342e, 0x4c76, 0x700a, 0x1c9a,
 0x1052, 0x402e, 0x5876, 0x7c0a, 0x289a,
 0x1c52, 0x4c2e, 0x6476, 0x040a, 0x349a,
 0x2852, 0x582e, 0x7076, 0x100a, 0x409a,
 0x3452, 0x642e, 0x7c76, 0x1c0a, 0x4c9a,
 0x4052, 0x702e, 0x0476, 0x280a, 0x589a,
 0x4c52, 0x7c2e, 0x1076, 0x340a, 0x649a,
 0x5852, 0x042e, 0x1c76, 0x400a, 0x709a,
 0x6452, 0x102e, 0x2876, 0x4c0a, 0x7c9a,
 0x7052, 0x1c2e, 0x3476, 0x580a, 0x049a,
 0x7c54, 0x2830, 0x4078, 0x640c, 0x109c,
 0x0454, 0x3430, 0x4c78, 0x700c, 0x1c9c,
 0x1054, 0x4030, 0x5878, 0x7c0c, 0x289c,
 0x1c54, 0x4c30, 0x6478, 0x040c, 0x349c,
 0x2854, 0x5830, 0x7078, 0x100c, 0x409c,
 0x3454, 0x6430, 0x7c78, 0x1c0c, 0x4c9c,
 0x4054, 0x7030, 0x0478, 0x280c, 0x589c,
 0x4c54, 0x7c30, 0x1078, 0x340c, 0x649c,
 0x5854, 0x0430, 0x1c78, 0x400c, 0x709c,
 0x6454, 0x1030, 0x2878, 0x4c0c, 0x7c9c,
 0x7054, 0x1c30, 0x3478, 0x580c, 0x049c,
 0x7c56, 0x2832, 0x407a, 0x640e, 0x109e,
 0x0456, 0x3432, 0x4c7a, 0x700e, 0x1c9e,
 0x1056, 0x4032, 0x587a, 0x7c0e, 0x289e,
 0x1c56, 0x4c32, 0x647a, 0x040e, 0x349e,
 0x2856, 0x5832, 0x707a, 0x100e, 0x409e,
 0x3456, 0x6432, 0x7c7a, 0x1c0e, 0x4c9e,
 0x4056, 0x7032, 0x047a, 0x280e, 0x589e,
 0x4c56, 0x7c32, 0x107a, 0x340e, 0x649e,
 0x5856, 0x0432, 0x1c7a, 0x400e, 0x709e,
 0x6456, 0x1032, 0x287a, 0x4c0e, 0x7c9e,
 0x7056, 0x1c32, 0x347a, 0x580e, 0x049e,
 0x7c58, 0x2834, 0x407c, 0x6410, 0x10a0,
 0x0458, 0x3434, 0x4c7c, 0x7010, 0x1ca0,
 0x1058, 0x4034, 0x587c, 0x7c10, 0x28a0,
 0x1c58, 0x4c34, 0x647c, 0x0410, 0x34a0,
 0x2858, 0x5834, 0x707c, 0x1010, 0x40a0,
 0x3458, 0x6434, 0x7c7c, 0x1c10, 0x4ca0,
 0x4058, 0x7034, 0x047c, 0x2810, 0x58a0,
 0x4c58, 0x7c34, 0x107c, 0x3410, 0x64a0,
 0x5858, 0x0434, 0x1c7c, 0x4010, 0x70a0,
 0x6458, 0x1034, 0x287c, 0x4c10, 0x7ca0,
 0x7058, 0x1c34, 0x347c, 0x5810, 0x04a0,
 0x8048, 0x2c24, 0x446c, 0x6800, 0x1490,
 0x0848, 0x3824, 0x506c, 0x7400, 0x2090,
 0x1448, 0x4424, 0x5c6c, 0x8000, 0x2c90,
 0x2048, 0x5024, 0x686c, 0x0800, 0x3890,
 0x2c48, 0x5c24, 0x746c, 0x1400, 0x4490,
 0x3848, 0x6824, 0x806c, 0x2000, 0x5090,
 0x4448, 0x7424, 0x086c, 0x2c00, 0x5c90,
 0x5048, 0x8024, 0x146c, 0x3800, 0x6890,
 0x5c48, 0x0824, 0x206c, 0x4400, 0x7490,
 0x6848, 0x1424, 0x2c6c, 0x5000, 0x8090,
 0x7448, 0x2024, 0x386c, 0x5c00, 0x0890,
 0x804a, 0x2c26, 0x446e, 0x6802, 0x1492,
 0x084a, 0x3826, 0x506e, 0x7402, 0x2092,
 0x144a, 0x4426, 0x5c6e, 0x8002, 0x2c92,
 0x204a, 0x5026, 0x686e, 0x0802, 0x3892,
 0x2c4a, 0x5c26, 0x746e, 0x1402, 0x4492,
 0x384a, 0x6826, 0x806e, 0x2002, 0x5092,
 0x444a, 0x7426, 0x086e, 0x2c02, 0x5c92,
 0x504a, 0x8026, 0x146e, 0x3802, 0x6892,
 0x5c4a, 0x0826, 0x206e, 0x4402, 0x7492,
 0x684a, 0x1426, 0x2c6e, 0x5002, 0x8092,
 0x744a, 0x2026, 0x386e, 0x5c02, 0x0892,
 0x804c, 0x2c28, 0x4470, 0x6804, 0x1494,
 0x084c, 0x3828, 0x5070, 0x7404, 0x2094,
 0x144c, 0x4428, 0x5c70, 0x8004, 0x2c94,
 0x204c, 0x5028, 0x6870, 0x0804, 0x3894,
 0x2c4c, 0x5c28, 0x7470, 0x1404, 0x4494,
 0x384c, 0x6828, 0x8070, 0x2004, 0x5094,
 0x444c, 0x7428, 0x0870, 0x2c04, 0x5c94,
 0x504c, 0x8028, 0x1470, 0x3804, 0x6894,
 0x5c4c, 0x0828, 0x2070, 0x4404, 0x7494,
 0x684c, 0x1428, 0x2c70, 0x5004, 0x8094,
 0x744c, 0x2028, 0x3870, 0x5c04, 0x0894,
 0x804e, 0x2c2a, 0x4472, 0x6806, 0x1496,
 0x084e, 0x382a, 0x5072, 0x7406, 0x2096,
 0x144e, 0x442a, 0x5c72, 0x8006, 0x2c96,
 0x204e, 0x502a, 0x6872, 0x0806, 0x3896,
 0x2c4e, 0x5c2a, 0x7472, 0x1406, 0x4496,
 0x384e, 0x682a, 0x8072, 0x2006, 0x5096,
 0x444e, 0x742a, 0x0872, 0x2c06, 0x5c96,
 0x504e, 0x802a, 0x1472, 0x3806, 0x6896,
 0x5c4e, 0x082a, 0x2072, 0x4406, 0x7496,
 0x684e, 0x142a, 0x2c72, 0x5006, 0x8096,
 0x744e, 0x202a, 0x3872, 0x5c06, 0x0896,
 0x8050, 0x2c2c, 0x4474, 0x6808, 0x1498,
 0x0850, 0x382c, 0x5074, 0x7408, 0x2098,
 0x1450, 0x442c, 0x5c74, 0x8008, 0x2c98,
 0x2050, 0x502c, 0x6874, 0x0808, 0x3898,
 0x2c50, 0x5c2c, 0x7474, 0x1408, 0x4498,
 0x3850, 0x682c, 0x8074, 0x2008, 0x5098,
 0x4450, 0x742c, 0x0874, 0x2c08, 0x5c98,
 0x5050, 0x802c, 0x1474, 0x3808, 0x6898,
 0x5c50, 0x082c, 0x2074, 0x4408, 0x7498,
 0x6850, 0x142c, 0x2c74, 0x5008, 0x8098,
 0x7450, 0x202c, 0x3874, 0x5c08, 0x0898,
 0x8052, 0x2c2e, 0x4476, 0x680a, 0x149a,
 0x0852, 0x382e, 0x5076, 0x740a, 0x209a,
 0x1452, 0x442e, 0x5c76, 0x800a, 0x2c9a,
 0x2052, 0x502e, 0x6876, 0x080a, 0x389a,
 0x2c52, 0x5c2e, 0x7476, 0x140a, 0x449a,
 0x3852, 0x682e, 0x8076, 0x200a, 0x509a,
 0x4452, 0x742e, 0x0876, 0x2c0a, 0x5c9a,
 0x5052, 0x802e, 0x1476, 0x380a, 0x689a,
 0x5c52, 0x082e, 0x2076, 0x440a, 0x749a,
 0x6852, 0x142e, 0x2c76, 0x500a, 0x809a,
 0x7452, 0x202e, 0x3876, 0x5c0a, 0x089a,
 0x8054, 0x2c30, 0x4478, 0x680c, 0x149c,
 0x0854, 0x3830, 0x5078, 0x740c, 0x209c,
 0x1454, 0x4430, 0x5c78, 0x800c, 0x2c9c,
 0x2054, 0x5030, 0x6878, 0x080c, 0x389c,
 0x2c54, 0x5c30, 0x7478, 0x140c, 0x449c,
 0x3854, 0x6830, 0x8078, 0x200c, 0x509c,
 0x4454, 0x7430, 0x0878, 0x2c0c, 0x5c9c,
 0x5054, 0x8030, 0x1478, 0x380c, 0x689c,
 0x5c54, 0x0830, 0x2078, 0x440c, 0x749c,
 0x6854, 0x1430, 0x2c78, 0x500c, 0x809c,
 0x7454, 0x2030, 0x3878, 0x5c0c, 0x089c,
 0x8056, 0x2c32, 0x447a, 0x680e, 0x149e,
 0x0856, 0x3832, 0x507a, 0x740e, 0x209e,
 0x1456, 0x4432, 0x5c7a, 0x800e, 0x2c9e,
 0x2056, 0x5032, 0x687a, 0x080e, 0x389e,
 0x2c56, 0x5c32, 0x747a, 0x140e, 0x449e,
 0x3856, 0x6832, 0x807a, 0x200e, 0x509e,
 0x4456, 0x7432, 0x087a, 0x2c0e, 0x5c9e,
 0x5056, 0x8032, 0x147a, 0x380e, 0x689e,
 0x5c56, 0x0832, 0x207a, 0x440e, 0x749e,
 0x6856, 0x1432, 0x2c7a, 0x500e, 0x809e,
 0x7456, 0x2032, 0x387a, 0x5c0e, 0x089e,
 0x8058, 0x2c34, 0x447c, 0x6810, 0x14a0,
 0x0858, 0x3834, 0x507c, 0x7410, 0x20a0,
 0x1458, 0x4434, 0x5c7c, 0x8010, 0x2ca0,
 0x2058, 0x5034, 0x687c, 0x0810, 0x38a0,
 0x2c58, 0x5c34, 0x747c, 0x1410, 0x44a0,
 0x3858, 0x6834, 0x807c, 0x2010, 0x50a0,
 0x4458, 0x7434, 0x087c, 0x2c10, 0x5ca0,
 0x5058, 0x8034, 0x147c, 0x3810, 0x68a0,
 0x5c58, 0x0834, 0x207c, 0x4410, 0x74a0,
 0x6858, 0x1434, 0x2c7c, 0x5010, 0x80a0,
 0x7458, 0x2034, 0x387c, 0x5c10, 0x08a0,
 0x8448, 0x3024, 0x486c, 0x6c00, 0x1890,
 0x0c48, 0x3c24, 0x546c, 0x7800, 0x2490,
 0x1848, 0x4824, 0x606c, 0x8400, 0x3090,
 0x2448, 0x5424, 0x6c6c, 0x0c00, 0x3c90,
 0x3048, 0x6024, 0x786c, 0x1800, 0x4890,
 0x3c48, 0x6c24, 0x846c, 0x2400, 0x5490,
 0x4848, 0x7824, 0x0c6c, 0x3000, 0x6090,
 0x5448, 0x8424, 0x186c, 0x3c00, 0x6c90,
 0x6048, 0x0c24, 0x246c, 0x4800, 0x7890,
 0x6c48, 0x1824, 0x306c, 0x5400, 0x8490,
 0x7848, 0x2424, 0x3c6c, 0x6000, 0x0c90,
 0x844a, 0x3026, 0x486e, 0x6c02, 0x1892,
 0x0c4a, 0x3c26, 0x546e, 0x7802, 0x2492,
 0x184a, 0x4826, 0x606e, 0x8402, 0x3092,
 0x244a, 0x5426, 0x6c6e, 0x0c02, 0x3c92,
 0x304a, 0x6026, 0x786e, 0x1802, 0x4892,
 0x3c4a, 0x6c26, 0x846e, 0x2402, 0x5492,
 0x484a, 0x7826, 0x0c6e, 0x3002, 0x6092,
 0x544a, 0x8426, 0x186e, 0x3c02, 0x6c92,
 0x604a, 0x0c26, 0x246e, 0x4802, 0x7892,
 0x6c4a, 0x1826, 0x306e, 0x5402, 0x8492,
 0x784a, 0x2426, 0x3c6e, 0x6002, 0x0c92,
 0x844c, 0x3028, 0x4870, 0x6c04, 0x1894,
 0x0c4c, 0x3c28, 0x5470, 0x7804, 0x2494,
 0x184c, 0x4828, 0x6070, 0x8404, 0x3094,
 0x244c, 0x5428, 0x6c70, 0x0c04, 0x3c94,
 0x304c, 0x6028, 0x7870, 0x1804, 0x4894,
 0x3c4c, 0x6c28, 0x8470, 0x2404, 0x5494,
 0x484c, 0x7828, 0x0c70, 0x3004, 0x6094,
 0x544c, 0x8428, 0x1870, 0x3c04, 0x6c94,
 0x604c, 0x0c28, 0x2470, 0x4804, 0x7894,
 0x6c4c, 0x1828, 0x3070, 0x5404, 0x8494,
 0x784c, 0x2428, 0x3c70, 0x6004, 0x0c94,
 0x844e, 0x302a, 0x4872, 0x6c06, 0x1896,
 0x0c4e, 0x3c2a, 0x5472, 0x7806, 0x2496,
 0x184e, 0x482a, 0x6072, 0x8406, 0x3096,
 0x244e, 0x542a, 0x6c72, 0x0c06, 0x3c96,
 0x304e, 0x602a, 0x7872, 0x1806, 0x4896,
 0x3c4e, 0x6c2a, 0x8472, 0x2406, 0x5496,
 0x484e, 0x782a, 0x0c72, 0x3006, 0x6096,
 0x544e, 0x842a, 0x1872, 0x3c06, 0x6c96,
 0x604e, 0x0c2a, 0x2472, 0x4806, 0x7896,
 0x6c4e, 0x182a, 0x3072, 0x5406, 0x8496,
 0x784e, 0x242a, 0x3c72, 0x6006, 0x0c96,
 0x8450, 0x302c, 0x4874, 0x6c08, 0x1898,
 0x0c50, 0x3c2c, 0x5474, 0x7808, 0x2498,
 0x1850, 0x482c, 0x6074, 0x8408, 0x3098,
 0x2450, 0x542c, 0x6c74, 0x0c08, 0x3c98,
 0x3050, 0x602c, 0x7874, 0x1808, 0x4898,
 0x3c50, 0x6c2c, 0x8474, 0x2408, 0x5498,
 0x4850, 0x782c, 0x0c74, 0x3008, 0x6098,
 0x5450, 0x842c, 0x1874, 0x3c08, 0x6c98,
 0x6050, 0x0c2c, 0x2474, 0x4808, 0x7898,
 0x6c50, 0x182c, 0x3074, 0x5408, 0x8498,
 0x7850, 0x242c, 0x3c74, 0x6008, 0x0c98,
 0x8452, 0x302e, 0x4876, 0x6c0a, 0x189a,
 0x0c52, 0x3c2e, 0x5476, 0x780a, 0x249a,
 0x1852, 0x482e, 0x6076, 0x840a, 0x309a,
 0x2452, 0x542e, 0x6c76, 0x0c0a, 0x3c9a,
 0x3052, 0x602e, 0x7876, 0x180a, 0x489a,
 0x3c52, 0x6c2e, 0x8476, 0x240a, 0x549a,
 0x4852, 0x782e, 0x0c76, 0x300a, 0x609a,
 0x5452, 0x842e, 0x1876, 0x3c0a, 0x6c9a,
 0x6052, 0x0c2e, 0x2476, 0x480a, 0x789a,
 0x6c52, 0x182e, 0x3076, 0x540a, 0x849a,
 0x7852, 0x242e, 0x3c76, 0x600a, 0x0c9a,
 0x8454, 0x3030, 0x4878, 0x6c0c, 0x189c,
 0x0c54, 0x3c30, 0x5478, 0x780c, 0x249c,
 0x1854, 0x4830, 0x6078, 0x840c, 0x309c,
 0x2454, 0x5430, 0x6c78, 0x0c0c, 0x3c9c,
 0x3054, 0x6030, 0x7878, 0x180c, 0x489c,
 0x3c54, 0x6c30, 0x8478, 0x240c, 0x549c,
 0x4854, 0x7830, 0x0c78, 0x300c, 0x609c,
 0x5454, 0x8430, 0x1878, 0x3c0c, 0x6c9c,
 0x6054, 0x0c30, 0x2478, 0x480c, 0x789c,
 0x6c54, 0x1830, 0x3078, 0x540c, 0x849c,
 0x7854, 0x2430, 0x3c78, 0x600c, 0x0c9c,
 0x8456, 0x3032, 0x487a, 0x6c0e, 0x189e,
 0x0c56, 0x3c32, 0x547a, 0x780e, 0x249e,
 0x1856, 0x4832, 0x607a, 0x840e, 0x309e,
 0x2456, 0x5432, 0x6c7a, 0x0c0e, 0x3c9e,
 0x3056, 0x6032, 0x787a, 0x180e, 0x489e,
 0x3c56, 0x6c32, 0x847a, 0x240e, 0x549e,
 0x4856, 0x7832, 0x0c7a, 0x300e, 0x609e,
 0x5456, 0x8432, 0x187a, 0x3c0e, 0x6c9e,
 0x6056, 0x0c32, 0x247a, 0x480e, 0x789e,
 0x6c56, 0x1832, 0x307a, 0x540e, 0x849e,
 0x7856, 0x2432, 0x3c7a, 0x600e, 0x0c9e,
 0x8458, 0x3034, 0x487c, 0x6c10, 0x18a0,
 0x0c58, 0x3c34, 0x547c, 0x7810, 0x24a0,
 0x1858, 0x4834, 0x607c, 0x8410, 0x30a0,
 0x2458, 0x5434, 0x6c7c, 0x0c10, 0x3ca0,
 0x3058, 0x6034, 0x787c, 0x1810, 0x48a0,
 0x3c58, 0x6c34, 0x847c, 0x2410, 0x54a0,
 0x4858, 0x7834, 0x0c7c, 0x3010, 0x60a0,
 0x5458, 0x8434, 0x187c, 0x3c10, 0x6ca0,
 0x6058, 0x0c34, 0x247c, 0x4810, 0x78a0,
 0x6c58, 0x1834, 0x307c, 0x5410, 0x84a0,
 0x7858, 0x2434, 0x3c7c, 0x6010, 0x0ca0,
 0x7c48, 0x2824, 0x406c, 0x6400, 0x1090,
 0x0448, 0x3424, 0x4c6c, 0x7000, 0x1c90,
 0x1048, 0x4024, 0x586c, 0x7c00, 0x2890,
 0x1c48, 0x4c24, 0x646c, 0x0400, 0x3490,
 0x2848, 0x5824, 0x706c, 0x1000, 0x4090,
 0x3448, 0x6424, 0x7c6c, 0x1c00, 0x4c90,
 0x4048, 0x7024, 0x046c, 0x2800, 0x5890,
 0x4c48, 0x7c24, 0x106c, 0x3400, 0x6490,
 0x5848, 0x0424, 0x1c6c, 0x4000, 0x7090,
 0x6448, 0x1024, 0x286c, 0x4c00, 0x7c90,
 0x7048, 0x1c24, 0x346c, 0x5800, 0x0490,
 0x7c4a, 0x2826, 0x406e, 0x6402, 0x1092,
 0x044a, 0x3426, 0x4c6e, 0x7002, 0x1c92,
 0x104a, 0x4026, 0x586e, 0x7c02, 0x2892,
 0x1c4a, 0x4c26, 0x646e, 0x0402, 0x3492,
 0x284a, 0x5826, 0x706e, 0x1002, 0x4092,
 0x344a, 0x6426, 0x7c6e, 0x1c02, 0x4c92,
 0x404a, 0x7026, 0x046e, 0x2802, 0x5892,
 0x4c4a, 0x7c26, 0x106e, 0x3402, 0x6492,
 0x584a, 0x0426, 0x1c6e, 0x4002, 0x7092,
 0x644a, 0x1026, 0x286e, 0x4c02, 0x7c92,
 0x704a, 0x1c26, 0x346e, 0x5802, 0x0492,
 0x7c4c, 0x2828, 0x4070, 0x6404, 0x1094,
 0x044c, 0x3428, 0x4c70, 0x7004, 0x1c94,
 0x104c, 0x4028, 0x5870, 0x7c04, 0x2894,
 0x1c4c, 0x4c28, 0x6470, 0x0404, 0x3494,
 0x284c, 0x5828, 0x7070, 0x1004, 0x4094,
 0x285a, 0x5836, 0x707e, 0x1012, 0x40a2,
 0x345a, 0x6436, 0x7c7e, 0x1c12, 0x4ca2,
 0x405a, 0x7036, 0x047e, 0x2812, 0x58a2,
 0x4c5a, 0x7c36, 0x107e, 0x3412, 0x64a2,
 0x585a, 0x0436, 0x1c7e, 0x4012, 0x70a2,
 0x645a, 0x1036, 0x287e, 0x4c12, 0x7ca2,
 0x705a, 0x1c36, 0x347e, 0x5812, 0x04a2,
 0x7c5a, 0x2836, 0x407e, 0x6412, 0x10a2,
 0x045a, 0x3436, 0x4c7e, 0x7012, 0x1ca2,
 0x105a, 0x4036, 0x587e, 0x7c12, 0x28a2,
 0x1c5a, 0x4c36, 0x647e, 0x0412, 0x34a2,
 0x285c, 0x5838, 0x7080, 0x1014, 0x40a4,
 0x345c, 0x6438, 0x7c80, 0x1c14, 0x4ca4,
 0x405c, 0x7038, 0x0480, 0x2814, 0x58a4,
 0x4c5c, 0x7c38, 0x1080, 0x3414, 0x64a4,
 0x585c, 0x0438, 0x1c80, 0x4014, 0x70a4,
 0x645c, 0x1038, 0x2880, 0x4c14, 0x7ca4,
 0x705c, 0x1c38, 0x3480, 0x5814, 0x04a4,
 0x7c5c, 0x2838, 0x4080, 0x6414, 0x10a4,
 0x045c, 0x3438, 0x4c80, 0x7014, 0x1ca4,
 0x105c, 0x4038, 0x5880, 0x7c14, 0x28a4,
 0x1c5c, 0x4c38, 0x6480, 0x0414, 0x34a4,
 0x285e, 0x583a, 0x7082, 0x1016, 0x40a6,
 0x345e, 0x643a, 0x7c82, 0x1c16, 0x4ca6,
 0x405e, 0x703a, 0x0482, 0x2816, 0x58a6,
 0x4c5e, 0x7c3a, 0x1082, 0x3416, 0x64a6,
 0x585e, 0x043a, 0x1c82, 0x4016, 0x70a6,
 0x645e, 0x103a, 0x2882, 0x4c16, 0x7ca6,
 0x705e, 0x1c3a, 0x3482, 0x5816, 0x04a6,
 0x7c5e, 0x283a, 0x4082, 0x6416, 0x10a6,
 0x045e, 0x343a, 0x4c82, 0x7016, 0x1ca6,
 0x105e, 0x403a, 0x5882, 0x7c16, 0x28a6,
 0x1c5e, 0x4c3a, 0x6482, 0x0416, 0x34a6,
 0x2860, 0x583c, 0x7084, 0x1018, 0x40a8,
 0x3460, 0x643c, 0x7c84, 0x1c18, 0x4ca8,
 0x4060, 0x703c, 0x0484, 0x2818, 0x58a8,
 0x4c60, 0x7c3c, 0x1084, 0x3418, 0x64a8,
 0x5860, 0x043c, 0x1c84, 0x4018, 0x70a8,
 0x6460, 0x103c, 0x2884, 0x4c18, 0x7ca8,
 0x7060, 0x1c3c, 0x3484, 0x5818, 0x04a8,
 0x7c60, 0x283c, 0x4084, 0x6418, 0x10a8,
 0x0460, 0x343c, 0x4c84, 0x7018, 0x1ca8,
 0x1060, 0x403c, 0x5884, 0x7c18, 0x28a8,
 0x1c60, 0x4c3c, 0x6484, 0x0418, 0x34a8,
 0x2862, 0x583e, 0x7086, 0x101a, 0x40aa,
 0x3462, 0x643e, 0x7c86, 0x1c1a, 0x4caa,
 0x4062, 0x703e, 0x0486, 0x281a, 0x58aa,
 0x4c62, 0x7c3e, 0x1086, 0x341a, 0x64aa,
 0x5862, 0x043e, 0x1c86, 0x401a, 0x70aa,
 0x6462, 0x103e, 0x2886, 0x4c1a, 0x7caa,
 0x7062, 0x1c3e, 0x3486, 0x581a, 0x04aa,
 0x7c62, 0x283e, 0x4086, 0x641a, 0x10aa,
 0x0462, 0x343e, 0x4c86, 0x701a, 0x1caa,
 0x1062, 0x403e, 0x5886, 0x7c1a, 0x28aa,
 0x1c62, 0x4c3e, 0x6486, 0x041a, 0x34aa,
 0x2864, 0x5840, 0x7088, 0x101c, 0x40ac,
 0x3464, 0x6440, 0x7c88, 0x1c1c, 0x4cac,
 0x4064, 0x7040, 0x0488, 0x281c, 0x58ac,
 0x4c64, 0x7c40, 0x1088, 0x341c, 0x64ac,
 0x5864, 0x0440, 0x1c88, 0x401c, 0x70ac,
 0x6464, 0x1040, 0x2888, 0x4c1c, 0x7cac,
 0x7064, 0x1c40, 0x3488, 0x581c, 0x04ac,
 0x7c64, 0x2840, 0x4088, 0x641c, 0x10ac,
 0x0464, 0x3440, 0x4c88, 0x701c, 0x1cac,
 0x1064, 0x4040, 0x5888, 0x7c1c, 0x28ac,
 0x1c64, 0x4c40, 0x6488, 0x041c, 0x34ac,
 0x2866, 0x5842, 0x708a, 0x101e, 0x40ae,
 0x3466, 0x6442, 0x7c8a, 0x1c1e, 0x4cae,
 0x4066, 0x7042, 0x048a, 0x281e, 0x58ae,
 0x4c66, 0x7c42, 0x108a, 0x341e, 0x64ae,
 0x5866, 0x0442, 0x1c8a, 0x401e, 0x70ae,
 0x6466, 0x1042, 0x288a, 0x4c1e, 0x7cae,
 0x7066, 0x1c42, 0x348a, 0x581e, 0x04ae,
 0x7c66, 0x2842, 0x408a, 0x641e, 0x10ae,
 0x0466, 0x3442, 0x4c8a, 0x701e, 0x1cae,
 0x1066, 0x4042, 0x588a, 0x7c1e, 0x28ae,
 0x1c66, 0x4c42, 0x648a, 0x041e, 0x34ae,
 0x2868, 0x5844, 0x708c, 0x1020, 0x40b0,
 0x3468, 0x6444, 0x7c8c, 0x1c20, 0x4cb0,
 0x4068, 0x7044, 0x048c, 0x2820, 0x58b0,
 0x4c68, 0x7c44, 0x108c, 0x3420, 0x64b0,
 0x5868, 0x0444, 0x1c8c, 0x4020, 0x70b0,
 0x6468, 0x1044, 0x288c, 0x4c20, 0x7cb0,
 0x7068, 0x1c44, 0x348c, 0x5820, 0x04b0,
 0x7c68, 0x2844, 0x408c, 0x6420, 0x10b0,
 0x0468, 0x3444, 0x4c8c, 0x7020, 0x1cb0,
 0x1068, 0x4044, 0x588c, 0x7c20, 0x28b0,
 0x1c68, 0x4c44, 0x648c, 0x0420, 0x34b0,
 0x286a, 0x5846, 0x708e, 0x1022, 0x40b2,
 0x346a, 0x6446, 0x7c8e, 0x1c22, 0x4cb2,
 0x406a, 0x7046, 0x048e, 0x2822, 0x58b2,
 0x4c6a, 0x7c46, 0x108e, 0x3422, 0x64b2,
 0x586a, 0x0446, 0x1c8e, 0x4022, 0x70b2,
 0x646a, 0x1046, 0x288e, 0x4c22, 0x7cb2,
 0x706a, 0x1c46, 0x348e, 0x5822, 0x04b2,
 0x7c6a, 0x2846, 0x408e, 0x6422, 0x10b2,
 0x046a, 0x3446, 0x4c8e, 0x7022, 0x1cb2,
 0x106a, 0x4046, 0x588e, 0x7c22, 0x28b2,
 0x1c6a, 0x4c46, 0x648e, 0x0422, 0x34b2,
 0x2c5a, 0x5c36, 0x747e, 0x1412, 0x44a2,
 0x385a, 0x6836, 0x807e, 0x2012, 0x50a2,
 0x445a, 0x7436, 0x087e, 0x2c12, 0x5ca2,
 0x505a, 0x8036, 0x147e, 0x3812, 0x68a2,
 0x5c5a, 0x0836, 0x207e, 0x4412, 0x74a2,
 0x685a, 0x1436, 0x2c7e, 0x5012, 0x80a2,
 0x745a, 0x2036, 0x387e, 0x5c12, 0x08a2,
 0x805a, 0x2c36, 0x447e, 0x6812, 0x14a2,
 0x085a, 0x3836, 0x507e, 0x7412, 0x20a2,
 0x145a, 0x4436, 0x5c7e, 0x8012, 0x2ca2,
 0x205a, 0x5036, 0x687e, 0x0812, 0x38a2,
 0x2c5c, 0x5c38, 0x7480, 0x1414, 0x44a4,
 0x385c, 0x6838, 0x8080, 0x2014, 0x50a4,
 0x445c, 0x7438, 0x0880, 0x2c14, 0x5ca4,
 0x505c, 0x8038, 0x1480, 0x3814, 0x68a4,
 0x5c5c, 0x0838, 0x2080, 0x4414, 0x74a4,
 0x685c, 0x1438, 0x2c80, 0x5014, 0x80a4,
 0x745c, 0x2038, 0x3880, 0x5c14, 0x08a4,
 0x805c, 0x2c38, 0x4480, 0x6814, 0x14a4,
 0x085c, 0x3838, 0x5080, 0x7414, 0x20a4,
 0x145c, 0x4438, 0x5c80, 0x8014, 0x2ca4,
 0x205c, 0x5038, 0x6880, 0x0814, 0x38a4,
 0x2c5e, 0x5c3a, 0x7482, 0x1416, 0x44a6,
 0x385e, 0x683a, 0x8082, 0x2016, 0x50a6,
 0x445e, 0x743a, 0x0882, 0x2c16, 0x5ca6,
 0x505e, 0x803a, 0x1482, 0x3816, 0x68a6,
 0x5c5e, 0x083a, 0x2082, 0x4416, 0x74a6,
 0x685e, 0x143a, 0x2c82, 0x5016, 0x80a6,
 0x745e, 0x203a, 0x3882, 0x5c16, 0x08a6,
 0x805e, 0x2c3a, 0x4482, 0x6816, 0x14a6,
 0x085e, 0x383a, 0x5082, 0x7416, 0x20a6,
 0x145e, 0x443a, 0x5c82, 0x8016, 0x2ca6,
 0x205e, 0x503a, 0x6882, 0x0816, 0x38a6,
 0x2c60, 0x5c3c, 0x7484, 0x1418, 0x44a8,
 0x3860, 0x683c, 0x8084, 0x2018, 0x50a8,
 0x4460, 0x743c, 0x0884, 0x2c18, 0x5ca8,
 0x5060, 0x803c, 0x1484, 0x3818, 0x68a8,
 0x5c60, 0x083c, 0x2084, 0x4418, 0x74a8,
 0x6860, 0x143c, 0x2c84, 0x5018, 0x80a8,
 0x7460, 0x203c, 0x3884, 0x5c18, 0x08a8,
 0x8060, 0x2c3c, 0x4484, 0x6818, 0x14a8,
 0x0860, 0x383c, 0x5084, 0x7418, 0x20a8,
 0x1460, 0x443c, 0x5c84, 0x8018, 0x2ca8,
 0x2060, 0x503c, 0x6884, 0x0818, 0x38a8,
 0x2c62, 0x5c3e, 0x7486, 0x141a, 0x44aa,
 0x3862, 0x683e, 0x8086, 0x201a, 0x50aa,
 0x4462, 0x743e, 0x0886, 0x2c1a, 0x5caa,
 0x5062, 0x803e, 0x1486, 0x381a, 0x68aa,
 0x5c62, 0x083e, 0x2086, 0x441a, 0x74aa,
 0x6862, 0x143e, 0x2c86, 0x501a, 0x80aa,
 0x7462, 0x203e, 0x3886, 0x5c1a, 0x08aa,
 0x8062, 0x2c3e, 0x4486, 0x681a, 0x14aa,
 0x0862, 0x383e, 0x5086, 0x741a, 0x20aa,
 0x1462, 0x443e, 0x5c86, 0x801a, 0x2caa,
 0x2062, 0x503e, 0x6886, 0x081a, 0x38aa,
 0x2c64, 0x5c40, 0x7488, 0x141c, 0x44ac,
 0x3864, 0x6840, 0x8088, 0x201c, 0x50ac,
 0x4464, 0x7440, 0x0888, 0x2c1c, 0x5cac,
 0x5064, 0x8040, 0x1488, 0x381c, 0x68ac,
 0x5c64, 0x0840, 0x2088, 0x441c, 0x74ac,
 0x6864, 0x1440, 0x2c88, 0x501c, 0x80ac,
 0x7464, 0x2040, 0x3888, 0x5c1c, 0x08ac,
 0x8064, 0x2c40, 0x4488, 0x681c, 0x14ac,
 0x0864, 0x3840, 0x5088, 0x741c, 0x20ac,
 0x1464, 0x4440, 0x5c88, 0x801c, 0x2cac,
 0x2064, 0x5040, 0x6888, 0x081c, 0x38ac,
 0x2c66, 0x5c42, 0x748a, 0x141e, 0x44ae,
 0x3866, 0x6842, 0x808a, 0x201e, 0x50ae,
 0x4466, 0x7442, 0x088a, 0x2c1e, 0x5cae,
 0x5066, 0x8042, 0x148a, 0x381e, 0x68ae,
 0x5c66, 0x0842, 0x208a, 0x441e, 0x74ae,
 0x6866, 0x1442, 0x2c8a, 0x501e, 0x80ae,
 0x7466, 0x2042, 0x388a, 0x5c1e, 0x08ae,
 0x8066, 0x2c42, 0x448a, 0x681e, 0x14ae,
 0x0866, 0x3842, 0x508a, 0x741e, 0x20ae,
 0x1466, 0x4442, 0x5c8a, 0x801e, 0x2cae,
 0x2066, 0x5042, 0x688a, 0x081e, 0x38ae,
 0x2c68, 0x5c44, 0x748c, 0x1420, 0x44b0,
 0x3868, 0x6844, 0x808c, 0x2020, 0x50b0,
 0x4468, 0x7444, 0x088c, 0x2c20, 0x5cb0,
 0x5068, 0x8044, 0x148c, 0x3820, 0x68b0,
 0x5c68, 0x0844, 0x208c, 0x4420, 0x74b0,
 0x6868, 0x1444, 0x2c8c, 0x5020, 0x80b0,
 0x7468, 0x2044, 0x388c, 0x5c20, 0x08b0,
 0x8068, 0x2c44, 0x448c, 0x6820, 0x14b0,
 0x0868, 0x3844, 0x508c, 0x7420, 0x20b0,
 0x1468, 0x4444, 0x5c8c, 0x8020, 0x2cb0,
 0x2068, 0x5044, 0x688c, 0x0820, 0x38b0,
 0x2c6a, 0x5c46, 0x748e, 0x1422, 0x44b2,
 0x386a, 0x6846, 0x808e, 0x2022, 0x50b2,
 0x446a, 0x7446, 0x088e, 0x2c22, 0x5cb2,
 0x506a, 0x8046, 0x148e, 0x3822, 0x68b2,
 0x5c6a, 0x0846, 0x208e, 0x4422, 0x74b2,
 0x686a, 0x1446, 0x2c8e, 0x5022, 0x80b2,
 0x746a, 0x2046, 0x388e, 0x5c22, 0x08b2,
 0x806a, 0x2c46, 0x448e, 0x6822, 0x14b2,
 0x086a, 0x3846, 0x508e, 0x7422, 0x20b2,
 0x146a, 0x4446, 0x5c8e, 0x8022, 0x2cb2,
 0x206a, 0x5046, 0x688e, 0x0822, 0x38b2,
 0x305a, 0x6036, 0x787e, 0x1812, 0x48a2,
 0x3c5a, 0x6c36, 0x847e, 0x2412, 0x54a2,
 0x485a, 0x7836, 0x0c7e, 0x3012, 0x60a2,
 0x545a, 0x8436, 0x187e, 0x3c12, 0x6ca2,
 0x605a, 0x0c36, 0x247e, 0x4812, 0x78a2,
 0x6c5a, 0x1836, 0x307e, 0x5412, 0x84a2,
 0x785a, 0x2436, 0x3c7e, 0x6012, 0x0ca2,
 0x845a, 0x3036, 0x487e, 0x6c12, 0x18a2,
 0x0c5a, 0x3c36, 0x547e, 0x7812, 0x24a2,
 0x185a, 0x4836, 0x607e, 0x8412, 0x30a2,
 0x245a, 0x5436, 0x6c7e, 0x0c12, 0x3ca2,
 0x305c, 0x6038, 0x7880, 0x1814, 0x48a4,
 0x3c5c, 0x6c38, 0x8480, 0x2414, 0x54a4,
 0x485c, 0x7838, 0x0c80, 0x3014, 0x60a4,
 0x545c, 0x8438, 0x1880, 0x3c14, 0x6ca4,
 0x605c, 0x0c38, 0x2480, 0x4814, 0x78a4,
 0x6c5c, 0x1838, 0x3080, 0x5414, 0x84a4,
 0x785c, 0x2438, 0x3c80, 0x6014, 0x0ca4,
 0x845c, 0x3038, 0x4880, 0x6c14, 0x18a4,
 0x0c5c, 0x3c38, 0x5480, 0x7814, 0x24a4,
 0x185c, 0x4838, 0x6080, 0x8414, 0x30a4,
 0x245c, 0x5438, 0x6c80, 0x0c14, 0x3ca4,
 0x305e, 0x603a, 0x7882, 0x1816, 0x48a6,
 0x3c5e, 0x6c3a, 0x8482, 0x2416, 0x54a6,
 0x485e, 0x783a, 0x0c82, 0x3016, 0x60a6,
 0x545e, 0x843a, 0x1882, 0x3c16, 0x6ca6,
 0x605e, 0x0c3a, 0x2482, 0x4816, 0x78a6,
 0x6c5e, 0x183a, 0x3082, 0x5416, 0x84a6,
 0x785e, 0x243a, 0x3c82, 0x6016, 0x0ca6,
 0x845e, 0x303a, 0x4882, 0x6c16, 0x18a6,
 0x0c5e, 0x3c3a, 0x5482, 0x7816, 0x24a6,
 0x185e, 0x483a, 0x6082, 0x8416, 0x30a6,
 0x245e, 0x543a, 0x6c82, 0x0c16, 0x3ca6,
 0x3060, 0x603c, 0x7884, 0x1818, 0x48a8,
 0x3c60, 0x6c3c, 0x8484, 0x2418, 0x54a8,
 0x4860, 0x783c, 0x0c84, 0x3018, 0x60a8,
 0x5460, 0x843c, 0x1884, 0x3c18, 0x6ca8,
 0x6060, 0x0c3c, 0x2484, 0x4818, 0x78a8,
 0x6c60, 0x183c, 0x3084, 0x5418, 0x84a8,
 0x7860, 0x243c, 0x3c84, 0x6018, 0x0ca8,
 0x8460, 0x303c, 0x4884, 0x6c18, 0x18a8,
 0x0c60, 0x3c3c, 0x5484, 0x7818, 0x24a8,
 0x1860, 0x483c, 0x6084, 0x8418, 0x30a8,
 0x2460, 0x543c, 0x6c84, 0x0c18, 0x3ca8,
 0x3062, 0x603e, 0x7886, 0x181a, 0x48aa,
 0x3c62, 0x6c3e, 0x8486, 0x241a, 0x54aa,
 0x4862, 0x783e, 0x0c86, 0x301a, 0x60aa,
 0x5462, 0x843e, 0x1886, 0x3c1a, 0x6caa,
 0x6062, 0x0c3e, 0x2486, 0x481a, 0x78aa,
 0x6c62, 0x183e, 0x3086, 0x541a, 0x84aa,
 0x7862, 0x243e, 0x3c86, 0x601a, 0x0caa,
 0x8462, 0x303e, 0x4886, 0x6c1a, 0x18aa,
 0x0c62, 0x3c3e, 0x5486, 0x781a, 0x24aa,
 0x1862, 0x483e, 0x6086, 0x841a, 0x30aa,
 0x2462, 0x543e, 0x6c86, 0x0c1a, 0x3caa,
 0x3064, 0x6040, 0x7888, 0x181c, 0x48ac,
 0x3c64, 0x6c40, 0x8488, 0x241c, 0x54ac,
 0x4864, 0x7840, 0x0c88, 0x301c, 0x60ac,
 0x5464, 0x8440, 0x1888, 0x3c1c, 0x6cac,
 0x6064, 0x0c40, 0x2488, 0x481c, 0x78ac,
 0x6c64, 0x1840, 0x3088, 0x541c, 0x84ac,
 0x7864, 0x2440, 0x3c88, 0x601c, 0x0cac,
 0x8464, 0x3040, 0x4888, 0x6c1c, 0x18ac,
 0x0c64, 0x3c40, 0x5488, 0x781c, 0x24ac,
 0x1864, 0x4840, 0x6088, 0x841c, 0x30ac,
 0x2464, 0x5440, 0x6c88, 0x0c1c, 0x3cac,
 0x3066, 0x6042, 0x788a, 0x181e, 0x48ae,
 0x3c66, 0x6c42, 0x848a, 0x241e, 0x54ae,
 0x4866, 0x7842, 0x0c8a, 0x301e, 0x60ae,
 0x5466, 0x8442, 0x188a, 0x3c1e, 0x6cae,
 0x6066, 0x0c42, 0x248a, 0x481e, 0x78ae,
 0x6c66, 0x1842, 0x308a, 0x541e, 0x84ae,
 0x7866, 0x2442, 0x3c8a, 0x601e, 0x0cae,
 0x8466, 0x3042, 0x488a, 0x6c1e, 0x18ae,
 0x0c66, 0x3c42, 0x548a, 0x781e, 0x24ae,
 0x1866, 0x4842, 0x608a, 0x841e, 0x30ae,
 0x2466, 0x5442, 0x6c8a, 0x0c1e, 0x3cae,
 0x3068, 0x6044, 0x788c, 0x1820, 0x48b0,
 0x3c68, 0x6c44, 0x848c, 0x2420, 0x54b0,
 0x4868, 0x7844, 0x0c8c, 0x3020, 0x60b0,
 0x5468, 0x8444, 0x188c, 0x3c20, 0x6cb0,
 0x6068, 0x0c44, 0x248c, 0x4820, 0x78b0,
 0x6c68, 0x1844, 0x308c, 0x5420, 0x84b0,
 0x7868, 0x2444, 0x3c8c, 0x6020, 0x0cb0,
 0x8468, 0x3044, 0x488c, 0x6c20, 0x18b0,
 0x0c68, 0x3c44, 0x548c, 0x7820, 0x24b0,
 0x1868, 0x4844, 0x608c, 0x8420, 0x30b0,
 0x2468, 0x5444, 0x6c8c, 0x0c20, 0x3cb0,
 0x306a, 0x6046, 0x788e, 0x1822, 0x48b2,
 0x3c6a, 0x6c46, 0x848e, 0x2422, 0x54b2,
 0x486a, 0x7846, 0x0c8e, 0x3022, 0x60b2,
 0x546a, 0x8446, 0x188e, 0x3c22, 0x6cb2,
 0x606a, 0x0c46, 0x248e, 0x4822, 0x78b2,
 0x6c6a, 0x1846, 0x308e, 0x5422, 0x84b2,
 0x786a, 0x2446, 0x3c8e, 0x6022, 0x0cb2,
 0x846a, 0x3046, 0x488e, 0x6c22, 0x18b2,
 0x0c6a, 0x3c46, 0x548e, 0x7822, 0x24b2,
 0x186a, 0x4846, 0x608e, 0x8422, 0x30b2,
 0x246a, 0x5446, 0x6c8e, 0x0c22, 0x3cb2,
 0x285a, 0x5836, 0x707e, 0x1012, 0x40a2,
 0x345a, 0x6436, 0x7c7e, 0x1c12, 0x4ca2,
 0x405a, 0x7036, 0x047e, 0x2812, 0x58a2,
 0x4c5a, 0x7c36, 0x107e, 0x3412, 0x64a2,
 0x585a, 0x0436, 0x1c7e, 0x4012, 0x70a2,
 0x645a, 0x1036, 0x287e, 0x4c12, 0x7ca2,
 0x705a, 0x1c36, 0x347e, 0x5812, 0x04a2,
 0x7c5a, 0x2836, 0x407e, 0x6412, 0x10a2,
 0x045a, 0x3436, 0x4c7e, 0x7012, 0x1ca2,
 0x105a, 0x4036, 0x587e, 0x7c12, 0x28a2,
 0x1c5a, 0x4c36, 0x647e, 0x0412, 0x34a2,
 0x285c, 0x5838, 0x7080, 0x1014, 0x40a4,
 0x345c, 0x6438, 0x7c80, 0x1c14, 0x4ca4,
 0x405c, 0x7038, 0x0480, 0x2814, 0x58a4,
 0x4c5c, 0x7c38, 0x1080, 0x3414, 0x64a4,
 0x585c, 0x0438, 0x1c80, 0x4014, 0x70a4,
 0x645c, 0x1038, 0x2880, 0x4c14, 0x7ca4,
 0x705c, 0x1c38, 0x3480, 0x5814, 0x04a4,
 0x7c5c, 0x2838, 0x4080, 0x6414, 0x10a4,
 0x045c, 0x3438, 0x4c80, 0x7014, 0x1ca4,
 0x105c, 0x4038, 0x5880, 0x7c14, 0x28a4,
 0x1c5c, 0x4c38, 0x6480, 0x0414, 0x34a4,
 0x285e, 0x583a, 0x7082, 0x1016, 0x40a6,
 0x345e, 0x643a, 0x7c82, 0x1c16, 0x4ca6,
 0x405e, 0x703a, 0x0482, 0x2816, 0x58a6,
 0x4c5e, 0x7c3a, 0x1082, 0x3416, 0x64a6,
 0x585e, 0x043a, 0x1c82, 0x4016, 0x70a6,
};

static const uint16_t dv_place_720p60[2*10*27*5] = {
 0x1230, 0x3618, 0x4848, 0x0000, 0x2460,
 0x2430, 0x4818, 0x0048, 0x1200, 0x3660,
 0x3630, 0x0018, 0x1248, 0x2400, 0x4860,
 0x4830, 0x1218, 0x2448, 0x3600, 0x0060,
 0x0030, 0x2418, 0x3648, 0x4800, 0x1260,
 0x1232, 0x361a, 0x484a, 0x0002, 0x2462,
 0x2432, 0x481a, 0x004a, 0x1202, 0x3662,
 0x3632, 0x001a, 0x124a, 0x2402, 0x4862,
 0x4832, 0x121a, 0x244a, 0x3602, 0x0062,
 0x0032, 0x241a, 0x364a, 0x4802, 0x1262,
 0x1234, 0x361c, 0x484c, 0x0004, 0x2464,
 0x2434, 0x481c, 0x004c, 0x1204, 0x3664,
 0x3634, 0x001c, 0x124c, 0x2404, 0x4864,
 0x4834, 0x121c, 0x244c, 0x3604, 0x0064,
 0x0034, 0x241c, 0x364c, 0x4804, 0x1264,
 0x1236, 0x361e, 0x484e, 0x0006, 0x2466,
 0x2436, 0x481e, 0x004e, 0x1206, 0x3666,
 0x3636, 0x001e, 0x124e, 0x2406, 0x4866,
 0x4836, 0x121e, 0x244e, 0x3606, 0x0066,
 0x0036, 0x241e, 0x364e, 0x4806, 0x1266,
 0x1238, 0x3620, 0x4850, 0x0008, 0x2468,
 0x2438, 0x4820, 0x0050, 0x1208, 0x3668,
 0x3638, 0x0020, 0x1250, 0x2408, 0x4868,
 0x4838, 0x1220, 0x2450, 0x3608, 0x0068,
 0x0038, 0x2420, 0x3650, 0x4808, 0x1268,
 0x123a, 0x3622, 0x4852, 0x000a, 0x246a,
 0x243a, 0x4822, 0x0052, 0x120a, 0x366a,
 0x363a, 0x0022, 0x1252, 0x240a, 0x486a,
 0x483a, 0x1222, 0x2452, 0x360a, 0x006a,
 0x003a, 0x2422, 0x3652, 0x480a, 0x126a,
 0x1430, 0x3818, 0x4a48, 0x0200, 0x2660,
 0x2630, 0x4a18, 0x0248, 0x1400, 0x3860,
 0x3830, 0x0218, 0x1448, 0x2600, 0x4a60,
 0x4a30, 0x1418, 0x2648, 0x3800, 0x0260,
 0x0230, 0x2618, 0x3848, 0x4a00, 0x1460,
 0x1432, 0x381a, 0x4a4a, 0x0202, 0x2662,
 0x2632, 0x4a1a, 0x024a, 0x1402, 0x3862,
 0x3832, 0x021a, 0x144a, 0x2602, 0x4a62,
 0x4a32, 0x141a, 0x264a, 0x3802, 0x0262,
 0x0232, 0x261a, 0x384a, 0x4a02, 0x1462,
 0x1434, 0x381c, 0x4a4c, 0x0204, 0x2664,
 0x2634, 0x4a1c, 0x024c, 0x1404, 0x3864,
 0x3834, 0x021c, 0x144c, 0x2604, 0x4a64,
 0x4a34, 0x141c, 0x264c, 0x3804, 0x0264,
 0x0234, 0x261c, 0x384c, 0x4a04, 0x1464,
 0x1436, 0x381e, 0x4a4e, 0x0206, 0x2666,
 0x2636, 0x4a1e, 0x024e, 0x1406, 0x3866,
 0x3836, 0x021e, 0x144e, 0x2606, 0x4a66,
 0x4a36, 0x141e, 0x264e, 0x3806, 0x0266,
 0x0236, 0x261e, 0x384e, 0x4a06, 0x1466,
 0x1438, 0x3820, 0x4a50, 0x0208, 0x2668,
 0x2638, 0x4a20, 0x0250, 0x1408, 0x3868,
 0x3838, 0x0220, 0x1450, 0x2608, 0x4a68,
 0x4a38, 0x1420, 0x2650, 0x3808, 0x0268,
 0x0238, 0x2620, 0x3850, 0x4a08, 0x1468,
 0x143a, 0x3822, 0x4a52, 0x020a, 0x266a,
 0x263a, 0x4a22, 0x0252, 0x140a, 0x386a,
 0x383a, 0x0222, 0x1452, 0x260a, 0x4a6a,
 0x4a3a, 0x1422, 0x2652, 0x380a, 0x026a,
 0x023a, 0x2622, 0x3852, 0x4a0a, 0x146a,
 0x1630, 0x3a18, 0x4c48, 0x0400, 0x2860,
 0x2830, 0x4c18, 0x0448, 0x1600, 0x3a60,
 0x3a30, 0x0418, 0x1648, 0x2800, 0x4c60,
 0x4c30, 0x1618, 0x2848, 0x3a00, 0x0460,
 0x0430, 0x2818, 0x3a48, 0x4c00, 0x1660,
 0x1632, 0x3a1a, 0x4c4a, 0x0402, 0x2862,
 0x2832, 0x4c1a, 0x044a, 0x1602, 0x3a62,
 0x3a32, 0x041a, 0x164a, 0x2802, 0x4c62,
 0x4c32, 0x161a, 0x284a, 0x3a02, 0x0462,
 0x0432, 0x281a, 0x3a4a, 0x4c02, 0x1662,
 0x1634, 0x3a1c, 0x4c4c, 0x0404, 0x2864,
 0x2834, 0x4c1c, 0x044c, 0x1604, 0x3a64,
 0x3a34, 0x041c, 0x164c, 0x2804, 0x4c64,
 0x4c34, 0x161c, 0x284c, 0x3a04, 0x0464,
 0x0434, 0x281c, 0x3a4c, 0x4c04, 0x1664,
 0x1636, 0x3a1e, 0x4c4e, 0x0406, 0x2866,
 0x2836, 0x4c1e, 0x044e, 0x1606, 0x3a66,
 0x3a36, 0x041e, 0x164e, 0x2806, 0x4c66,
 0x4c36, 0x161e, 0x284e, 0x3a06, 0x0466,
 0x0436, 0x281e, 0x3a4e, 0x4c06, 0x1666,
 0x1638, 0x3a20, 0x4c50, 0x0408, 0x2868,
 0x2838, 0x4c20, 0x0450, 0x1608, 0x3a68,
 0x3a38, 0x0420, 0x1650, 0x2808, 0x4c68,
 0x4c38, 0x1620, 0x2850, 0x3a08, 0x0468,
 0x0438, 0x2820, 0x3a50, 0x4c08, 0x1668,
 0x163a, 0x3a22, 0x4c52, 0x040a, 0x286a,
 0x283a, 0x4c22, 0x0452, 0x160a, 0x3a6a,
 0x3a3a, 0x0422, 0x1652, 0x280a, 0x4c6a,
 0x4c3a, 0x1622, 0x2852, 0x3a0a, 0x046a,
 0x043a, 0x2822, 0x3a52, 0x4c0a, 0x166a,
 0x1830, 0x3c18, 0x4e48, 0x0600, 0x2a60,
 0x2a30, 0x4e18, 0x0648, 0x1800, 0x3c60,
 0x3c30, 0x0618, 0x1848, 0x2a00, 0x4e60,
 0x4e30, 0x1818, 0x2a48, 0x3c00, 0x0660,
 0x0630, 0x2a18, 0x3c48, 0x4e00, 0x1860,
 0x1832, 0x3c1a, 0x4e4a, 0x0602, 0x2a62,
 0x2a32, 0x4e1a, 0x064a, 0x1802, 0x3c62,
 0x3c32, 0x061a, 0x184a, 0x2a02, 0x4e62,
 0x4e32, 0x181a, 0x2a4a, 0x3c02, 0x0662,
 0x0632, 0x2a1a, 0x3c4a, 0x4e02, 0x1862,
 0x1834, 0x3c1c, 0x4e4c, 0x0604, 0x2a64,
 0x2a34, 0x4e1c, 0x064c, 0x1804, 0x3c64,
 0x3c34, 0x061c, 0x184c, 0x2a04, 0x4e64,
 0x4e34, 0x181c, 0x2a4c, 0x3c04, 0x0664,
 0x0634, 0x2a1c, 0x3c4c, 0x4e04, 0x1864,
 0x1836, 0x3c1e, 0x4e4e, 0x0606, 0x2a66,
 0x2a36, 0x4e1e, 0x064e, 0x1806, 0x3c66,
 0x3c36, 0x061e, 0x184e, 0x2a06, 0x4e66,
 0x4e36, 0x181e, 0x2a4e, 0x3c06, 0x0666,
 0x0636, 0x2a1e, 0x3c4e, 0x4e06, 0x1866,
 0x1838, 0x3c20, 0x4e50, 0x0608, 0x2a68,
 0x2a38, 0x4e20, 0x0650, 0x1808, 0x3c68,
 0x3c38, 0x0620, 0x1850, 0x2a08, 0x4e68,
 0x4e38, 0x1820, 0x2a50, 0x3c08, 0x0668,
 0x0638, 0x2a20, 0x3c50, 0x4e08, 0x1868,
 0x183a, 0x3c22, 0x4e52, 0x060a, 0x2a6a,
 0x2a3a, 0x4e22, 0x0652, 0x180a, 0x3c6a,
 0x3c3a, 0x0622, 0x1852, 0x2a0a, 0x4e6a,
 0x4e3a, 0x1822, 0x2a52, 0x3c0a, 0x066a,
 0x063a, 0x2a22, 0x3c52, 0x4e0a, 0x186a,
 0x1a30, 0x3e18, 0x5048, 0x0800, 0x2c60,
 0x2c30, 0x5018, 0x0848, 0x1a00, 0x3e60,
 0x3e30, 0x0818, 0x1a48, 0x2c00, 0x5060,
 0x5030, 0x1a18, 0x2c48, 0x3e00, 0x0860,
 0x0830, 0x2c18, 0x3e48, 0x5000, 0x1a60,
 0x1a32, 0x3e1a, 0x504a, 0x0802, 0x2c62,
 0x2c32, 0x501a, 0x084a, 0x1a02, 0x3e62,
 0x3e32, 0x081a, 0x1a4a, 0x2c02, 0x5062,
 0x5032, 0x1a1a, 0x2c4a, 0x3e02, 0x0862,
 0x0832, 0x2c1a, 0x3e4a, 0x5002, 0x1a62,
 0x1a34, 0x3e1c, 0x504c, 0x0804, 0x2c64,
 0x2c34, 0x501c, 0x084c, 0x1a04, 0x3e64,
 0x3e34, 0x081c, 0x1a4c, 0x2c04, 0x5064,
 0x5034, 0x1a1c, 0x2c4c, 0x3e04, 0x0864,
 0x0834, 0x2c1c, 0x3e4c, 0x5004, 0x1a64,
 0x1a36, 0x3e1e, 0x504e, 0x0806, 0x2c66,
 0x2c36, 0x501e, 0x084e, 0x1a06, 0x3e66,
 0x3e36, 0x081e, 0x1a4e, 0x2c06, 0x5066,
 0x5036, 0x1a1e, 0x2c4e, 0x3e06, 0x0866,
 0x0836, 0x2c1e, 0x3e4e, 0x5006, 0x1a66,
 0x1a38, 0x3e20, 0x5050, 0x0808, 0x2c68,
 0x2c38, 0x5020, 0x0850, 0x1a08, 0x3e68,
 0x3e38, 0x0820, 0x1a50, 0x2c08, 0x5068,
 0x5038, 0x1a20, 0x2c50, 0x3e08, 0x0868,
 0x0838, 0x2c20, 0x3e50, 0x5008, 0x1a68,
 0x1a3a, 0x3e22, 0x5052, 0x080a, 0x2c6a,
 0x2c3a, 0x5022, 0x0852, 0x1a0a, 0x3e6a,
 0x3e3a, 0x0822, 0x1a52, 0x2c0a, 0x506a,
 0x503a, 0x1a22, 0x2c52, 0x3e0a, 0x086a,
 0x083a, 0x2c22, 0x3e52, 0x500a, 0x1a6a,
 0x1c30, 0x4018, 0x5248, 0x0a00, 0x2e60,
 0x2e30, 0x5218, 0x0a48, 0x1c00, 0x4060,
 0x4030, 0x0a18, 0x1c48, 0x2e00, 0x5260,
 0x5230, 0x1c18, 0x2e48, 0x4000, 0x0a60,
 0x0a30, 0x2e18, 0x4048, 0x5200, 0x1c60,
 0x1c32, 0x401a, 0x524a, 0x0a02, 0x2e62,
 0x2e32, 0x521a, 0x0a4a, 0x1c02, 0x4062,
 0x4032, 0x0a1a, 0x1c4a, 0x2e02, 0x5262,
 0x5232, 0x1c1a, 0x2e4a, 0x4002, 0x0a62,
 0x0a32, 0x2e1a, 0x404a, 0x5202, 0x1c62,
 0x1c34, 0x401c, 0x524c, 0x0a04, 0x2e64,
 0x2e34, 0x521c, 0x0a4c, 0x1c04, 0x4064,
 0x4034, 0x0a1c, 0x1c4c, 0x2e04, 0x5264,
 0x5234, 0x1c1c, 0x2e4c, 0x4004, 0x0a64,
 0x0a34, 0x2e1c, 0x404c, 0x5204, 0x1c64,
 0x1c36, 0x401e, 0x524e, 0x0a06, 0x2e66,
 0x2e36, 0x521e, 0x0a4e, 0x1c06, 0x4066,
 0x4036, 0x0a1e, 0x1c4e, 0x2e06, 0x5266,
 0x5236, 0x1c1e, 0x2e4e, 0x4006, 0x0a66,
 0x0a36, 0x2e1e, 0x404e, 0x5206, 0x1c66,
 0x1c38, 0x4020, 0x5250, 0x0a08, 0x2e68,
 0x2e38, 0x5220, 0x0a50, 0x1c08, 0x4068,
 0x4038, 0x0a20, 0x1c50, 0x2e08, 0x5268,
 0x5238, 0x1c20, 0x2e50, 0x4008, 0x0a68,
 0x0a38, 0x2e20, 0x4050, 0x5208, 0x1c68,
 0x1c3a, 0x4022, 0x5252, 0x0a0a, 0x2e6a,
 0x2e3a, 0x5222, 0x0a52, 0x1c0a, 0x406a,
 0x403a, 0x0a22, 0x1c52, 0x2e0a, 0x526a,
 0x523a, 0x1c22, 0x2e52, 0x400a, 0x0a6a,
 0x0a3a, 0x2e22, 0x4052, 0x520a, 0x1c6a,
 0x1e30, 0x4218, 0x5448, 0x0c00, 0x3060,
 0x3030, 0x5418, 0x0c48, 0x1e00, 0x4260,
 0x4230, 0x0c18, 0x1e48, 0x3000, 0x5460,
 0x5430, 0x1e18, 0x3048, 0x4200, 0x0c60,
 0x0c30, 0x3018, 0x4248, 0x5400, 0x1e60,
 0x1e32, 0x421a, 0x544a, 0x0c02, 0x3062,
 0x3032, 0x541a, 0x0c4a, 0x1e02, 0x4262,
 0x4232, 0x0c1a, 0x1e4a, 0x3002, 0x5462,
 0x5432, 0x1e1a, 0x304a, 0x4202, 0x0c62,
 0x0c32, 0x301a, 0x424a, 0x5402, 0x1e62,
 0x1e34, 0x421c, 0x544c, 0x0c04, 0x3064,
 0x3034, 0x541c, 0x0c4c, 0x1e04, 0x4264,
 0x4234, 0x0c1c, 0x1e4c, 0x3004, 0x5464,
 0x5434, 0x1e1c, 0x304c, 0x4204, 0x0c64,
 0x0c34, 0x301c, 0x424c, 0x5404, 0x1e64,
 0x1e36, 0x421e, 0x544e, 0x0c06, 0x3066,
 0x3036, 0x541e, 0x0c4e, 0x1e06, 0x4266,
 0x4236, 0x0c1e, 0x1e4e, 0x3006, 0x5466,
 0x5436, 0x1e1e, 0x304e, 0x4206, 0x0c66,
 0x0c36, 0x301e, 0x424e, 0x5406, 0x1e66,
 0x1e38, 0x4220, 0x5450, 0x0c08, 0x3068,
 0x3038, 0x5420, 0x0c50, 0x1e08, 0x4268,
 0x4238, 0x0c20, 0x1e50, 0x3008, 0x5468,
 0x5438, 0x1e20, 0x3050, 0x4208, 0x0c68,
 0x0c38, 0x3020, 0x4250, 0x5408, 0x1e68,
 0x1e3a, 0x4222, 0x5452, 0x0c0a, 0x306a,
 0x303a, 0x5422, 0x0c52, 0x1e0a, 0x426a,
 0x423a, 0x0c22, 0x1e52, 0x300a, 0x546a,
 0x543a, 0x1e22, 0x3052, 0x420a, 0x0c6a,
 0x0c3a, 0x3022, 0x4252, 0x540a, 0x1e6a,
 0x2030, 0x4418, 0x5648, 0x0e00, 0x3260,
 0x3230, 0x5618, 0x0e48, 0x2000, 0x4460,
 0x4430, 0x0e18, 0x2048, 0x3200, 0x5660,
 0x5630, 0x2018, 0x3248, 0x4400, 0x0e60,
 0x0e30, 0x3218, 0x4448, 0x5600, 0x2060,
 0x2032, 0x441a, 0x564a, 0x0e02, 0x3262,
 0x3232, 0x561a, 0x0e4a, 0x2002, 0x4462,
 0x4432, 0x0e1a, 0x204a, 0x3202, 0x5662,
 0x5632, 0x201a, 0x324a, 0x4402, 0x0e62,
 0x0e32, 0x321a, 0x444a, 0x5602, 0x2062,
 0x2034, 0x441c, 0x564c, 0x0e04, 0x3264,
 0x3234, 0x561c, 0x0e4c, 0x2004, 0x4464,
 0x4434, 0x0e1c, 0x204c, 0x3204, 0x5664,
 0x5634, 0x201c, 0x324c, 0x4404, 0x0e64,
 0x0e34, 0x321c, 0x444c, 0x5604, 0x2064,
 0x2036, 0x441e, 0x564e, 0x0e06, 0x3266,
 0x3236, 0x561e, 0x0e4e, 0x2006, 0x4466,
 0x4436, 0x0e1e, 0x204e, 0x3206, 0x5666,
 0x5636, 0x201e, 0x324e, 0x4406, 0x0e66,
 0x0e36, 0x321e, 0x444e, 0x5606, 0x2066,
 0x2038, 0x4420, 0x5650, 0x0e08, 0x3268,
 0x3238, 0x5620, 0x0e50, 0x2008, 0x4468,
 0x4438, 0x0e20, 0x2050, 0x3208, 0x5668,
 0x5638, 0x2020, 0x3250, 0x4408, 0x0e68,
 0x0e38, 0x3220, 0x4450, 0x5608, 0x2068,
 0x203a, 0x4422, 0x5652, 0x0e0a, 0x326a,
 0x323a, 0x5622, 0x0e52, 0x200a, 0x446a,
 0x443a, 0x0e22, 0x2052, 0x320a, 0x566a,
 0x563a, 0x2022, 0x3252, 0x440a, 0x0e6a,
 0x0e3a, 0x3222, 0x4452, 0x560a, 0x206a,
 0x2230, 0x4618, 0x5848, 0x1000, 0x3460,
 0x3430, 0x5818, 0x1048, 0x2200, 0x4660,
 0x4630, 0x1018, 0x2248, 0x3400, 0x5860,
 0x5830, 0x2218, 0x3448, 0x4600, 0x1060,
 0x1030, 0x3418, 0x4648, 0x5800, 0x2260,
 0x2232, 0x461a, 0x584a, 0x1002, 0x3462,
 0x3432, 0x581a, 0x104a, 0x2202, 0x4662,
 0x4632, 0x101a, 0x224a, 0x3402, 0x5862,
 0x5832, 0x221a, 0x344a, 0x4602, 0x1062,
 0x1032, 0x341a, 0x464a, 0x5802, 0x2262,
 0x2234, 0x461c, 0x584c, 0x1004, 0x3464,
 0x3434, 0x581c, 0x104c, 0x2204, 0x4664,
 0x4634, 0x101c, 0x224c, 0x3404, 0x5864,
 0x5834, 0x221c, 0x344c, 0x4604, 0x1064,
 0x1034, 0x341c, 0x464c, 0x5804, 0x2264,
 0x2236, 0x461e, 0x584e, 0x1006, 0x3466,
 0x3436, 0x581e, 0x104e, 0x2206, 0x4666,
 0x4636, 0x101e, 0x224e, 0x3406, 0x5866,
 0x5836, 0x221e, 0x344e, 0x4606, 0x1066,
 0x1036, 0x341e, 0x464e, 0x5806, 0x2266,
 0x2238, 0x4620, 0x5850, 0x1008, 0x3468,
 0x3438, 0x5820, 0x1050, 0x2208, 0x4668,
 0x4638, 0x1020, 0x2250, 0x3408, 0x5868,
 0x5838, 0x2220, 0x3450, 0x4608, 0x1068,
 0x1038, 0x3420, 0x4650, 0x5808, 0x2268,
 0x223a, 0x4622, 0x5852, 0x100a, 0x346a,
 0x343a, 0x5822, 0x1052, 0x220a, 0x466a,
 0x463a, 0x1022, 0x2252, 0x340a, 0x586a,
 0x583a, 0x2222, 0x3452, 0x460a, 0x106a,
 0x103a, 0x3422, 0x4652, 0x580a, 0x226a,
 0x363c, 0x0024, 0x1254, 0x240c, 0x486c,
 0x483c, 0x1224, 0x2454, 0x360c, 0x006c,
 0x003c, 0x2424, 0x3654, 0x480c, 0x126c,
 0x123c, 0x3624, 0x4854, 0x000c, 0x246c,
 0x243c, 0x4824, 0x0054, 0x120c, 0x366c,
 0x363e, 0x0026, 0x1256, 0x240e, 0x486e,
 0x483e, 0x1226, 0x2456, 0x360e, 0x006e,
 0x003e, 0x2426, 0x3656, 0x480e, 0x126e,
 0x123e, 0x3626, 0x4856, 0x000e, 0x246e,
 0x243e, 0x4826, 0x0056, 0x120e, 0x366e,
 0x3640, 0x0028, 0x1258, 0x2410, 0x4870,
 0x4840, 0x1228, 0x2458, 0x3610, 0x0070,
 0x0040, 0x2428, 0x3658, 0x4810, 0x1270,
 0x1240, 0x3628, 0x4858, 0x0010, 0x2470,
 0x2440, 0x4828, 0x0058, 0x1210, 0x3670,
 0x3642, 0x002a, 0x125a, 0x2412, 0x4872,
 0x4842, 0x122a, 0x245a, 0x3612, 0x0072,
 0x0042, 0x242a, 0x365a, 0x4812, 0x1272,
 0x1242, 0x362a, 0x485a, 0x0012, 0x2472,
 0x2442, 0x482a, 0x005a, 0x1212, 0x3672,
 0x3644, 0x002c, 0x125c, 0x2414, 0x4874,
 0x4844, 0x122c, 0x245c, 0x3614, 0x0074,
 0x0044, 0x242c, 0x365c, 0x4814, 0x1274,
 0x1244, 0x362c, 0x485c, 0x0014, 0x2474,
 0x2444, 0x482c, 0x005c, 0x1214, 0x3674,
 0x3646, 0x002e, 0x125e, 0x2416, 0x4876,
 0x4846, 0x122e, 0x245e, 0x3616, 0x0076,
 0x0046, 0x242e, 0x365e, 0x4816, 0x1276,
 0x1246, 0x362e, 0x485e, 0x0016, 0x2476,
 0x2446, 0x482e, 0x005e, 0x1216, 0x3676,
 0x383c, 0x0224, 0x1454, 0x260c, 0x4a6c,
 0x4a3c, 0x1424, 0x2654, 0x380c, 0x026c,
 0x023c, 0x2624, 0x3854, 0x4a0c, 0x146c,
 0x143c, 0x3824, 0x4a54, 0x020c, 0x266c,
 0x263c, 0x4a24, 0x0254, 0x140c, 0x386c,
 0x383e, 0x0226, 0x1456, 0x260e, 0x4a6e,
 0x4a3e, 0x1426, 0x2656, 0x380e, 0x026e,
 0x023e, 0x2626, 0x3856, 0x4a0e, 0x146e,
 0x143e, 0x3826, 0x4a56, 0x020e, 0x266e,
 0x263e, 0x4a26, 0x0256, 0x140e, 0x386e,
 0x3840, 0x0228, 0x1458, 0x2610, 0x4a70,
 0x4a40, 0x1428, 0x2658, 0x3810, 0x0270,
 0x0240, 0x2628, 0x3858, 0x4a10, 0x1470,
 0x1440, 0x3828, 0x4a58, 0x0210, 0x2670,
 0x2640, 0x4a28, 0x0258, 0x1410, 0x3870,
 0x3842, 0x022a, 0x145a, 0x2612, 0x4a72,
 0x4a42, 0x142a, 0x265a, 0x3812, 0x0272,
 0x0242, 0x262a, 0x385a, 0x4a12, 0x1472,
 0x1442, 0x382a, 0x4a5a, 0x0212, 0x2672,
 0x2642, 0x4a2a, 0x025a, 0x1412, 0x3872,
 0x3844, 0x022c, 0x145c, 0x2614, 0x4a74,
 0x4a44, 0x142c, 0x265c, 0x3814, 0x0274,
 0x0244, 0x262c, 0x385c, 0x4a14, 0x1474,
 0x1444, 0x382c, 0x4a5c, 0x0214, 0x2674,
 0x2644, 0x4a2c, 0x025c, 0x1414, 0x3874,
 0x3846, 0x022e, 0x145e, 0x2616, 0x4a76,
 0x4a46, 0x142e, 0x265e, 0x3816, 0x0276,
 0x0246, 0x262e, 0x385e, 0x4a16, 0x1476,
 0x1446, 0x382e, 0x4a5e, 0x0216, 0x2676,
 0x2646, 0x4a2e, 0x025e, 0x1416, 0x3876,
 0x3a3c, 0x0424, 0x1654, 0x280c, 0x4c6c,
 0x4c3c, 0x1624, 0x2854, 0x3a0c, 0x046c,
 0x043c, 0x2824, 0x3a54, 0x4c0c, 0x166c,
 0x163c, 0x3a24, 0x4c54, 0x040c, 0x286c,
 0x283c, 0x4c24, 0x0454, 0x160c, 0x3a6c,
 0x3a3e, 0x0426, 0x1656, 0x280e, 0x4c6e,
 0x4c3e, 0x1626, 0x2856, 0x3a0e, 0x046e,
 0x043e, 0x2826, 0x3a56, 0x4c0e, 0x166e,
 0x163e, 0x3a26, 0x4c56, 0x040e, 0x286e,
 0x283e, 0x4c26, 0x0456, 0x160e, 0x3a6e,
 0x3a40, 0x0428, 0x1658, 0x2810, 0x4c70,
 0x4c40, 0x1628, 0x2858, 0x3a10, 0x0470,
 0x0440, 0x2828, 0x3a58, 0x4c10, 0x1670,
 0x1640, 0x3a28, 0x4c58, 0x0410, 0x2870,
 0x2840, 0x4c28, 0x0458, 0x1610, 0x3a70,
 0x3a42, 0x042a, 0x165a, 0x2812, 0x4c72,
 0x4c42, 0x162a, 0x285a, 0x3a12, 0x0472,
 0x0442, 0x282a, 0x3a5a, 0x4c12, 0x1672,
 0x1642, 0x3a2a, 0x4c5a, 0x0412, 0x2872,
 0x2842, 0x4c2a, 0x045a, 0x1612, 0x3a72,
 0x3a44, 0x042c, 0x165c, 0x2814, 0x4c74,
 0x4c44, 0x162c, 0x285c, 0x3a14, 0x0474,
 0x0444, 0x282c, 0x3a5c, 0x4c14, 0x1674,
 0x1644, 0x3a2c, 0x4c5c, 0x0414, 0x2874,
 0x2844, 0x4c2c, 0x045c, 0x1614, 0x3a74,
 0x3a46, 0x042e, 0x165e, 0x2816, 0x4c76,
 0x4c46, 0x162e, 0x285e, 0x3a16, 0x0476,
 0x0446, 0x282e, 0x3a5e, 0x4c16, 0x1676,
 0x1646, 0x3a2e, 0x4c5e, 0x0416, 0x2876,
 0x2846, 0x4c2e, 0x045e, 0x1616, 0x3a76,
 0x3c3c, 0x0624, 0x1854, 0x2a0c, 0x4e6c,
 0x4e3c, 0x1824, 0x2a54, 0x3c0c, 0x066c,
 0x063c, 0x2a24, 0x3c54, 0x4e0c, 0x186c,
 0x183c, 0x3c24, 0x4e54, 0x060c, 0x2a6c,
 0x2a3c, 0x4e24, 0x0654, 0x180c, 0x3c6c,
 0x3c3e, 0x0626, 0x1856, 0x2a0e, 0x4e6e,
 0x4e3e, 0x1826, 0x2a56, 0x3c0e, 0x066e,
 0x063e, 0x2a26, 0x3c56, 0x4e0e, 0x186e,
 0x183e, 0x3c26, 0x4e56, 0x060e, 0x2a6e,
 0x2a3e, 0x4e26, 0x0656, 0x180e, 0x3c6e,
 0x3c40, 0x0628, 0x1858, 0x2a10, 0x4e70,
 0x4e40, 0x1828, 0x2a58, 0x3c10, 0x0670,
 0x0640, 0x2a28, 0x3c58, 0x4e10, 0x1870,
 0x1840, 0x3c28, 0x4e58, 0x0610, 0x2a70,
 0x2a40, 0x4e28, 0x0658, 0x1810, 0x3c70,
 0x3c42, 0x062a, 0x185a, 0x2a12, 0x4e72,
 0x4e42, 0x182a, 0x2a5a, 0x3c12, 0x0672,
 0x0642, 0x2a2a, 0x3c5a, 0x4e12, 0x1872,
 0x1842, 0x3c2a, 0x4e5a, 0x0612, 0x2a72,
 0x2a42, 0x4e2a, 0x065a, 0x1812, 0x3c72,
 0x3c44, 0x062c, 0x185c, 0x2a14, 0x4e74,
 0x4e44, 0x182c, 0x2a5c, 0x3c14, 0x0674,
 0x0644, 0x2a2c, 0x3c5c, 0x4e14, 0x1874,
 0x1844, 0x3c2c, 0x4e5c, 0x0614, 0x2a74,
 0x2a44, 0x4e2c, 0x065c, 0x1814, 0x3c74,
 0x3c46, 0x062e, 0x185e, 0x2a16, 0x4e76,
 0x4e46, 0x182e, 0x2a5e, 0x3c16, 0x0676,
 0x0646, 0x2a2e, 0x3c5e, 0x4e16, 0x1876,
 0x1846, 0x3c2e, 0x4e5e, 0x0616, 0x2a76,
 0x2a46, 0x4e2e, 0x065e, 0x1816, 0x3c76,
 0x3e3c, 0x0824, 0x1a54, 0x2c0c, 0x506c,
 0x503c, 0x1a24, 0x2c54, 0x3e0c, 0x086c,
 0x083c, 0x2c24, 0x3e54, 0x500c, 0x1a6c,
 0x1a3c, 0x3e24, 0x5054, 0x080c, 0x2c6c,
 0x2c3c, 0x5024, 0x0854, 0x1a0c, 0x3e6c,
 0x3e3e, 0x0826, 0x1a56, 0x2c0e, 0x506e,
 0x503e, 0x1a26, 0x2c56, 0x3e0e, 0x086e,
 0x083e, 0x2c26, 0x3e56, 0x500e, 0x1a6e,
 0x1a3e, 0x3e26, 0x5056, 0x080e, 0x2c6e,
 0x2c3e, 0x5026, 0x0856, 0x1a0e, 0x3e6e,
 0x3e40, 0x0828, 0x1a58, 0x2c10, 0x5070,
 0x5040, 0x1a28, 0x2c58, 0x3e10, 0x0870,
 0x0840, 0x2c28, 0x3e58, 0x5010, 0x1a70,
 0x1a40, 0x3e28, 0x5058, 0x0810, 0x2c70,
 0x2c40, 0x5028, 0x0858, 0x1a10, 0x3e70,
 0x3e42, 0x082a, 0x1a5a, 0x2c12, 0x5072,
 0x5042, 0x1a2a, 0x2c5a, 0x3e12, 0x0872,
 0x0842, 0x2c2a, 0x3e5a, 0x5012, 0x1a72,
 0x1a42, 0x3e2a, 0x505a, 0x0812, 0x2c72,
 0x2c42, 0x502a, 0x085a, 0x1a12, 0x3e72,
 0x3e44, 0x082c, 0x1a5c, 0x2c14, 0x5074,
 0x5044, 0x1a2c, 0x2c5c, 0x3e14, 0x0874,
 0x0844, 0x2c2c, 0x3e5c, 0x5014, 0x1a74,
 0x1a44, 0x3e2c, 0x505c, 0x0814, 0x2c74,
 0x2c44, 0x502c, 0x085c, 0x1a14, 0x3e74,
 0x3e46, 0x082e, 0x1a5e, 0x2c16, 0x5076,
 0x5046, 0x1a2e, 0x2c5e, 0x3e16, 0x0876,
 0x0846, 0x2c2e, 0x3e5e, 0x5016, 0x1a76,
 0x1a46, 0x3e2e, 0x505e, 0x0816, 0x2c76,
 0x2c46, 0x502e, 0x085e, 0x1a16, 0x3e76,
 0x403c, 0x0a24, 0x1c54, 0x2e0c, 0x526c,
 0x523c, 0x1c24, 0x2e54, 0x400c, 0x0a6c,
 0x0a3c, 0x2e24, 0x4054, 0x520c, 0x1c6c,
 0x1c3c, 0x4024, 0x5254, 0x0a0c, 0x2e6c,
 0x2e3c, 0x5224, 0x0a54, 0x1c0c, 0x406c,
 0x403e, 0x0a26, 0x1c56, 0x2e0e, 0x526e,
 0x523e, 0x1c26, 0x2e56, 0x400e, 0x0a6e,
 0x0a3e, 0x2e26, 0x4056, 0x520e, 0x1c6e,
 0x1c3e, 0x4026, 0x5256, 0x0a0e, 0x2e6e,
 0x2e3e, 0x5226, 0x0a56, 0x1c0e, 0x406e,
 0x4040, 0x0a28, 0x1c58, 0x2e10, 0x5270,
 0x5240, 0x1c28, 0x2e58, 0x4010, 0x0a70,
 0x0a40, 0x2e28, 0x4058, 0x5210, 0x1c70,
 0x1c40, 0x4028, 0x5258, 0x0a10, 0x2e70,
 0x2e40, 0x5228, 0x0a58, 0x1c10, 0x4070,
 0x4042, 0x0a2a, 0x1c5a, 0x2e12, 0x5272,
 0x5242, 0x1c2a, 0x2e5a, 0x4012, 0x0a72,
 0x0a42, 0x2e2a, 0x405a, 0x5212, 0x1c72,
 0x1c42, 0x402a, 0x525a, 0x0a12, 0x2e72,
 0x2e42, 0x522a, 0x0a5a, 0x1c12, 0x4072,
 0x4044, 0x0a2c, 0x1c5c, 0x2e14, 0x5274,
 0x5244, 0x1c2c, 0x2e5c, 0x4014, 0x0a74,
 0x0a44, 0x2e2c, 0x405c, 0x5214, 0x1c74,
 0x1c44, 0x402c, 0x525c, 0x0a14, 0x2e74,
 0x2e44, 0x522c, 0x0a5c, 0x1c14, 0x4074,
 0x4046, 0x0a2e, 0x1c5e, 0x2e16, 0x5276,
 0x5246, 0x1c2e, 0x2e5e, 0x4016, 0x0a76,
 0x0a46, 0x2e2e, 0x405e, 0x5216, 0x1c76,
 0x1c46, 0x402e, 0x525e, 0x0a16, 0x2e76,
 0x2e46, 0x522e, 0x0a5e, 0x1c16, 0x4076,
 0x423c, 0x0c24, 0x1e54, 0x300c, 0x546c,
 0x543c, 0x1e24, 0x3054, 0x420c, 0x0c6c,
 0x0c3c, 0x3024, 0x4254, 0x540c, 0x1e6c,
 0x1e3c, 0x4224, 0x5454, 0x0c0c, 0x306c,
 0x303c, 0x5424, 0x0c54, 0x1e0c, 0x426c,
 0x423e, 0x0c26, 0x1e56, 0x300e, 0x546e,
 0x543e, 0x1e26, 0x3056, 0x420e, 0x0c6e,
 0x0c3e, 0x3026, 0x4256, 0x540e, 0x1e6e,
 0x1e3e, 0x4226, 0x5456, 0x0c0e, 0x306e,
 0x303e, 0x5426, 0x0c56, 0x1e0e, 0x426e,
 0x4240, 0x0c28, 0x1e58, 0x3010, 0x5470,
 0x5440, 0x1e28, 0x3058, 0x4210, 0x0c70,
 0x0c40, 0x3028, 0x4258, 0x5410, 0x1e70,
 0x1e40, 0x4228, 0x5458, 0x0c10, 0x3070,
 0x3040, 0x5428, 0x0c58, 0x1e10, 0x4270,
 0x4242, 0x0c2a, 0x1e5a, 0x3012, 0x5472,
 0x5442, 0x1e2a, 0x305a, 0x4212, 0x0c72,
 0x0c42, 0x302a, 0x425a, 0x5412, 0x1e72,
 0x1e42, 0x422a, 0x545a, 0x0c12, 0x3072,
 0x3042, 0x542a, 0x0c5a, 0x1e12, 0x4272,
 0x4244, 0x0c2c, 0x1e5c, 0x3014, 0x5474,
 0x5444, 0x1e2c, 0x305c, 0x4214, 0x0c74,
 0x0c44, 0x302c, 0x425c, 0x5414, 0x1e74,
 0x1e44, 0x422c, 0x545c, 0x0c14, 0x3074,
 0x3044, 0x542c, 0x0c5c, 0x1e14, 0x4274,
 0x4246, 0x0c2e, 0x1e5e, 0x3016, 0x5476,
 0x5446, 0x1e2e, 0x305e, 0x4216, 0x0c76,
 0x0c46, 0x302e, 0x425e, 0x5416, 0x1e76,
 0x1e46, 0x422e, 0x545e, 0x0c16, 0x3076,
 0x3046, 0x542e, 0x0c5e, 0x1e16, 0x4276,
 0x443c, 0x0e24, 0x2054, 0x320c, 0x566c,
 0x563c, 0x2024, 0x3254, 0x440c, 0x0e6c,
 0x0e3c, 0x3224, 0x4454, 0x560c, 0x206c,
 0x203c, 0x4424, 0x5654, 0x0e0c, 0x326c,
 0x323c, 0x5624, 0x0e54, 0x200c, 0x446c,
 0x443e, 0x0e26, 0x2056, 0x320e, 0x566e,
 0x563e, 0x2026, 0x3256, 0x440e, 0x0e6e,
 0x0e3e, 0x3226, 0x4456, 0x560e, 0x206e,
 0x203e, 0x4426, 0x5656, 0x0e0e, 0x326e,
 0x323e, 0x5626, 0x0e56, 0x200e, 0x446e,
 0x4440, 0x0e28, 0x2058, 0x3210, 0x5670,
 0x5640, 0x2028, 0x3258, 0x4410, 0x0e70,
 0x0e40, 0x3228, 0x4458, 0x5610, 0x2070,
 0x2040, 0x4428, 0x5658, 0x0e10, 0x3270,
 0x3240, 0x5628, 0x0e58, 0x2010, 0x4470,
 0x4442, 0x0e2a, 0x205a, 0x3212, 0x5672,
 0x5642, 0x202a, 0x325a, 0x4412, 0x0e72,
 0x0e42, 0x322a, 0x445a, 0x5612, 0x2072,
 0x2042, 0x442a, 0x565a, 0x0e12, 0x3272,
 0x3242, 0x562a, 0x0e5a, 0x2012, 0x4472,
 0x4444, 0x0e2c, 0x205c, 0x3214, 0x5674,
 0x5644, 0x202c, 0x325c, 0x4414, 0x0e74,
 0x0e44, 0x322c, 0x445c, 0x5614, 0x2074,
 0x2044, 0x442c, 0x565c, 0x0e14, 0x3274,
 0x3244, 0x562c, 0x0e5c, 0x2014, 0x4474,
 0x4446, 0x0e2e, 0x205e, 0x3216, 0x5676,
 0x5646, 0x202e, 0x325e, 0x4416, 0x0e76,
 0x0e46, 0x322e, 0x445e, 0x5616, 0x2076,
 0x2046, 0x442e, 0x565e, 0x0e16, 0x3276,
 0x3246, 0x562e, 0x0e5e, 0x2016, 0x4476,
 0x463c, 0x1024, 0x2254, 0x340c, 0x586c,
 0x583c, 0x2224, 0x3454, 0x460c, 0x106c,
 0x103c, 0x3424, 0x4654, 0x580c, 0x226c,
 0x223c, 0x4624, 0x5854, 0x100c, 0x346c,
 0x343c, 0x5824, 0x1054, 0x220c, 0x466c,
 0x463e, 0x1026, 0x2256, 0x340e, 0x586e,
 0x583e, 0x2226, 0x3456, 0x460e, 0x106e,
 0x103e, 0x3426, 0x4656, 0x580e, 0x226e,
 0x223e, 0x4626, 0x5856, 0x100e, 0x346e,
 0x343e, 0x5826, 0x1056, 0x220e, 0x466e,
 0x4640, 0x1028, 0x2258, 0x3410, 0x5870,
 0x5840, 0x2228, 0x3458, 0x4610, 0x1070,
 0x1040, 0x3428, 0x4658, 0x5810, 0x2270,
 0x2240, 0x4628, 0x5858, 0x1010, 0x3470,
 0x3440, 0x5828, 0x1058, 0x2210, 0x4670,
 0x4642, 0x102a, 0x225a, 0x3412, 0x5872,
 0x5842, 0x222a, 0x345a, 0x4612, 0x1072,
 0x1042, 0x342a, 0x465a, 0x5812, 0x2272,
 0x2242, 0x462a, 0x585a, 0x1012, 0x3472,
 0x3442, 0x582a, 0x105a, 0x2212, 0x4672,
 0x4644, 0x102c, 0x225c, 0x3414, 0x5874,
 0x5844, 0x222c, 0x345c, 0x4614, 0x1074,
 0x1044, 0x342c, 0x465c, 0x5814, 0x2274,
 0x2244, 0x462c, 0x585c, 0x1014, 0x3474,
 0x3444, 0x582c, 0x105c, 0x2214, 0x4674,
 0x4646, 0x102e, 0x225e, 0x3416, 0x5876,
 0x5846, 0x222e, 0x345e, 0x4616, 0x1076,
 0x1046, 0x342e, 0x465e, 0x5816, 0x2276,
 0x2246, 0x462e, 0x585e, 0x1016, 0x3476,
 0x3446, 0x582e, 0x105e, 0x2216, 0x4676,
};

static const uint16_t dv_place_720p50[2*12*27*5] = {
 0x1230, 0x3618, 0x4848, 0x0000, 0x2460,
 0x2430, 0x4818, 0x0048, 0x1200, 0x3660,
 0x3630, 0x0018, 0x1248, 0x2400, 0x4860,
 0x4830, 0x1218, 0x2448, 0x3600, 0x0060,
 0x0030, 0x2418, 0x3648, 0x4800, 0x1260,
 0x1232, 0x361a, 0x484a, 0x0002, 0x2462,
 0x2432, 0x481a, 0x004a, 0x1202, 0x3662,
 0x3632, 0x001a, 0x124a, 0x2402, 0x4862,
 0x4832, 0x121a, 0x244a, 0x3602, 0x0062,
 0x0032, 0x241a, 0x364a, 0x4802, 0x1262,
 0x1234, 0x361c, 0x484c, 0x0004, 0x2464,
 0x2434, 0x481c, 0x004c, 0x1204, 0x3664,
 0x3634, 0x001c, 0x124c, 0x2404, 0x4864,
 0x4834, 0x121c, 0x244c, 0x3604, 0x0064,
 0x0034, 0x241c, 0x364c, 0x4804, 0x1264,
 0x1236, 0x361e, 0x484e, 0x0006, 0x2466,
 0x2436, 0x481e, 0x004e, 0x1206, 0x3666,
 0x3636, 0x001e, 0x124e, 0x2406, 0x4866,
 0x4836, 0x121e, 0x244e, 0x3606, 0x0066,
 0x0036, 0x241e, 0x364e, 0x4806, 0x1266,
 0x1238, 0x3620, 0x4850, 0x0008, 0x2468,
 0x2438, 0x4820, 0x0050, 0x1208, 0x3668,
 0x3638, 0x0020, 0x1250, 0x2408, 0x4868,
 0x4838, 0x1220, 0x2450, 0x3608, 0x0068,
 0x0038, 0x2420, 0x3650, 0x4808, 0x1268,
 0x123a, 0x3622, 0x4852, 0x000a, 0x246a,
 0x243a, 0x4822, 0x0052, 0x120a, 0x366a,
 0x363a, 0x0022, 0x1252, 0x240a, 0x486a,
 0x483a, 0x1222, 0x2452, 0x360a, 0x006a,
 0x003a, 0x2422, 0x3652, 0x480a, 0x126a,
 0x1430, 0x3818, 0x4a48, 0x0200, 0x2660,
 0x2630, 0x4a18, 0x0248, 0x1400, 0x3860,
 0x3830, 0x0218, 0x1448, 0x2600, 0x4a60,
 0x4a30, 0x1418, 0x2648, 0x3800, 0x0260,
 0x0230, 0x2618, 0x3848, 0x4a00, 0x1460,
 0x1432, 0x381a, 0x4a4a, 0x0202, 0x2662,
 0x2632, 0x4a1a, 0x024a, 0x1402, 0x3862,
 0x3832, 0x021a, 0x144a, 0x2602, 0x4a62,
 0x4a32, 0x141a, 0x264a, 0x3802, 0x0262,
 0x0232, 0x261a, 0x384a, 0x4a02, 0x1462,
 0x1434, 0x381c, 0x4a4c, 0x0204, 0x2664,
 0x2634, 0x4a1c, 0x024c, 0x1404, 0x3864,
 0x3834, 0x021c, 0x144c, 0x2604, 0x4a64,
 0x4a34, 0x141c, 0x264c, 0x3804, 0x0264,
 0x0234, 0x261c, 0x384c, 0x4a04, 0x1464,
 0x1436, 0x381e, 0x4a4e, 0x0206, 0x2666,
 0x2636, 0x4a1e, 0x024e, 0x1406, 0x3866,
 0x3836, 0x021e, 0x144e, 0x2606, 0x4a66,
 0x4a36, 0x141e, 0x264e, 0x3806, 0x0266,
 0x0236, 0x261e, 0x384e, 0x4a06, 0x1466,
 0x1438, 0x3820, 0x4a50, 0x0208, 0x2668,
 0x2638, 0x4a20, 0x0250, 0x1408, 0x3868,
 0x3838, 0x0220, 0x1450, 0x2608, 0x4a68,
 0x4a38, 0x1420, 0x2650, 0x3808, 0x0268,
 0x0238, 0x2620, 0x3850, 0x4a08, 0x1468,
 0x143a, 0x3822, 0x4a52, 0x020a, 0x266a,
 0x263a, 0x4a22, 0x0252, 0x140a, 0x386a,
 0x383a, 0x0222, 0x1452, 0x260a, 0x4a6a,
 0x4a3a, 0x1422, 0x2652, 0x380a, 0x026a,
 0x023a, 0x2622, 0x3852, 0x4a0a, 0x146a,
 0x1630, 0x3a18, 0x4c48, 0x0400, 0x2860,
 0x2830, 0x4c18, 0x0448, 0x1600, 0x3a60,
 0x3a30, 0x0418, 0x1648, 0x2800, 0x4c60,
 0x4c30, 0x1618, 0x2848, 0x3a00, 0x0460,
 0x0430, 0x2818, 0x3a48, 0x4c00, 0x1660,
 0x1632, 0x3a1a, 0x4c4a, 0x0402, 0x2862,
 0x2832, 0x4c1a, 0x044a, 0x1602, 0x3a62,
 0x3a32, 0x041a, 0x164a, 0x2802, 0x4c62,
 0x4c32, 0x161a, 0x284a, 0x3a02, 0x0462,
 0x0432, 0x281a, 0x3a4a, 0x4c02, 0x1662,
 0x1634, 0x3a1c, 0x4c4c, 0x0404, 0x2864,
 0x2834, 0x4c1c, 0x044c, 0x1604, 0x3a64,
 0x3a34, 0x041c, 0x164c, 0x2804, 0x4c64,
 0x4c34, 0x161c, 0x284c, 0x3a04, 0x0464,
 0x0434, 0x281c, 0x3a4c, 0x4c04, 0x1664,
 0x1636, 0x3a1e, 0x4c4e, 0x0406, 0x2866,
 0x2836, 0x4c1e, 0x044e, 0x1606, 0x3a66,
 0x3a36, 0x041e, 0x164e, 0x2806, 0x4c66,
 0x4c36, 0x161e, 0x284e, 0x3a06, 0x0466,
 0x0436, 0x281e, 0x3a4e, 0x4c06, 0x1666,
 0x1638, 0x3a20, 0x4c50, 0x0408, 0x2868,
 0x2838, 0x4c20, 0x0450, 0x1608, 0x3a68,
 0x3a38, 0x0420, 0x1650, 0x2808, 0x4c68,
 0x4c38, 0x1620, 0x2850, 0x3a08, 0x0468,
 0x0438, 0x2820, 0x3a50, 0x4c08, 0x1668,
 0x163a, 0x3a22, 0x4c52, 0x040a, 0x286a,
 0x283a, 0x4c22, 0x0452, 0x160a, 0x3a6a,
 0x3a3a, 0x0422, 0x1652, 0x280a, 0x4c6a,
 0x4c3a, 0x1622, 0x2852, 0x3a0a, 0x046a,
 0x043a, 0x2822, 0x3a52, 0x4c0a, 0x166a,
 0x1830, 0x3c18, 0x4e48, 0x0600, 0x2a60,
 0x2a30, 0x4e18, 0x0648, 0x1800, 0x3c60,
 0x3c30, 0x0618, 0x1848, 0x2a00, 0x4e60,
 0x4e30, 0x1818, 0x2a48, 0x3c00, 0x0660,
 0x0630, 0x2a18, 0x3c48, 0x4e00, 0x1860,
 0x1832, 0x3c1a, 0x4e4a, 0x0602, 0x2a62,
 0x2a32, 0x4e1a, 0x064a, 0x1802, 0x3c62,
 0x3c32, 0x061a, 0x184a, 0x2a02, 0x4e62,
 0x4e32, 0x181a, 0x2a4a, 0x3c02, 0x0662,
 0x0632, 0x2a1a, 0x3c4a, 0x4e02, 0x1862,
 0x1834, 0x3c1c, 0x4e4c, 0x0604, 0x2a64,
 0x2a34, 0x4e1c, 0x064c, 0x1804, 0x3c64,
 0x3c34, 0x061c, 0x184c, 0x2a04, 0x4e64,
 0x4e34, 0x181c, 0x2a4c, 0x3c04, 0x0664,
 0x0634, 0x2a1c, 0x3c4c, 0x4e04, 0x1864,
 0x1836, 0x3c1e, 0x4e4e, 0x0606, 0x2a66,
 0x2a36, 0x4e1e, 0x064e, 0x1806, 0x3c66,
 0x3c36, 0x061e, 0x184e, 0x2a06, 0x4e66,
 0x4e36, 0x181e, 0x2a4e, 0x3c06, 0x0666,
 0x0636, 0x2a1e, 0x3c4e, 0x4e06, 0x1866,
 0x1838, 0x3c20, 0x4e50, 0x0608, 0x2a68,
 0x2a38, 0x4e20, 0x0650, 0x1808, 0x3c68,
 0x3c38, 0x0620, 0x1850, 0x2a08, 0x4e68,
 0x4e38, 0x1820, 0x2a50, 0x3c08, 0x0668,
 0x0638, 0x2a20, 0x3c50, 0x4e08, 0x1868,
 0x183a, 0x3c22, 0x4e52, 0x060a, 0x2a6a,
 0x2a3a, 0x4e22, 0x0652, 0x180a, 0x3c6a,
 0x3c3a, 0x0622, 0x1852, 0x2a0a, 0x4e6a,
 0x4e3a, 0x1822, 0x2a52, 0x3c0a, 0x066a,
 0x063a, 0x2a22, 0x3c52, 0x4e0a, 0x186a,
 0x1a30, 0x3e18, 0x5048, 0x0800, 0x2c60,
 0x2c30, 0x5018, 0x0848, 0x1a00, 0x3e60,
 0x3e30, 0x0818, 0x1a48, 0x2c00, 0x5060,
 0x5030, 0x1a18, 0x2c48, 0x3e00, 0x0860,
 0x0830, 0x2c18, 0x3e48, 0x5000, 0x1a60,
 0x1a32, 0x3e1a, 0x504a, 0x0802, 0x2c62,
 0x2c32, 0x501a, 0x084a, 0x1a02, 0x3e62,
 0x3e32, 0x081a, 0x1a4a, 0x2c02, 0x5062,
 0x5032, 0x1a1a, 0x2c4a, 0x3e02, 0x0862,
 0x0832, 0x2c1a, 0x3e4a, 0x5002, 0x1a62,
 0x1a34, 0x3e1c, 0x504c, 0x0804, 0x2c64,
 0x2c34, 0x501c, 0x084c, 0x1a04, 0x3e64,
 0x3e34, 0x081c, 0x1a4c, 0x2c04, 0x5064,
 0x5034, 0x1a1c, 0x2c4c, 0x3e04, 0x0864,
 0x0834, 0x2c1c, 0x3e4c, 0x5004, 0x1a64,
 0x1a36, 0x3e1e, 0x504e, 0x0806, 0x2c66,
 0x2c36, 0x501e, 0x084e, 0x1a06, 0x3e66,
 0x3e36, 0x081e, 0x1a4e, 0x2c06, 0x5066,
 0x5036, 0x1a1e, 0x2c4e, 0x3e06, 0x0866,
 0x0836, 0x2c1e, 0x3e4e, 0x5006, 0x1a66,
 0x1a38, 0x3e20, 0x5050, 0x0808, 0x2c68,
 0x2c38, 0x5020, 0x0850, 0x1a08, 0x3e68,
 0x3e38, 0x0820, 0x1a50, 0x2c08, 0x5068,
 0x5038, 0x1a20, 0x2c50, 0x3e08, 0x0868,
 0x0838, 0x2c20, 0x3e50, 0x5008, 0x1a68,
 0x1a3a, 0x3e22, 0x5052, 0x080a, 0x2c6a,
 0x2c3a, 0x5022, 0x0852, 0x1a0a, 0x3e6a,
 0x3e3a, 0x0822, 0x1a52, 0x2c0a, 0x506a,
 0x503a, 0x1a22, 0x2c52, 0x3e0a, 0x086a,
 0x083a, 0x2c22, 0x3e52, 0x500a, 0x1a6a,
 0x1c30, 0x4018, 0x5248, 0x0a00, 0x2e60,
 0x2e30, 0x5218, 0x0a48, 0x1c00, 0x4060,
 0x4030, 0x0a18, 0x1c48, 0x2e00, 0x5260,
 0x5230, 0x1c18, 0x2e48, 0x4000, 0x0a60,
 0x0a30, 0x2e18, 0x4048, 0x5200, 0x1c60,
 0x1c32, 0x401a, 0x524a, 0x0a02, 0x2e62,
 0x2e32, 0x521a, 0x0a4a, 0x1c02, 0x4062,
 0x4032, 0x0a1a, 0x1c4a, 0x2e02, 0x5262,
 0x5232, 0x1c1a, 0x2e4a, 0x4002, 0x0a62,
 0x0a32, 0x2e1a, 0x404a, 0x5202, 0x1c62,
 0x1c34, 0x401c, 0x524c, 0x0a04, 0x2e64,
 0x2e34, 0x521c, 0x0a4c, 0x1c04, 0x4064,
 0x4034, 0x0a1c, 0x1c4c, 0x2e04, 0x5264,
 0x5234, 0x1c1c, 0x2e4c, 0x4004, 0x0a64,
 0x0a34, 0x2e1c, 0x404c, 0x5204, 0x1c64,
 0x1c36, 0x401e, 0x524e, 0x0a06, 0x2e66,
 0x2e36, 0x521e, 0x0a4e, 0x1c06, 0x4066,
 0x4036, 0x0a1e, 0x1c4e, 0x2e06, 0x5266,
 0x5236, 0x1c1e, 0x2e4e, 0x4006, 0x0a66,
 0x0a36, 0x2e1e, 0x404e, 0x5206, 0x1c66,
 0x1c38, 0x4020, 0x5250, 0x0a08, 0x2e68,
 0x2e38, 0x5220, 0x0a50, 0x1c08, 0x4068,
 0x4038, 0x0a20, 0x1c50, 0x2e08, 0x5268,
 0x5238, 0x1c20, 0x2e50, 0x4008, 0x0a68,
 0x0a38, 0x2e20, 0x4050, 0x5208, 0x1c68,
 0x1c3a, 0x4022, 0x5252, 0x0a0a, 0x2e6a,
 0x2e3a, 0x5222, 0x0a52, 0x1c0a, 0x406a,
 0x403a, 0x0a22, 0x1c52, 0x2e0a, 0x526a,
 0x523a, 0x1c22, 0x2e52, 0x400a, 0x0a6a,
 0x0a3a, 0x2e22, 0x4052, 0x520a, 0x1c6a,
 0x1e30, 0x4218, 0x5448, 0x0c00, 0x3060,
 0x3030, 0x5418, 0x0c48, 0x1e00, 0x4260,
 0x4230, 0x0c18, 0x1e48, 0x3000, 0x5460,
 0x5430, 0x1e18, 0x3048, 0x4200, 0x0c60,
 0x0c30, 0x3018, 0x4248, 0x5400, 0x1e60,
 0x1e32, 0x421a, 0x544a, 0x0c02, 0x3062,
 0x3032, 0x541a, 0x0c4a, 0x1e02, 0x4262,
 0x4232, 0x0c1a, 0x1e4a, 0x3002, 0x5462,
 0x5432, 0x1e1a, 0x304a, 0x4202, 0x0c62,
 0x0c32, 0x301a, 0x424a, 0x5402, 0x1e62,
 0x1e34, 0x421c, 0x544c, 0x0c04, 0x3064,
 0x3034, 0x541c, 0x0c4c, 0x1e04, 0x4264,
 0x4234, 0x0c1c, 0x1e4c, 0x3004, 0x5464,
 0x5434, 0x1e1c, 0x304c, 0x4204, 0x0c64,
 0x0c34, 0x301c, 0x424c, 0x5404, 0x1e64,
 0x1e36, 0x421e, 0x544e, 0x0c06, 0x3066,
 0x3036, 0x541e, 0x0c4e, 0x1e06, 0x4266,
 0x4236, 0x0c1e, 0x1e4e, 0x3006, 0x5466,
 0x5436, 0x1e1e, 0x304e, 0x4206, 0x0c66,
 0x0c36, 0x301e, 0x424e, 0x5406, 0x1e66,
 0x1e38, 0x4220, 0x5450, 0x0c08, 0x3068,
 0x3038, 0x5420, 0x0c50, 0x1e08, 0x4268,
 0x4238, 0x0c20, 0x1e50, 0x3008, 0x5468,
 0x5438, 0x1e20, 0x3050, 0x4208, 0x0c68,
 0x0c38, 0x3020, 0x4250, 0x5408, 0x1e68,
 0x1e3a, 0x4222, 0x5452, 0x0c0a, 0x306a,
 0x303a, 0x5422, 0x0c52, 0x1e0a, 0x426a,
 0x423a, 0x0c22, 0x1e52, 0x300a, 0x546a,
 0x543a, 0x1e22, 0x3052, 0x420a, 0x0c6a,
 0x0c3a, 0x3022, 0x4252, 0x540a, 0x1e6a,
 0x2030, 0x4418, 0x5648, 0x0e00, 0x3260,
 0x3230, 0x5618, 0x0e48, 0x2000, 0x4460,
 0x4430, 0x0e18, 0x2048, 0x3200, 0x5660,
 0x5630, 0x2018, 0x3248, 0x4400, 0x0e60,
 0x0e30, 0x3218, 0x4448, 0x5600, 0x2060,
 0x2032, 0x441a, 0x564a, 0x0e02, 0x3262,
 0x3232, 0x561a, 0x0e4a, 0x2002, 0x4462,
 0x4432, 0x0e1a, 0x204a, 0x3202, 0x5662,
 0x5632, 0x201a, 0x324a, 0x4402, 0x0e62,
 0x0e32, 0x321a, 0x444a, 0x5602, 0x2062,
 0x2034, 0x441c, 0x564c, 0x0e04, 0x3264,
 0x3234, 0x561c, 0x0e4c, 0x2004, 0x4464,
 0x4434, 0x0e1c, 0x204c, 0x3204, 0x5664,
 0x5634, 0x201c, 0x324c, 0x4404, 0x0e64,
 0x0e34, 0x321c, 0x444c, 0x5604, 0x2064,
 0x2036, 0x441e, 0x564e, 0x0e06, 0x3266,
 0x3236, 0x561e, 0x0e4e, 0x2006, 0x4466,
 0x4436, 0x0e1e, 0x204e, 0x3206, 0x5666,
 0x5636, 0x201e, 0x324e, 0x4406, 0x0e66,
 0x0e36, 0x321e, 0x444e, 0x5606, 0x2066,
 0x2038, 0x4420, 0x5650, 0x0e08, 0x3268,
 0x3238, 0x5620, 0x0e50, 0x2008, 0x4468,
 0x4438, 0x0e20, 0x2050, 0x3208, 0x5668,
 0x5638, 0x2020, 0x3250, 0x4408, 0x0e68,
 0x0e38, 0x3220, 0x4450, 0x5608, 0x2068,
 0x203a, 0x4422, 0x5652, 0x0e0a, 0x326a,
 0x323a, 0x5622, 0x0e52, 0x200a, 0x446a,
 0x443a, 0x0e22, 0x2052, 0x320a, 0x566a,
 0x563a, 0x2022, 0x3252, 0x440a, 0x0e6a,
 0x0e3a, 0x3222, 0x4452, 0x560a, 0x206a,
 0x2230, 0x4618, 0x5848, 0x1000, 0x3460,
 0x3430, 0x5818, 0x1048, 0x2200, 0x4660,
 0x4630, 0x1018, 0x2248, 0x3400, 0x5860,
 0x5830, 0x2218, 0x3448, 0x4600, 0x1060,
 0x1030, 0x3418, 0x4648, 0x5800, 0x2260,
 0x2232, 0x461a, 0x584a, 0x1002, 0x3462,
 0x3432, 0x581a, 0x104a, 0x2202, 0x4662,
 0x4632, 0x101a, 0x224a, 0x3402, 0x5862,
 0x5832, 0x221a, 0x344a, 0x4602, 0x1062,
 0x1032, 0x341a, 0x464a, 0x5802, 0x2262,
 0x2234, 0x461c, 0x584c, 0x1004, 0x3464,
 0x3434, 0x581c, 0x104c, 0x2204, 0x4664,
 0x4634, 0x101c, 0x224c, 0x3404, 0x5864,
 0x5834, 0x221c, 0x344c, 0x4604, 0x1064,
 0x1034, 0x341c, 0x464c, 0x5804, 0x2264,
 0x2236, 0x461e, 0x584e, 0x1006, 0x3466,
 0x3436, 0x581e, 0x104e, 0x2206, 0x4666,
 0x4636, 0x101e, 0x224e, 0x3406, 0x5866,
 0x5836, 0x221e, 0x344e, 0x4606, 0x1066,
 0x1036, 0x341e, 0x464e, 0x5806, 0x2266,
 0x2238, 0x4620, 0x5850, 0x1008, 0x3468,
 0x3438, 0x5820, 0x1050, 0x2208, 0x4668,
 0x4638, 0x1020, 0x2250, 0x3408, 0x5868,
 0x5838, 0x2220, 0x3450, 0x4608, 0x1068,
 0x1038, 0x3420, 0x4650, 0x5808, 0x2268,
 0x223a, 0x4622, 0x5852, 0x100a, 0x346a,
 0x343a, 0x5822, 0x1052, 0x220a, 0x466a,
 0x463a, 0x1022, 0x2252, 0x340a, 0x586a,
 0x583a, 0x2222, 0x3452, 0x460a, 0x106a,
 0x103a, 0x3422, 0x4652, 0x580a, 0x226a,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x363c, 0x0024, 0x1254, 0x240c, 0x486c,
 0x483c, 0x1224, 0x2454, 0x360c, 0x006c,
 0x003c, 0x2424, 0x3654, 0x480c, 0x126c,
 0x123c, 0x3624, 0x4854, 0x000c, 0x246c,
 0x243c, 0x4824, 0x0054, 0x120c, 0x366c,
 0x363e, 0x0026, 0x1256, 0x240e, 0x486e,
 0x483e, 0x1226, 0x2456, 0x360e, 0x006e,
 0x003e, 0x2426, 0x3656, 0x480e, 0x126e,
 0x123e, 0x3626, 0x4856, 0x000e, 0x246e,
 0x243e, 0x4826, 0x0056, 0x120e, 0x366e,
 0x3640, 0x0028, 0x1258, 0x2410, 0x4870,
 0x4840, 0x1228, 0x2458, 0x3610, 0x0070,
 0x0040, 0x2428, 0x3658, 0x4810, 0x1270,
 0x1240, 0x3628, 0x4858, 0x0010, 0x2470,
 0x2440, 0x4828, 0x0058, 0x1210, 0x3670,
 0x3642, 0x002a, 0x125a, 0x2412, 0x4872,
 0x4842, 0x122a, 0x245a, 0x3612, 0x0072,
 0x0042, 0x242a, 0x365a, 0x4812, 0x1272,
 0x1242, 0x362a, 0x485a, 0x0012, 0x2472,
 0x2442, 0x482a, 0x005a, 0x1212, 0x3672,
 0x3644, 0x002c, 0x125c, 0x2414, 0x4874,
 0x4844, 0x122c, 0x245c, 0x3614, 0x0074,
 0x0044, 0x242c, 0x365c, 0x4814, 0x1274,
 0x1244, 0x362c, 0x485c, 0x0014, 0x2474,
 0x2444, 0x482c, 0x005c, 0x1214, 0x3674,
 0x3646, 0x002e, 0x125e, 0x2416, 0x4876,
 0x4846, 0x122e, 0x245e, 0x3616, 0x0076,
 0x0046, 0x242e, 0x365e, 0x4816, 0x1276,
 0x1246, 0x362e, 0x485e, 0x0016, 0x2476,
 0x2446, 0x482e, 0x005e, 0x1216, 0x3676,
 0x383c, 0x0224, 0x1454, 0x260c, 0x4a6c,
 0x4a3c, 0x1424, 0x2654, 0x380c, 0x026c,
 0x023c, 0x2624, 0x3854, 0x4a0c, 0x146c,
 0x143c, 0x3824, 0x4a54, 0x020c, 0x266c,
 0x263c, 0x4a24, 0x0254, 0x140c, 0x386c,
 0x383e, 0x0226, 0x1456, 0x260e, 0x4a6e,
 0x4a3e, 0x1426, 0x2656, 0x380e, 0x026e,
 0x023e, 0x2626, 0x3856, 0x4a0e, 0x146e,
 0x143e, 0x3826, 0x4a56, 0x020e, 0x266e,
 0x263e, 0x4a26, 0x0256, 0x140e, 0x386e,
 0x3840, 0x0228, 0x1458, 0x2610, 0x4a70,
 0x4a40, 0x1428, 0x2658, 0x3810, 0x0270,
 0x0240, 0x2628, 0x3858, 0x4a10, 0x1470,
 0x1440, 0x3828, 0x4a58, 0x0210, 0x2670,
 0x2640, 0x4a28, 0x0258, 0x1410, 0x3870,
 0x3842, 0x022a, 0x145a, 0x2612, 0x4a72,
 0x4a42, 0x142a, 0x265a, 0x3812, 0x0272,
 0x0242, 0x262a, 0x385a, 0x4a12, 0x1472,
 0x1442, 0x382a, 0x4a5a, 0x0212, 0x2672,
 0x2642, 0x4a2a, 0x025a, 0x1412, 0x3872,
 0x3844, 0x022c, 0x145c, 0x2614, 0x4a74,
 0x4a44, 0x142c, 0x265c, 0x3814, 0x0274,
 0x0244, 0x262c, 0x385c, 0x4a14, 0x1474,
 0x1444, 0x382c, 0x4a5c, 0x0214, 0x2674,
 0x2644, 0x4a2c, 0x025c, 0x1414, 0x3874,
 0x3846, 0x022e, 0x145e, 0x2616, 0x4a76,
 0x4a46, 0x142e, 0x265e, 0x3816, 0x0276,
 0x0246, 0x262e, 0x385e, 0x4a16, 0x1476,
 0x1446, 0x382e, 0x4a5e, 0x0216, 0x2676,
 0x2646, 0x4a2e, 0x025e, 0x1416, 0x3876,
 0x3a3c, 0x0424, 0x1654, 0x280c, 0x4c6c,
 0x4c3c, 0x1624, 0x2854, 0x3a0c, 0x046c,
 0x043c, 0x2824, 0x3a54, 0x4c0c, 0x166c,
 0x163c, 0x3a24, 0x4c54, 0x040c, 0x286c,
 0x283c, 0x4c24, 0x0454, 0x160c, 0x3a6c,
 0x3a3e, 0x0426, 0x1656, 0x280e, 0x4c6e,
 0x4c3e, 0x1626, 0x2856, 0x3a0e, 0x046e,
 0x043e, 0x2826, 0x3a56, 0x4c0e, 0x166e,
 0x163e, 0x3a26, 0x4c56, 0x040e, 0x286e,
 0x283e, 0x4c26, 0x0456, 0x160e, 0x3a6e,
 0x3a40, 0x0428, 0x1658, 0x2810, 0x4c70,
 0x4c40, 0x1628, 0x2858, 0x3a10, 0x0470,
 0x0440, 0x2828, 0x3a58, 0x4c10, 0x1670,
 0x1640, 0x3a28, 0x4c58, 0x0410, 0x2870,
 0x2840, 0x4c28, 0x0458, 0x1610, 0x3a70,
 0x3a42, 0x042a, 0x165a, 0x2812, 0x4c72,
 0x4c42, 0x162a, 0x285a, 0x3a12, 0x0472,
 0x0442, 0x282a, 0x3a5a, 0x4c12, 0x1672,
 0x1642, 0x3a2a, 0x4c5a, 0x0412, 0x2872,
 0x2842, 0x4c2a, 0x045a, 0x1612, 0x3a72,
 0x3a44, 0x042c, 0x165c, 0x2814, 0x4c74,
 0x4c44, 0x162c, 0x285c, 0x3a14, 0x0474,
 0x0444, 0x282c, 0x3a5c, 0x4c14, 0x1674,
 0x1644, 0x3a2c, 0x4c5c, 0x0414, 0x2874,
 0x2844, 0x4c2c, 0x045c, 0x1614, 0x3a74,
 0x3a46, 0x042e, 0x165e, 0x2816, 0x4c76,
 0x4c46, 0x162e, 0x285e, 0x3a16, 0x0476,
 0x0446, 0x282e, 0x3a5e, 0x4c16, 0x1676,
 0x1646, 0x3a2e, 0x4c5e, 0x0416, 0x2876,
 0x2846, 0x4c2e, 0x045e, 0x1616, 0x3a76,
 0x3c3c, 0x0624, 0x1854, 0x2a0c, 0x4e6c,
 0x4e3c, 0x1824, 0x2a54, 0x3c0c, 0x066c,
 0x063c, 0x2a24, 0x3c54, 0x4e0c, 0x186c,
 0x183c, 0x3c24, 0x4e54, 0x060c, 0x2a6c,
 0x2a3c, 0x4e24, 0x0654, 0x180c, 0x3c6c,
 0x3c3e, 0x0626, 0x1856, 0x2a0e, 0x4e6e,
 0x4e3e, 0x1826, 0x2a56, 0x3c0e, 0x066e,
 0x063e, 0x2a26, 0x3c56, 0x4e0e, 0x186e,
 0x183e, 0x3c26, 0x4e56, 0x060e, 0x2a6e,
 0x2a3e, 0x4e26, 0x0656, 0x180e, 0x3c6e,
 0x3c40, 0x0628, 0x1858, 0x2a10, 0x4e70,
 0x4e40, 0x1828, 0x2a58, 0x3c10, 0x0670,
 0x0640, 0x2a28, 0x3c58, 0x4e10, 0x1870,
 0x1840, 0x3c28, 0x4e58, 0x0610, 0x2a70,
 0x2a40, 0x4e28, 0x0658, 0x1810, 0x3c70,
 0x3c42, 0x062a, 0x185a, 0x2a12, 0x4e72,
 0x4e42, 0x182a, 0x2a5a, 0x3c12, 0x0672,
 0x0642, 0x2a2a, 0x3c5a, 0x4e12, 0x1872,
 0x1842, 0x3c2a, 0x4e5a, 0x0612, 0x2a72,
 0x2a42, 0x4e2a, 0x065a, 0x1812, 0x3c72,
 0x3c44, 0x062c, 0x185c, 0x2a14, 0x4e74,
 0x4e44, 0x182c, 0x2a5c, 0x3c14, 0x0674,
 0x0644, 0x2a2c, 0x3c5c, 0x4e14, 0x1874,
 0x1844, 0x3c2c, 0x4e5c, 0x0614, 0x2a74,
 0x2a44, 0x4e2c, 0x065c, 0x1814, 0x3c74,
 0x3c46, 0x062e, 0x185e, 0x2a16, 0x4e76,
 0x4e46, 0x182e, 0x2a5e, 0x3c16, 0x0676,
 0x0646, 0x2a2e, 0x3c5e, 0x4e16, 0x1876,
 0x1846, 0x3c2e, 0x4e5e, 0x0616, 0x2a76,
 0x2a46, 0x4e2e, 0x065e, 0x1816, 0x3c76,
 0x3e3c, 0x0824, 0x1a54, 0x2c0c, 0x506c,
 0x503c, 0x1a24, 0x2c54, 0x3e0c, 0x086c,
 0x083c, 0x2c24, 0x3e54, 0x500c, 0x1a6c,
 0x1a3c, 0x3e24, 0x5054, 0x080c, 0x2c6c,
 0x2c3c, 0x5024, 0x0854, 0x1a0c, 0x3e6c,
 0x3e3e, 0x0826, 0x1a56, 0x2c0e, 0x506e,
 0x503e, 0x1a26, 0x2c56, 0x3e0e, 0x086e,
 0x083e, 0x2c26, 0x3e56, 0x500e, 0x1a6e,
 0x1a3e, 0x3e26, 0x5056, 0x080e, 0x2c6e,
 0x2c3e, 0x5026, 0x0856, 0x1a0e, 0x3e6e,
 0x3e40, 0x0828, 0x1a58, 0x2c10, 0x5070,
 0x5040, 0x1a28, 0x2c58, 0x3e10, 0x0870,
 0x0840, 0x2c28, 0x3e58, 0x5010, 0x1a70,
 0x1a40, 0x3e28, 0x5058, 0x0810, 0x2c70,
 0x2c40, 0x5028, 0x0858, 0x1a10, 0x3e70,
 0x3e42, 0x082a, 0x1a5a, 0x2c12, 0x5072,
 0x5042, 0x1a2a, 0x2c5a, 0x3e12, 0x0872,
 0x0842, 0x2c2a, 0x3e5a, 0x5012, 0x1a72,
 0x1a42, 0x3e2a, 0x505a, 0x0812, 0x2c72,
 0x2c42, 0x502a, 0x085a, 0x1a12, 0x3e72,
 0x3e44, 0x082c, 0x1a5c, 0x2c14, 0x5074,
 0x5044, 0x1a2c, 0x2c5c, 0x3e14, 0x0874,
 0x0844, 0x2c2c, 0x3e5c, 0x5014, 0x1a74,
 0x1a44, 0x3e2c, 0x505c, 0x0814, 0x2c74,
 0x2c44, 0x502c, 0x085c, 0x1a14, 0x3e74,
 0x3e46, 0x082e, 0x1a5e, 0x2c16, 0x5076,
 0x5046, 0x1a2e, 0x2c5e, 0x3e16, 0x0876,
 0x0846, 0x2c2e, 0x3e5e, 0x5016, 0x1a76,
 0x1a46, 0x3e2e, 0x505e, 0x0816, 0x2c76,
 0x2c46, 0x502e, 0x085e, 0x1a16, 0x3e76,
 0x403c, 0x0a24, 0x1c54, 0x2e0c, 0x526c,
 0x523c, 0x1c24, 0x2e54, 0x400c, 0x0a6c,
 0x0a3c, 0x2e24, 0x4054, 0x520c, 0x1c6c,
 0x1c3c, 0x4024, 0x5254, 0x0a0c, 0x2e6c,
 0x2e3c, 0x5224, 0x0a54, 0x1c0c, 0x406c,
 0x403e, 0x0a26, 0x1c56, 0x2e0e, 0x526e,
 0x523e, 0x1c26, 0x2e56, 0x400e, 0x0a6e,
 0x0a3e, 0x2e26, 0x4056, 0x520e, 0x1c6e,
 0x1c3e, 0x4026, 0x5256, 0x0a0e, 0x2e6e,
 0x2e3e, 0x5226, 0x0a56, 0x1c0e, 0x406e,
 0x4040, 0x0a28, 0x1c58, 0x2e10, 0x5270,
 0x5240, 0x1c28, 0x2e58, 0x4010, 0x0a70,
 0x0a40, 0x2e28, 0x4058, 0x5210, 0x1c70,
 0x1c40, 0x4028, 0x5258, 0x0a10, 0x2e70,
 0x2e40, 0x5228, 0x0a58, 0x1c10, 0x4070,
 0x4042, 0x0a2a, 0x1c5a, 0x2e12, 0x5272,
 0x5242, 0x1c2a, 0x2e5a, 0x4012, 0x0a72,
 0x0a42, 0x2e2a, 0x405a, 0x5212, 0x1c72,
 0x1c42, 0x402a, 0x525a, 0x0a12, 0x2e72,
 0x2e42, 0x522a, 0x0a5a, 0x1c12, 0x4072,
 0x4044, 0x0a2c, 0x1c5c, 0x2e14, 0x5274,
 0x5244, 0x1c2c, 0x2e5c, 0x4014, 0x0a74,
 0x0a44, 0x2e2c, 0x405c, 0x5214, 0x1c74,
 0x1c44, 0x402c, 0x525c, 0x0a14, 0x2e74,
 0x2e44, 0x522c, 0x0a5c, 0x1c14, 0x4074,
 0x4046, 0x0a2e, 0x1c5e, 0x2e16, 0x5276,
 0x5246, 0x1c2e, 0x2e5e, 0x4016, 0x0a76,
 0x0a46, 0x2e2e, 0x405e, 0x5216, 0x1c76,
 0x1c46, 0x402e, 0x525e, 0x0a16, 0x2e76,
 0x2e46, 0x522e, 0x0a5e, 0x1c16, 0x4076,
 0x423c, 0x0c24, 0x1e54, 0x300c, 0x546c,
 0x543c, 0x1e24, 0x3054, 0x420c, 0x0c6c,
 0x0c3c, 0x3024, 0x4254, 0x540c, 0x1e6c,
 0x1e3c, 0x4224, 0x5454, 0x0c0c, 0x306c,
 0x303c, 0x5424, 0x0c54, 0x1e0c, 0x426c,
 0x423e, 0x0c26, 0x1e56, 0x300e, 0x546e,
 0x543e, 0x1e26, 0x3056, 0x420e, 0x0c6e,
 0x0c3e, 0x3026, 0x4256, 0x540e, 0x1e6e,
 0x1e3e, 0x4226, 0x5456, 0x0c0e, 0x306e,
 0x303e, 0x5426, 0x0c56, 0x1e0e, 0x426e,
 0x4240, 0x0c28, 0x1e58, 0x3010, 0x5470,
 0x5440, 0x1e28, 0x3058, 0x4210, 0x0c70,
 0x0c40, 0x3028, 0x4258, 0x5410, 0x1e70,
 0x1e40, 0x4228, 0x5458, 0x0c10, 0x3070,
 0x3040, 0x5428, 0x0c58, 0x1e10, 0x4270,
 0x4242, 0x0c2a, 0x1e5a, 0x3012, 0x5472,
 0x5442, 0x1e2a, 0x305a, 0x4212, 0x0c72,
 0x0c42, 0x302a, 0x425a, 0x5412, 0x1e72,
 0x1e42, 0x422a, 0x545a, 0x0c12, 0x3072,
 0x3042, 0x542a, 0x0c5a, 0x1e12, 0x4272,
 0x4244, 0x0c2c, 0x1e5c, 0x3014, 0x5474,
 0x5444, 0x1e2c, 0x305c, 0x4214, 0x0c74,
 0x0c44, 0x302c, 0x425c, 0x5414, 0x1e74,
 0x1e44, 0x422c, 0x545c, 0x0c14, 0x3074,
 0x3044, 0x542c, 0x0c5c, 0x1e14, 0x4274,
 0x4246, 0x0c2e, 0x1e5e, 0x3016, 0x5476,
 0x5446, 0x1e2e, 0x305e, 0x4216, 0x0c76,
 0x0c46, 0x302e, 0x425e, 0x5416, 0x1e76,
 0x1e46, 0x422e, 0x545e, 0x0c16, 0x3076,
 0x3046, 0x542e, 0x0c5e, 0x1e16, 0x4276,
 0x443c, 0x0e24, 0x2054, 0x320c, 0x566c,
 0x563c, 0x2024, 0x3254, 0x440c, 0x0e6c,
 0x0e3c, 0x3224, 0x4454, 0x560c, 0x206c,
 0x203c, 0x4424, 0x5654, 0x0e0c, 0x326c,
 0x323c, 0x5624, 0x0e54, 0x200c, 0x446c,
 0x443e, 0x0e26, 0x2056, 0x320e, 0x566e,
 0x563e, 0x2026, 0x3256, 0x440e, 0x0e6e,
 0x0e3e, 0x3226, 0x4456, 0x560e, 0x206e,
 0x203e, 0x4426, 0x5656, 0x0e0e, 0x326e,
 0x323e, 0x5626, 0x0e56, 0x200e, 0x446e,
 0x4440, 0x0e28, 0x2058, 0x3210, 0x5670,
 0x5640, 0x2028, 0x3258, 0x4410, 0x0e70,
 0x0e40, 0x3228, 0x4458, 0x5610, 0x2070,
 0x2040, 0x4428, 0x5658, 0x0e10, 0x3270,
 0x3240, 0x5628, 0x0e58, 0x2010, 0x4470,
 0x4442, 0x0e2a, 0x205a, 0x3212, 0x5672,
 0x5642, 0x202a, 0x325a, 0x4412, 0x0e72,
 0x0e42, 0x322a, 0x445a, 0x5612, 0x2072,
 0x2042, 0x442a, 0x565a, 0x0e12, 0x3272,
 0x3242, 0x562a, 0x0e5a, 0x2012, 0x4472,
 0x4444, 0x0e2c, 0x205c, 0x3214, 0x5674,
 0x5644, 0x202c, 0x325c, 0x4414, 0x0e74,
 0x0e44, 0x322c, 0x445c, 0x5614, 0x2074,
 0x2044, 0x442c, 0x565c, 0x0e14, 0x3274,
 0x3244, 0x562c, 0x0e5c, 0x2014, 0x4474,
 0x4446, 0x0e2e, 0x205e, 0x3216, 0x5676,
 0x5646, 0x202e, 0x325e, 0x4416, 0x0e76,
 0x0e46, 0x322e, 0x445e, 0x5616, 0x2076,
 0x2046, 0x442e, 0x565e, 0x0e16, 0x3276,
 0x3246, 0x562e, 0x0e5e, 0x2016, 0x4476,
 0x463c, 0x1024, 0x2254, 0x340c, 0x586c,
 0x583c, 0x2224, 0x3454, 0x460c, 0x106c,
 0x103c, 0x3424, 0x4654, 0x580c, 0x226c,
 0x223c, 0x4624, 0x5854, 0x100c, 0x346c,
 0x343c, 0x5824, 0x1054, 0x220c, 0x466c,
 0x463e, 0x1026, 0x2256, 0x340e, 0x586e,
 0x583e, 0x2226, 0x3456, 0x460e, 0x106e,
 0x103e, 0x3426, 0x4656, 0x580e, 0x226e,
 0x223e, 0x4626, 0x5856, 0x100e, 0x346e,
 0x343e, 0x5826, 0x1056, 0x220e, 0x466e,
 0x4640, 0x1028, 0x2258, 0x3410, 0x5870,
 0x5840, 0x2228, 0x3458, 0x4610, 0x1070,
 0x1040, 0x3428, 0x4658, 0x5810, 0x2270,
 0x2240, 0x4628, 0x5858, 0x1010, 0x3470,
 0x3440, 0x5828, 0x1058, 0x2210, 0x4670,
 0x4642, 0x102a, 0x225a, 0x3412, 0x5872,
 0x5842, 0x222a, 0x345a, 0x4612, 0x1072,
 0x1042, 0x342a, 0x465a, 0x5812, 0x2272,
 0x2242, 0x462a, 0x585a, 0x1012, 0x3472,
 0x3442, 0x582a, 0x105a, 0x2212, 0x4672,
 0x4644, 0x102c, 0x225c, 0x3414, 0x5874,
 0x5844, 0x222c, 0x345c, 0x4614, 0x1074,
 0x1044, 0x342c, 0x465c, 0x5814, 0x2274,
 0x2244, 0x462c, 0x585c, 0x1014, 0x3474,
 0x3444, 0x582c, 0x105c, 0x2214, 0x4674,
 0x4646, 0x102e, 0x225e, 0x3416, 0x5876,
 0x5846, 0x222e, 0x345e, 0x4616, 0x1076,
 0x1046, 0x342e, 0x465e, 0x5816, 0x2276,
 0x2246, 0x462e, 0x585e, 0x1016, 0x3476,
 0x3446, 0x582e, 0x105e, 0x2216, 0x4676,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};


/* DV25/50 DCT coefficient weights and inverse weights */
/* created by dvtables.py */
static const int dv_weight_bits = 18;
static const int dv_weight_88[64] = {
 131072, 257107, 257107, 242189, 252167, 242189, 235923, 237536,
 237536, 235923, 229376, 231390, 223754, 231390, 229376, 222935,
 224969, 217965, 217965, 224969, 222935, 200636, 218652, 211916,
 212325, 211916, 218652, 200636, 188995, 196781, 205965, 206433,
 206433, 205965, 196781, 188995, 185364, 185364, 200636, 200704,
 200636, 185364, 185364, 174609, 180568, 195068, 195068, 180568,
 174609, 170091, 175557, 189591, 175557, 170091, 165371, 170627,
 170627, 165371, 160727, 153560, 160727, 144651, 144651, 136258,
};
static const int dv_weight_248[64] = {
 131072, 242189, 257107, 237536, 229376, 200636, 242189, 223754,
 224969, 196781, 262144, 242189, 229376, 200636, 257107, 237536,
 211916, 185364, 235923, 217965, 229376, 211916, 206433, 180568,
 242189, 223754, 224969, 196781, 211916, 185364, 235923, 217965,
 200704, 175557, 222935, 205965, 200636, 185364, 195068, 170627,
 229376, 211916, 206433, 180568, 200704, 175557, 222935, 205965,
 175557, 153560, 188995, 174609, 165371, 144651, 200636, 185364,
 195068, 170627, 175557, 153560, 188995, 174609, 165371, 144651,
};
static const int dv_iweight_bits = 14;
static const int dv_iweight_88[64] = {
 32768, 16710, 16710, 17735, 17015, 17735, 18197, 18079,
 18079, 18197, 18725, 18559, 19196, 18559, 18725, 19284,
 19108, 19692, 19692, 19108, 19284, 21400, 19645, 20262,
 20214, 20262, 19645, 21400, 22733, 21845, 20867, 20815,
 20815, 20867, 21845, 22733, 23173, 23173, 21400, 21400,
 21400, 23173, 23173, 24600, 23764, 22017, 22017, 23764,
 24600, 25267, 24457, 22672, 24457, 25267, 25971, 25191,
 25191, 25971, 26715, 27962, 26715, 29642, 29642, 31536,
};
static const int dv_iweight_248[64] = {
 32768, 17735, 16710, 18079, 18725, 21400, 17735, 19196,
 19108, 21845, 16384, 17735, 18725, 21400, 16710, 18079,
 20262, 23173, 18197, 19692, 18725, 20262, 20815, 23764,
 17735, 19196, 19108, 21845, 20262, 23173, 18197, 19692,
 21400, 24457, 19284, 20867, 21400, 23173, 22017, 25191,
 18725, 20262, 20815, 23764, 21400, 24457, 19284, 20867,
 24457, 27962, 22733, 24600, 25971, 29642, 21400, 23173,
 22017, 25191, 24457, 27962, 22733, 24600, 25971, 29642,
};

/* the "inverse" DV100 weights are actually just the spec weights (zig-zagged) */
static const int dv_iweight_1080_y[64] = {
    128, 16, 16, 17, 17, 17, 18, 18,
    18, 18, 18, 18, 19, 18, 18, 19,
    19, 19, 19, 19, 19, 42, 38, 40,
    40, 40, 38, 42, 44, 43, 41, 41,
    41, 41, 43, 44, 45, 45, 42, 42,
    42, 45, 45, 48, 46, 43, 43, 46,
    48, 49, 48, 44, 48, 49, 101, 98,
    98, 101, 104, 109, 104, 116, 116, 123,
};
static const int dv_iweight_1080_c[64] = {
    128, 16, 16, 17, 17, 17, 25, 25,
    25, 25, 26, 25, 26, 25, 26, 26,
    26, 27, 27, 26, 26, 42, 38, 40,
    40, 40, 38, 42, 44, 43, 41, 41,
    41, 41, 43, 44, 91, 91, 84, 84,
    84, 91, 91, 96, 93, 86, 86, 93,
    96, 197, 191, 177, 191, 197, 203, 197,
    197, 203, 209, 219, 209, 232, 232, 246,
};
static const int dv_iweight_720_y[64] = {
    128, 16, 16, 17, 17, 17, 18, 18,
    18, 18, 18, 18, 19, 18, 18, 19,
    19, 19, 19, 19, 19, 42, 38, 40,
    40, 40, 38, 42, 44, 43, 41, 41,
    41, 41, 43, 44, 68, 68, 63, 63,
    63, 68, 68, 96, 92, 86, 86, 92,
    96, 98, 96, 88, 96, 98, 202, 196,
    196, 202, 208, 218, 208, 232, 232, 246,
};
static const int dv_iweight_720_c[64] = {
    128, 24, 24, 26, 26, 26, 36, 36,
    36, 36, 36, 36, 38, 36, 36, 38,
    38, 38, 38, 38, 38, 84, 76, 80,
    80, 80, 76, 84, 88, 86, 82, 82,
    82, 82, 86, 88, 182, 182, 168, 168,
    168, 182, 182, 192, 186, 192, 172, 186,
    192, 394, 382, 354, 382, 394, 406, 394,
    394, 406, 418, 438, 418, 464, 464, 492,
};

static const uint8_t dv_audio_shuffle525[10][9] = {
  {  0, 30, 60, 20, 50, 80, 10, 40, 70 }, /* 1st channel */
  {  6, 36, 66, 26, 56, 86, 16, 46, 76 },
  { 12, 42, 72,  2, 32, 62, 22, 52, 82 },
  { 18, 48, 78,  8, 38, 68, 28, 58, 88 },
  { 24, 54, 84, 14, 44, 74,  4, 34, 64 },

  {  1, 31, 61, 21, 51, 81, 11, 41, 71 }, /* 2nd channel */
  {  7, 37, 67, 27, 57, 87, 17, 47, 77 },
  { 13, 43, 73,  3, 33, 63, 23, 53, 83 },
  { 19, 49, 79,  9, 39, 69, 29, 59, 89 },
  { 25, 55, 85, 15, 45, 75,  5, 35, 65 },
};

static const uint8_t dv_audio_shuffle625[12][9] = {
  {   0,  36,  72,  26,  62,  98,  16,  52,  88}, /* 1st channel */
  {   6,  42,  78,  32,  68, 104,  22,  58,  94},
  {  12,  48,  84,   2,  38,  74,  28,  64, 100},
  {  18,  54,  90,   8,  44,  80,  34,  70, 106},
  {  24,  60,  96,  14,  50,  86,   4,  40,  76},
  {  30,  66, 102,  20,  56,  92,  10,  46,  82},

  {   1,  37,  73,  27,  63,  99,  17,  53,  89}, /* 2nd channel */
  {   7,  43,  79,  33,  69, 105,  23,  59,  95},
  {  13,  49,  85,   3,  39,  75,  29,  65, 101},
  {  19,  55,  91,   9,  45,  81,  35,  71, 107},
  {  25,  61,  97,  15,  51,  87,   5,  41,  77},
  {  31,  67, 103,  21,  57,  93,  11,  47,  83},
};

static const av_unused int dv_audio_frequency[3] = {
    48000, 44100, 32000,
};

/* macroblock bit budgets */
static const uint8_t block_sizes_dv2550[8] = {
    112, 112, 112, 112, 80, 80, 0, 0,
};

static const uint8_t block_sizes_dv100[8] = {
    80, 80, 80, 80, 80, 80, 64, 64,
};

static const DVprofile dv_profiles[] = {
    { .dsf = 0,
      .video_stype = 0x0,
      .frame_size = 120000,        /* IEC 61834, SMPTE-314M - 525/60 (NTSC) */
      .difseg_size = 10,
      .n_difchan = 1,
      .frame_rate = 30000,
      .ltc_divisor = 30,
      .frame_rate_base = 1001,
      .height = 480,
      .width = 720,
      .sar = {{10, 11}, {40, 33}},
      .video_place = dv_place_411,
      .pix_fmt = PIX_FMT_YUV411P,
      .bpm = 6,
      .block_sizes = block_sizes_dv2550,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .video_stype = 0x0,
      .frame_size = 144000,        /* IEC 61834 - 625/50 (PAL) */
      .difseg_size = 12,
      .n_difchan = 1,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_420,
      .pix_fmt = PIX_FMT_YUV420P,
      .bpm = 6,
      .block_sizes = block_sizes_dv2550,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 1,
      .video_stype = 0x0,
      .frame_size = 144000,        /* SMPTE-314M - 625/50 (PAL) */
      .difseg_size = 12,
      .n_difchan = 1,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_411P,
      .pix_fmt = PIX_FMT_YUV411P,
      .bpm = 6,
      .block_sizes = block_sizes_dv2550,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 0,
      .video_stype = 0x4,
      .frame_size = 240000,        /* SMPTE-314M - 525/60 (NTSC) 50 Mbps */
      .difseg_size = 10,           /* also known as "DVCPRO50" */
      .n_difchan = 2,
      .frame_rate = 30000,
      .ltc_divisor = 30,
      .frame_rate_base = 1001,
      .height = 480,
      .width = 720,
      .sar = {{10, 11}, {40, 33}},
      .video_place = dv_place_422_525,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 6,
      .block_sizes = block_sizes_dv2550,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .video_stype = 0x4,
      .frame_size = 288000,        /* SMPTE-314M - 625/50 (PAL) 50 Mbps */
      .difseg_size = 12,           /* also known as "DVCPRO50" */
      .n_difchan = 2,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_422_625,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 6,
      .block_sizes = block_sizes_dv2550,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 0,
      .video_stype = 0x14,
      .frame_size = 480000,        /* SMPTE-370M - 1080i60 100 Mbps */
      .difseg_size = 10,           /* also known as "DVCPRO HD" */
      .n_difchan = 4,
      .frame_rate = 30000,
      .ltc_divisor = 30,
      .frame_rate_base = 1001,
      .height = 1080,
      .width = 1280,
      .sar = {{1, 1}, {1, 1}},
      .video_place = dv_place_1080i60,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 8,
      .block_sizes = block_sizes_dv100,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .video_stype = 0x14,
      .frame_size = 576000,        /* SMPTE-370M - 1080i50 100 Mbps */
      .difseg_size = 12,           /* also known as "DVCPRO HD" */
      .n_difchan = 4,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 1080,
      .width = 1440,
      .sar = {{1, 1}, {1, 1}},
      .video_place = dv_place_1080i50,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 8,
      .block_sizes = block_sizes_dv100,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 0,
      .video_stype = 0x18,
      .frame_size = 240000,        /* SMPTE-370M - 720p60 100 Mbps */
      .difseg_size = 10,           /* also known as "DVCPRO HD" */
      .n_difchan = 2,
      .frame_rate = 60000,
      .ltc_divisor = 60,
      .frame_rate_base = 1001,
      .height = 720,
      .width = 960,
      .sar = {{1, 1}, {1, 1}},
      .video_place = dv_place_720p60,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 8,
      .block_sizes = block_sizes_dv100,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .video_stype = 0x18,
      .frame_size = 288000,        /* SMPTE-370M - 720p50 100 Mbps */
      .difseg_size = 12,           /* also known as "DVCPRO HD" */
      .n_difchan = 2,
      .frame_rate = 50,
      .ltc_divisor = 50,
      .frame_rate_base = 1,
      .height = 720,
      .width = 960,
      .sar = {{1, 1}, {1, 1}},
      .video_place = dv_place_720p50,
      .pix_fmt = PIX_FMT_YUV422P,
      .bpm = 8,
      .block_sizes = block_sizes_dv100,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    }
};

enum dv_section_type {
     dv_sect_header  = 0x1f,
     dv_sect_subcode = 0x3f,
     dv_sect_vaux    = 0x56,
     dv_sect_audio   = 0x76,
     dv_sect_video   = 0x96,
};

enum dv_pack_type {
     dv_header525     = 0x3f, /* see dv_write_pack for important details on */
     dv_header625     = 0xbf, /* these two packs */
     dv_timecode      = 0x13,
     dv_audio_source  = 0x50,
     dv_audio_control = 0x51,
     dv_audio_recdate = 0x52,
     dv_audio_rectime = 0x53,
     dv_video_source  = 0x60,
     dv_video_control = 0x61,
     dv_video_recdate = 0x62,
     dv_video_rectime = 0x63,
     dv_unknown_pack  = 0xff,
};

#define DV_PROFILE_IS_HD(p) ((p)->video_stype & 0x10)
#define DV_PROFILE_IS_1080i50(p) (((p)->video_stype == 0x14) && ((p)->dsf == 1))
#define DV_PROFILE_IS_720p50(p)  (((p)->video_stype == 0x18) && ((p)->dsf == 1))

/* minimum number of bytes to read from a DV stream in order to determine the profile */
#define DV_PROFILE_BYTES (6*80) /* 6 DIF blocks */

/* largest possible DV frame, in bytes (1080i50) */
#define DV_MAX_FRAME_SIZE 576000

/* maximum number of blocks per macroblock in any DV format */
#define DV_MAX_BPM 8

static inline const DVprofile* dv_frame_profile(const uint8_t* frame)
{
   int i;

   int dsf = (frame[3] & 0x80) >> 7;

   int stype = frame[80*5 + 48 + 3] & 0x1f;

   /* 576i50 25Mbps 4:1:1 is a special case */
   if (dsf == 1 && stype == 0 && frame[5] & 0x07) {
       return &dv_profiles[2];
   }

   for (i=0; i<sizeof(dv_profiles)/sizeof(DVprofile); i++)
       if (dsf == dv_profiles[i].dsf && stype == dv_profiles[i].video_stype)
           return &dv_profiles[i];

   return NULL;
}

static inline const DVprofile* dv_codec_profile(AVCodecContext* codec)
{
    int i;

    for (i=0; i<sizeof(dv_profiles)/sizeof(DVprofile); i++)
       if (codec->height == dv_profiles[i].height && codec->pix_fmt == dv_profiles[i].pix_fmt &&
           codec->width == dv_profiles[i].width)
               return &dv_profiles[i];

    return NULL;
}

static inline int dv_write_dif_id(enum dv_section_type t, uint8_t chan_num, uint8_t seq_num,
                                  uint8_t dif_num, uint8_t* buf)
{
    buf[0] = (uint8_t)t;    /* Section type */
    buf[1] = (seq_num<<4) | /* DIF seq number 0-9 for 525/60; 0-11 for 625/50 */
             (chan_num << 3) | /* FSC: for 50Mb/s 0 - first channel; 1 - second */
             7;             /* reserved -- always 1 */
    buf[2] = dif_num;       /* DIF block number Video: 0-134, Audio: 0-8 */
    return 3;
}


static inline int dv_write_ssyb_id(uint8_t syb_num, uint8_t fr, uint8_t* buf)
{
    if (syb_num == 0 || syb_num == 6) {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 (0<<4)  | /* AP3 (Subcode application ID) */
                 0x0f;     /* reserved -- always 1 */
    }
    else if (syb_num == 11) {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 0x7f;     /* reserved -- always 1 */
    }
    else {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 (0<<4)  | /* APT (Track application ID) */
                 0x0f;     /* reserved -- always 1 */
    }
    buf[1] = 0xf0 |            /* reserved -- always 1 */
             (syb_num & 0x0f); /* SSYB number 0 - 11   */
    buf[2] = 0xff;             /* reserved -- always 1 */
    return 3;
}

#endif /* FFMPEG_DVDATA_H */
