/*
 * AAC data
 * Copyright (c) 2005-2006 Oded Shimon ( ods15 ods15 dyndns org )
 * Copyright (c) 2006-2007 Maxim Gavrilov ( maxim.gavrilov gmail com )
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file libavcodec/aactab.c
 * AAC data
 * @author Oded Shimon  ( ods15 ods15 dyndns org )
 * @author Maxim Gavrilov ( maxim.gavrilov gmail com )
 */

#include "libavutil/mem.h"
#include "aac.h"

#include <stdint.h>

DECLARE_ALIGNED(16, float,  ff_aac_kbd_long_1024)[1024];
DECLARE_ALIGNED(16, float,  ff_aac_kbd_short_128)[128];

const uint8_t ff_aac_num_swb_1024[] = {
    41, 41, 47, 49, 49, 51, 47, 47, 43, 43, 43, 40, 40
};

const uint8_t ff_aac_num_swb_128[] = {
    12, 12, 12, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15
};

const uint8_t ff_aac_pred_sfb_max[] = {
    33, 33, 38, 40, 40, 40, 41, 41, 37, 37, 37, 34, 34
};

const uint32_t ff_aac_scalefactor_code[121] = {
    0x3ffe8, 0x3ffe6, 0x3ffe7, 0x3ffe5, 0x7fff5, 0x7fff1, 0x7ffed, 0x7fff6,
    0x7ffee, 0x7ffef, 0x7fff0, 0x7fffc, 0x7fffd, 0x7ffff, 0x7fffe, 0x7fff7,
    0x7fff8, 0x7fffb, 0x7fff9, 0x3ffe4, 0x7fffa, 0x3ffe3, 0x1ffef, 0x1fff0,
    0x0fff5, 0x1ffee, 0x0fff2, 0x0fff3, 0x0fff4, 0x0fff1, 0x07ff6, 0x07ff7,
    0x03ff9, 0x03ff5, 0x03ff7, 0x03ff3, 0x03ff6, 0x03ff2, 0x01ff7, 0x01ff5,
    0x00ff9, 0x00ff7, 0x00ff6, 0x007f9, 0x00ff4, 0x007f8, 0x003f9, 0x003f7,
    0x003f5, 0x001f8, 0x001f7, 0x000fa, 0x000f8, 0x000f6, 0x00079, 0x0003a,
    0x00038, 0x0001a, 0x0000b, 0x00004, 0x00000, 0x0000a, 0x0000c, 0x0001b,
    0x00039, 0x0003b, 0x00078, 0x0007a, 0x000f7, 0x000f9, 0x001f6, 0x001f9,
    0x003f4, 0x003f6, 0x003f8, 0x007f5, 0x007f4, 0x007f6, 0x007f7, 0x00ff5,
    0x00ff8, 0x01ff4, 0x01ff6, 0x01ff8, 0x03ff8, 0x03ff4, 0x0fff0, 0x07ff4,
    0x0fff6, 0x07ff5, 0x3ffe2, 0x7ffd9, 0x7ffda, 0x7ffdb, 0x7ffdc, 0x7ffdd,
    0x7ffde, 0x7ffd8, 0x7ffd2, 0x7ffd3, 0x7ffd4, 0x7ffd5, 0x7ffd6, 0x7fff2,
    0x7ffdf, 0x7ffe7, 0x7ffe8, 0x7ffe9, 0x7ffea, 0x7ffeb, 0x7ffe6, 0x7ffe0,
    0x7ffe1, 0x7ffe2, 0x7ffe3, 0x7ffe4, 0x7ffe5, 0x7ffd7, 0x7ffec, 0x7fff4,
    0x7fff3,
};

const uint8_t ff_aac_scalefactor_bits[121] = {
    18, 18, 18, 18, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 18, 19, 18, 17, 17, 16, 17, 16, 16, 16, 16, 15, 15,
    14, 14, 14, 14, 14, 14, 13, 13, 12, 12, 12, 11, 12, 11, 10, 10,
    10,  9,  9,  8,  8,  8,  7,  6,  6,  5,  4,  3,  1,  4,  4,  5,
     6,  6,  7,  7,  8,  8,  9,  9, 10, 10, 10, 11, 11, 11, 11, 12,
    12, 13, 13, 13, 14, 14, 16, 15, 16, 15, 18, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19, 19,
};

static const uint16_t codes1[81] = {
    0x7f8, 0x1f1, 0x7fd, 0x3f5, 0x068, 0x3f0, 0x7f7, 0x1ec,
    0x7f5, 0x3f1, 0x072, 0x3f4, 0x074, 0x011, 0x076, 0x1eb,
    0x06c, 0x3f6, 0x7fc, 0x1e1, 0x7f1, 0x1f0, 0x061, 0x1f6,
    0x7f2, 0x1ea, 0x7fb, 0x1f2, 0x069, 0x1ed, 0x077, 0x017,
    0x06f, 0x1e6, 0x064, 0x1e5, 0x067, 0x015, 0x062, 0x012,
    0x000, 0x014, 0x065, 0x016, 0x06d, 0x1e9, 0x063, 0x1e4,
    0x06b, 0x013, 0x071, 0x1e3, 0x070, 0x1f3, 0x7fe, 0x1e7,
    0x7f3, 0x1ef, 0x060, 0x1ee, 0x7f0, 0x1e2, 0x7fa, 0x3f3,
    0x06a, 0x1e8, 0x075, 0x010, 0x073, 0x1f4, 0x06e, 0x3f7,
    0x7f6, 0x1e0, 0x7f9, 0x3f2, 0x066, 0x1f5, 0x7ff, 0x1f7,
    0x7f4,
};

static const uint8_t bits1[81] = {
    11,  9, 11, 10,  7, 10, 11,  9, 11, 10,  7, 10,  7,  5,  7,  9,
     7, 10, 11,  9, 11,  9,  7,  9, 11,  9, 11,  9,  7,  9,  7,  5,
     7,  9,  7,  9,  7,  5,  7,  5,  1,  5,  7,  5,  7,  9,  7,  9,
     7,  5,  7,  9,  7,  9, 11,  9, 11,  9,  7,  9, 11,  9, 11, 10,
     7,  9,  7,  5,  7,  9,  7, 10, 11,  9, 11, 10,  7,  9, 11,  9,
    11,
};

static const uint16_t codes2[81] = {
    0x1f3, 0x06f, 0x1fd, 0x0eb, 0x023, 0x0ea, 0x1f7, 0x0e8,
    0x1fa, 0x0f2, 0x02d, 0x070, 0x020, 0x006, 0x02b, 0x06e,
    0x028, 0x0e9, 0x1f9, 0x066, 0x0f8, 0x0e7, 0x01b, 0x0f1,
    0x1f4, 0x06b, 0x1f5, 0x0ec, 0x02a, 0x06c, 0x02c, 0x00a,
    0x027, 0x067, 0x01a, 0x0f5, 0x024, 0x008, 0x01f, 0x009,
    0x000, 0x007, 0x01d, 0x00b, 0x030, 0x0ef, 0x01c, 0x064,
    0x01e, 0x00c, 0x029, 0x0f3, 0x02f, 0x0f0, 0x1fc, 0x071,
    0x1f2, 0x0f4, 0x021, 0x0e6, 0x0f7, 0x068, 0x1f8, 0x0ee,
    0x022, 0x065, 0x031, 0x002, 0x026, 0x0ed, 0x025, 0x06a,
    0x1fb, 0x072, 0x1fe, 0x069, 0x02e, 0x0f6, 0x1ff, 0x06d,
    0x1f6,
};

static const uint8_t bits2[81] = {
    9, 7, 9, 8, 6, 8, 9, 8, 9, 8, 6, 7, 6, 5, 6, 7,
    6, 8, 9, 7, 8, 8, 6, 8, 9, 7, 9, 8, 6, 7, 6, 5,
    6, 7, 6, 8, 6, 5, 6, 5, 3, 5, 6, 5, 6, 8, 6, 7,
    6, 5, 6, 8, 6, 8, 9, 7, 9, 8, 6, 8, 8, 7, 9, 8,
    6, 7, 6, 4, 6, 8, 6, 7, 9, 7, 9, 7, 6, 8, 9, 7,
    9,
};

static const uint16_t codes3[81] = {
    0x0000, 0x0009, 0x00ef, 0x000b, 0x0019, 0x00f0, 0x01eb, 0x01e6,
    0x03f2, 0x000a, 0x0035, 0x01ef, 0x0034, 0x0037, 0x01e9, 0x01ed,
    0x01e7, 0x03f3, 0x01ee, 0x03ed, 0x1ffa, 0x01ec, 0x01f2, 0x07f9,
    0x07f8, 0x03f8, 0x0ff8, 0x0008, 0x0038, 0x03f6, 0x0036, 0x0075,
    0x03f1, 0x03eb, 0x03ec, 0x0ff4, 0x0018, 0x0076, 0x07f4, 0x0039,
    0x0074, 0x03ef, 0x01f3, 0x01f4, 0x07f6, 0x01e8, 0x03ea, 0x1ffc,
    0x00f2, 0x01f1, 0x0ffb, 0x03f5, 0x07f3, 0x0ffc, 0x00ee, 0x03f7,
    0x7ffe, 0x01f0, 0x07f5, 0x7ffd, 0x1ffb, 0x3ffa, 0xffff, 0x00f1,
    0x03f0, 0x3ffc, 0x01ea, 0x03ee, 0x3ffb, 0x0ff6, 0x0ffa, 0x7ffc,
    0x07f2, 0x0ff5, 0xfffe, 0x03f4, 0x07f7, 0x7ffb, 0x0ff7, 0x0ff9,
    0x7ffa,
};

static const uint8_t bits3[81] = {
     1,  4,  8,  4,  5,  8,  9,  9, 10,  4,  6,  9,  6,  6,  9,  9,
     9, 10,  9, 10, 13,  9,  9, 11, 11, 10, 12,  4,  6, 10,  6,  7,
    10, 10, 10, 12,  5,  7, 11,  6,  7, 10,  9,  9, 11,  9, 10, 13,
     8,  9, 12, 10, 11, 12,  8, 10, 15,  9, 11, 15, 13, 14, 16,  8,
    10, 14,  9, 10, 14, 12, 12, 15, 11, 12, 16, 10, 11, 15, 12, 12,
    15,
};

static const uint16_t codes4[81] = {
    0x007, 0x016, 0x0f6, 0x018, 0x008, 0x0ef, 0x1ef, 0x0f3,
    0x7f8, 0x019, 0x017, 0x0ed, 0x015, 0x001, 0x0e2, 0x0f0,
    0x070, 0x3f0, 0x1ee, 0x0f1, 0x7fa, 0x0ee, 0x0e4, 0x3f2,
    0x7f6, 0x3ef, 0x7fd, 0x005, 0x014, 0x0f2, 0x009, 0x004,
    0x0e5, 0x0f4, 0x0e8, 0x3f4, 0x006, 0x002, 0x0e7, 0x003,
    0x000, 0x06b, 0x0e3, 0x069, 0x1f3, 0x0eb, 0x0e6, 0x3f6,
    0x06e, 0x06a, 0x1f4, 0x3ec, 0x1f0, 0x3f9, 0x0f5, 0x0ec,
    0x7fb, 0x0ea, 0x06f, 0x3f7, 0x7f9, 0x3f3, 0xfff, 0x0e9,
    0x06d, 0x3f8, 0x06c, 0x068, 0x1f5, 0x3ee, 0x1f2, 0x7f4,
    0x7f7, 0x3f1, 0xffe, 0x3ed, 0x1f1, 0x7f5, 0x7fe, 0x3f5,
    0x7fc,
};

static const uint8_t bits4[81] = {
     4,  5,  8,  5,  4,  8,  9,  8, 11,  5,  5,  8,  5,  4,  8,  8,
     7, 10,  9,  8, 11,  8,  8, 10, 11, 10, 11,  4,  5,  8,  4,  4,
     8,  8,  8, 10,  4,  4,  8,  4,  4,  7,  8,  7,  9,  8,  8, 10,
     7,  7,  9, 10,  9, 10,  8,  8, 11,  8,  7, 10, 11, 10, 12,  8,
     7, 10,  7,  7,  9, 10,  9, 11, 11, 10, 12, 10,  9, 11, 11, 10,
    11,
};

static const uint16_t codes5[81] = {
    0x1fff, 0x0ff7, 0x07f4, 0x07e8, 0x03f1, 0x07ee, 0x07f9, 0x0ff8,
    0x1ffd, 0x0ffd, 0x07f1, 0x03e8, 0x01e8, 0x00f0, 0x01ec, 0x03ee,
    0x07f2, 0x0ffa, 0x0ff4, 0x03ef, 0x01f2, 0x00e8, 0x0070, 0x00ec,
    0x01f0, 0x03ea, 0x07f3, 0x07eb, 0x01eb, 0x00ea, 0x001a, 0x0008,
    0x0019, 0x00ee, 0x01ef, 0x07ed, 0x03f0, 0x00f2, 0x0073, 0x000b,
    0x0000, 0x000a, 0x0071, 0x00f3, 0x07e9, 0x07ef, 0x01ee, 0x00ef,
    0x0018, 0x0009, 0x001b, 0x00eb, 0x01e9, 0x07ec, 0x07f6, 0x03eb,
    0x01f3, 0x00ed, 0x0072, 0x00e9, 0x01f1, 0x03ed, 0x07f7, 0x0ff6,
    0x07f0, 0x03e9, 0x01ed, 0x00f1, 0x01ea, 0x03ec, 0x07f8, 0x0ff9,
    0x1ffc, 0x0ffc, 0x0ff5, 0x07ea, 0x03f3, 0x03f2, 0x07f5, 0x0ffb,
    0x1ffe,
};

static const uint8_t bits5[81] = {
    13, 12, 11, 11, 10, 11, 11, 12, 13, 12, 11, 10,  9,  8,  9, 10,
    11, 12, 12, 10,  9,  8,  7,  8,  9, 10, 11, 11,  9,  8,  5,  4,
     5,  8,  9, 11, 10,  8,  7,  4,  1,  4,  7,  8, 11, 11,  9,  8,
     5,  4,  5,  8,  9, 11, 11, 10,  9,  8,  7,  8,  9, 10, 11, 12,
    11, 10,  9,  8,  9, 10, 11, 12, 13, 12, 12, 11, 10, 10, 11, 12,
    13,
};

static const uint16_t codes6[81] = {
    0x7fe, 0x3fd, 0x1f1, 0x1eb, 0x1f4, 0x1ea, 0x1f0, 0x3fc,
    0x7fd, 0x3f6, 0x1e5, 0x0ea, 0x06c, 0x071, 0x068, 0x0f0,
    0x1e6, 0x3f7, 0x1f3, 0x0ef, 0x032, 0x027, 0x028, 0x026,
    0x031, 0x0eb, 0x1f7, 0x1e8, 0x06f, 0x02e, 0x008, 0x004,
    0x006, 0x029, 0x06b, 0x1ee, 0x1ef, 0x072, 0x02d, 0x002,
    0x000, 0x003, 0x02f, 0x073, 0x1fa, 0x1e7, 0x06e, 0x02b,
    0x007, 0x001, 0x005, 0x02c, 0x06d, 0x1ec, 0x1f9, 0x0ee,
    0x030, 0x024, 0x02a, 0x025, 0x033, 0x0ec, 0x1f2, 0x3f8,
    0x1e4, 0x0ed, 0x06a, 0x070, 0x069, 0x074, 0x0f1, 0x3fa,
    0x7ff, 0x3f9, 0x1f6, 0x1ed, 0x1f8, 0x1e9, 0x1f5, 0x3fb,
    0x7fc,
};

static const uint8_t bits6[81] = {
    11, 10,  9,  9,  9,  9,  9, 10, 11, 10,  9,  8,  7,  7,  7,  8,
     9, 10,  9,  8,  6,  6,  6,  6,  6,  8,  9,  9,  7,  6,  4,  4,
     4,  6,  7,  9,  9,  7,  6,  4,  4,  4,  6,  7,  9,  9,  7,  6,
     4,  4,  4,  6,  7,  9,  9,  8,  6,  6,  6,  6,  6,  8,  9, 10,
     9,  8,  7,  7,  7,  7,  8, 10, 11, 10,  9,  9,  9,  9,  9, 10,
    11,
};

static const uint16_t codes7[64] = {
    0x000, 0x005, 0x037, 0x074, 0x0f2, 0x1eb, 0x3ed, 0x7f7,
    0x004, 0x00c, 0x035, 0x071, 0x0ec, 0x0ee, 0x1ee, 0x1f5,
    0x036, 0x034, 0x072, 0x0ea, 0x0f1, 0x1e9, 0x1f3, 0x3f5,
    0x073, 0x070, 0x0eb, 0x0f0, 0x1f1, 0x1f0, 0x3ec, 0x3fa,
    0x0f3, 0x0ed, 0x1e8, 0x1ef, 0x3ef, 0x3f1, 0x3f9, 0x7fb,
    0x1ed, 0x0ef, 0x1ea, 0x1f2, 0x3f3, 0x3f8, 0x7f9, 0x7fc,
    0x3ee, 0x1ec, 0x1f4, 0x3f4, 0x3f7, 0x7f8, 0xffd, 0xffe,
    0x7f6, 0x3f0, 0x3f2, 0x3f6, 0x7fa, 0x7fd, 0xffc, 0xfff,
};

static const uint8_t bits7[64] = {
     1,  3,  6,  7,  8,  9, 10, 11,  3,  4,  6,  7,  8,  8,  9,  9,
     6,  6,  7,  8,  8,  9,  9, 10,  7,  7,  8,  8,  9,  9, 10, 10,
     8,  8,  9,  9, 10, 10, 10, 11,  9,  8,  9,  9, 10, 10, 11, 11,
    10,  9,  9, 10, 10, 11, 12, 12, 11, 10, 10, 10, 11, 11, 12, 12,
};

static const uint16_t codes8[64] = {
    0x00e, 0x005, 0x010, 0x030, 0x06f, 0x0f1, 0x1fa, 0x3fe,
    0x003, 0x000, 0x004, 0x012, 0x02c, 0x06a, 0x075, 0x0f8,
    0x00f, 0x002, 0x006, 0x014, 0x02e, 0x069, 0x072, 0x0f5,
    0x02f, 0x011, 0x013, 0x02a, 0x032, 0x06c, 0x0ec, 0x0fa,
    0x071, 0x02b, 0x02d, 0x031, 0x06d, 0x070, 0x0f2, 0x1f9,
    0x0ef, 0x068, 0x033, 0x06b, 0x06e, 0x0ee, 0x0f9, 0x3fc,
    0x1f8, 0x074, 0x073, 0x0ed, 0x0f0, 0x0f6, 0x1f6, 0x1fd,
    0x3fd, 0x0f3, 0x0f4, 0x0f7, 0x1f7, 0x1fb, 0x1fc, 0x3ff,
};

static const uint8_t bits8[64] = {
     5,  4,  5,  6,  7,  8,  9, 10,  4,  3,  4,  5,  6,  7,  7,  8,
     5,  4,  4,  5,  6,  7,  7,  8,  6,  5,  5,  6,  6,  7,  8,  8,
     7,  6,  6,  6,  7,  7,  8,  9,  8,  7,  6,  7,  7,  8,  8, 10,
     9,  7,  7,  8,  8,  8,  9,  9, 10,  8,  8,  8,  9,  9,  9, 10,
};

static const uint16_t codes9[169] = {
    0x0000, 0x0005, 0x0037, 0x00e7, 0x01de, 0x03ce, 0x03d9, 0x07c8,
    0x07cd, 0x0fc8, 0x0fdd, 0x1fe4, 0x1fec, 0x0004, 0x000c, 0x0035,
    0x0072, 0x00ea, 0x00ed, 0x01e2, 0x03d1, 0x03d3, 0x03e0, 0x07d8,
    0x0fcf, 0x0fd5, 0x0036, 0x0034, 0x0071, 0x00e8, 0x00ec, 0x01e1,
    0x03cf, 0x03dd, 0x03db, 0x07d0, 0x0fc7, 0x0fd4, 0x0fe4, 0x00e6,
    0x0070, 0x00e9, 0x01dd, 0x01e3, 0x03d2, 0x03dc, 0x07cc, 0x07ca,
    0x07de, 0x0fd8, 0x0fea, 0x1fdb, 0x01df, 0x00eb, 0x01dc, 0x01e6,
    0x03d5, 0x03de, 0x07cb, 0x07dd, 0x07dc, 0x0fcd, 0x0fe2, 0x0fe7,
    0x1fe1, 0x03d0, 0x01e0, 0x01e4, 0x03d6, 0x07c5, 0x07d1, 0x07db,
    0x0fd2, 0x07e0, 0x0fd9, 0x0feb, 0x1fe3, 0x1fe9, 0x07c4, 0x01e5,
    0x03d7, 0x07c6, 0x07cf, 0x07da, 0x0fcb, 0x0fda, 0x0fe3, 0x0fe9,
    0x1fe6, 0x1ff3, 0x1ff7, 0x07d3, 0x03d8, 0x03e1, 0x07d4, 0x07d9,
    0x0fd3, 0x0fde, 0x1fdd, 0x1fd9, 0x1fe2, 0x1fea, 0x1ff1, 0x1ff6,
    0x07d2, 0x03d4, 0x03da, 0x07c7, 0x07d7, 0x07e2, 0x0fce, 0x0fdb,
    0x1fd8, 0x1fee, 0x3ff0, 0x1ff4, 0x3ff2, 0x07e1, 0x03df, 0x07c9,
    0x07d6, 0x0fca, 0x0fd0, 0x0fe5, 0x0fe6, 0x1feb, 0x1fef, 0x3ff3,
    0x3ff4, 0x3ff5, 0x0fe0, 0x07ce, 0x07d5, 0x0fc6, 0x0fd1, 0x0fe1,
    0x1fe0, 0x1fe8, 0x1ff0, 0x3ff1, 0x3ff8, 0x3ff6, 0x7ffc, 0x0fe8,
    0x07df, 0x0fc9, 0x0fd7, 0x0fdc, 0x1fdc, 0x1fdf, 0x1fed, 0x1ff5,
    0x3ff9, 0x3ffb, 0x7ffd, 0x7ffe, 0x1fe7, 0x0fcc, 0x0fd6, 0x0fdf,
    0x1fde, 0x1fda, 0x1fe5, 0x1ff2, 0x3ffa, 0x3ff7, 0x3ffc, 0x3ffd,
    0x7fff,
};

static const uint8_t bits9[169] = {
     1,  3,  6,  8,  9, 10, 10, 11, 11, 12, 12, 13, 13,  3,  4,  6,
     7,  8,  8,  9, 10, 10, 10, 11, 12, 12,  6,  6,  7,  8,  8,  9,
    10, 10, 10, 11, 12, 12, 12,  8,  7,  8,  9,  9, 10, 10, 11, 11,
    11, 12, 12, 13,  9,  8,  9,  9, 10, 10, 11, 11, 11, 12, 12, 12,
    13, 10,  9,  9, 10, 11, 11, 11, 12, 11, 12, 12, 13, 13, 11,  9,
    10, 11, 11, 11, 12, 12, 12, 12, 13, 13, 13, 11, 10, 10, 11, 11,
    12, 12, 13, 13, 13, 13, 13, 13, 11, 10, 10, 11, 11, 11, 12, 12,
    13, 13, 14, 13, 14, 11, 10, 11, 11, 12, 12, 12, 12, 13, 13, 14,
    14, 14, 12, 11, 11, 12, 12, 12, 13, 13, 13, 14, 14, 14, 15, 12,
    11, 12, 12, 12, 13, 13, 13, 13, 14, 14, 15, 15, 13, 12, 12, 12,
    13, 13, 13, 13, 14, 14, 14, 14, 15,
};

static const uint16_t codes10[169] = {
    0x022, 0x008, 0x01d, 0x026, 0x05f, 0x0d3, 0x1cf, 0x3d0,
    0x3d7, 0x3ed, 0x7f0, 0x7f6, 0xffd, 0x007, 0x000, 0x001,
    0x009, 0x020, 0x054, 0x060, 0x0d5, 0x0dc, 0x1d4, 0x3cd,
    0x3de, 0x7e7, 0x01c, 0x002, 0x006, 0x00c, 0x01e, 0x028,
    0x05b, 0x0cd, 0x0d9, 0x1ce, 0x1dc, 0x3d9, 0x3f1, 0x025,
    0x00b, 0x00a, 0x00d, 0x024, 0x057, 0x061, 0x0cc, 0x0dd,
    0x1cc, 0x1de, 0x3d3, 0x3e7, 0x05d, 0x021, 0x01f, 0x023,
    0x027, 0x059, 0x064, 0x0d8, 0x0df, 0x1d2, 0x1e2, 0x3dd,
    0x3ee, 0x0d1, 0x055, 0x029, 0x056, 0x058, 0x062, 0x0ce,
    0x0e0, 0x0e2, 0x1da, 0x3d4, 0x3e3, 0x7eb, 0x1c9, 0x05e,
    0x05a, 0x05c, 0x063, 0x0ca, 0x0da, 0x1c7, 0x1ca, 0x1e0,
    0x3db, 0x3e8, 0x7ec, 0x1e3, 0x0d2, 0x0cb, 0x0d0, 0x0d7,
    0x0db, 0x1c6, 0x1d5, 0x1d8, 0x3ca, 0x3da, 0x7ea, 0x7f1,
    0x1e1, 0x0d4, 0x0cf, 0x0d6, 0x0de, 0x0e1, 0x1d0, 0x1d6,
    0x3d1, 0x3d5, 0x3f2, 0x7ee, 0x7fb, 0x3e9, 0x1cd, 0x1c8,
    0x1cb, 0x1d1, 0x1d7, 0x1df, 0x3cf, 0x3e0, 0x3ef, 0x7e6,
    0x7f8, 0xffa, 0x3eb, 0x1dd, 0x1d3, 0x1d9, 0x1db, 0x3d2,
    0x3cc, 0x3dc, 0x3ea, 0x7ed, 0x7f3, 0x7f9, 0xff9, 0x7f2,
    0x3ce, 0x1e4, 0x3cb, 0x3d8, 0x3d6, 0x3e2, 0x3e5, 0x7e8,
    0x7f4, 0x7f5, 0x7f7, 0xffb, 0x7fa, 0x3ec, 0x3df, 0x3e1,
    0x3e4, 0x3e6, 0x3f0, 0x7e9, 0x7ef, 0xff8, 0xffe, 0xffc,
    0xfff,
};

static const uint8_t bits10[169] = {
     6,  5,  6,  6,  7,  8,  9, 10, 10, 10, 11, 11, 12,  5,  4,  4,
     5,  6,  7,  7,  8,  8,  9, 10, 10, 11,  6,  4,  5,  5,  6,  6,
     7,  8,  8,  9,  9, 10, 10,  6,  5,  5,  5,  6,  7,  7,  8,  8,
     9,  9, 10, 10,  7,  6,  6,  6,  6,  7,  7,  8,  8,  9,  9, 10,
    10,  8,  7,  6,  7,  7,  7,  8,  8,  8,  9, 10, 10, 11,  9,  7,
     7,  7,  7,  8,  8,  9,  9,  9, 10, 10, 11,  9,  8,  8,  8,  8,
     8,  9,  9,  9, 10, 10, 11, 11,  9,  8,  8,  8,  8,  8,  9,  9,
    10, 10, 10, 11, 11, 10,  9,  9,  9,  9,  9,  9, 10, 10, 10, 11,
    11, 12, 10,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11, 12, 11,
    10,  9, 10, 10, 10, 10, 10, 11, 11, 11, 11, 12, 11, 10, 10, 10,
    10, 10, 10, 11, 11, 12, 12, 12, 12,
};

static const uint16_t codes11[289] = {
    0x000, 0x006, 0x019, 0x03d, 0x09c, 0x0c6, 0x1a7, 0x390,
    0x3c2, 0x3df, 0x7e6, 0x7f3, 0xffb, 0x7ec, 0xffa, 0xffe,
    0x38e, 0x005, 0x001, 0x008, 0x014, 0x037, 0x042, 0x092,
    0x0af, 0x191, 0x1a5, 0x1b5, 0x39e, 0x3c0, 0x3a2, 0x3cd,
    0x7d6, 0x0ae, 0x017, 0x007, 0x009, 0x018, 0x039, 0x040,
    0x08e, 0x0a3, 0x0b8, 0x199, 0x1ac, 0x1c1, 0x3b1, 0x396,
    0x3be, 0x3ca, 0x09d, 0x03c, 0x015, 0x016, 0x01a, 0x03b,
    0x044, 0x091, 0x0a5, 0x0be, 0x196, 0x1ae, 0x1b9, 0x3a1,
    0x391, 0x3a5, 0x3d5, 0x094, 0x09a, 0x036, 0x038, 0x03a,
    0x041, 0x08c, 0x09b, 0x0b0, 0x0c3, 0x19e, 0x1ab, 0x1bc,
    0x39f, 0x38f, 0x3a9, 0x3cf, 0x093, 0x0bf, 0x03e, 0x03f,
    0x043, 0x045, 0x09e, 0x0a7, 0x0b9, 0x194, 0x1a2, 0x1ba,
    0x1c3, 0x3a6, 0x3a7, 0x3bb, 0x3d4, 0x09f, 0x1a0, 0x08f,
    0x08d, 0x090, 0x098, 0x0a6, 0x0b6, 0x0c4, 0x19f, 0x1af,
    0x1bf, 0x399, 0x3bf, 0x3b4, 0x3c9, 0x3e7, 0x0a8, 0x1b6,
    0x0ab, 0x0a4, 0x0aa, 0x0b2, 0x0c2, 0x0c5, 0x198, 0x1a4,
    0x1b8, 0x38c, 0x3a4, 0x3c4, 0x3c6, 0x3dd, 0x3e8, 0x0ad,
    0x3af, 0x192, 0x0bd, 0x0bc, 0x18e, 0x197, 0x19a, 0x1a3,
    0x1b1, 0x38d, 0x398, 0x3b7, 0x3d3, 0x3d1, 0x3db, 0x7dd,
    0x0b4, 0x3de, 0x1a9, 0x19b, 0x19c, 0x1a1, 0x1aa, 0x1ad,
    0x1b3, 0x38b, 0x3b2, 0x3b8, 0x3ce, 0x3e1, 0x3e0, 0x7d2,
    0x7e5, 0x0b7, 0x7e3, 0x1bb, 0x1a8, 0x1a6, 0x1b0, 0x1b2,
    0x1b7, 0x39b, 0x39a, 0x3ba, 0x3b5, 0x3d6, 0x7d7, 0x3e4,
    0x7d8, 0x7ea, 0x0ba, 0x7e8, 0x3a0, 0x1bd, 0x1b4, 0x38a,
    0x1c4, 0x392, 0x3aa, 0x3b0, 0x3bc, 0x3d7, 0x7d4, 0x7dc,
    0x7db, 0x7d5, 0x7f0, 0x0c1, 0x7fb, 0x3c8, 0x3a3, 0x395,
    0x39d, 0x3ac, 0x3ae, 0x3c5, 0x3d8, 0x3e2, 0x3e6, 0x7e4,
    0x7e7, 0x7e0, 0x7e9, 0x7f7, 0x190, 0x7f2, 0x393, 0x1be,
    0x1c0, 0x394, 0x397, 0x3ad, 0x3c3, 0x3c1, 0x3d2, 0x7da,
    0x7d9, 0x7df, 0x7eb, 0x7f4, 0x7fa, 0x195, 0x7f8, 0x3bd,
    0x39c, 0x3ab, 0x3a8, 0x3b3, 0x3b9, 0x3d0, 0x3e3, 0x3e5,
    0x7e2, 0x7de, 0x7ed, 0x7f1, 0x7f9, 0x7fc, 0x193, 0xffd,
    0x3dc, 0x3b6, 0x3c7, 0x3cc, 0x3cb, 0x3d9, 0x3da, 0x7d3,
    0x7e1, 0x7ee, 0x7ef, 0x7f5, 0x7f6, 0xffc, 0xfff, 0x19d,
    0x1c2, 0x0b5, 0x0a1, 0x096, 0x097, 0x095, 0x099, 0x0a0,
    0x0a2, 0x0ac, 0x0a9, 0x0b1, 0x0b3, 0x0bb, 0x0c0, 0x18f,
    0x004,
};

static const uint8_t bits11[289] = {
     4,  5,  6,  7,  8,  8,  9, 10, 10, 10, 11, 11, 12, 11, 12, 12,
    10,  5,  4,  5,  6,  7,  7,  8,  8,  9,  9,  9, 10, 10, 10, 10,
    11,  8,  6,  5,  5,  6,  7,  7,  8,  8,  8,  9,  9,  9, 10, 10,
    10, 10,  8,  7,  6,  6,  6,  7,  7,  8,  8,  8,  9,  9,  9, 10,
    10, 10, 10,  8,  8,  7,  7,  7,  7,  8,  8,  8,  8,  9,  9,  9,
    10, 10, 10, 10,  8,  8,  7,  7,  7,  7,  8,  8,  8,  9,  9,  9,
     9, 10, 10, 10, 10,  8,  9,  8,  8,  8,  8,  8,  8,  8,  9,  9,
     9, 10, 10, 10, 10, 10,  8,  9,  8,  8,  8,  8,  8,  8,  9,  9,
     9, 10, 10, 10, 10, 10, 10,  8, 10,  9,  8,  8,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 10, 11,  8, 10,  9,  9,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 10, 11, 11,  8, 11,  9,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 11, 10, 11, 11,  8, 11, 10,  9,  9, 10,
     9, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11,  8, 11, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11,  9, 11, 10,  9,
     9, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11,  9, 11, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11,  9, 12,
    10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 12, 12,  9,
     9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,
     5,
};

const uint16_t * const ff_aac_spectral_codes[11] = {
    codes1,  codes2,  codes3, codes4, codes5, codes6, codes7, codes8,
    codes9, codes10, codes11,
};

const uint8_t * const ff_aac_spectral_bits[11] = {
    bits1,  bits2,  bits3, bits4, bits5, bits6, bits7, bits8,
    bits9, bits10, bits11,
};

const uint16_t ff_aac_spectral_sizes[11] = {
    81, 81, 81, 81, 81, 81, 64, 64, 169, 169, 289,
};

/* NOTE:
 * 64.0f is a special value indicating the existence of an escape code in the
 * bitstream.
 */
static const DECLARE_ALIGNED_16(float, codebook_vector0)[324] = {
 -1.0000000, -1.0000000, -1.0000000, -1.0000000,
 -1.0000000, -1.0000000, -1.0000000,  0.0000000,
 -1.0000000, -1.0000000, -1.0000000,  1.0000000,
 -1.0000000, -1.0000000,  0.0000000, -1.0000000,
 -1.0000000, -1.0000000,  0.0000000,  0.0000000,
 -1.0000000, -1.0000000,  0.0000000,  1.0000000,
 -1.0000000, -1.0000000,  1.0000000, -1.0000000,
 -1.0000000, -1.0000000,  1.0000000,  0.0000000,
 -1.0000000, -1.0000000,  1.0000000,  1.0000000,
 -1.0000000,  0.0000000, -1.0000000, -1.0000000,
 -1.0000000,  0.0000000, -1.0000000,  0.0000000,
 -1.0000000,  0.0000000, -1.0000000,  1.0000000,
 -1.0000000,  0.0000000,  0.0000000, -1.0000000,
 -1.0000000,  0.0000000,  0.0000000,  0.0000000,
 -1.0000000,  0.0000000,  0.0000000,  1.0000000,
 -1.0000000,  0.0000000,  1.0000000, -1.0000000,
 -1.0000000,  0.0000000,  1.0000000,  0.0000000,
 -1.0000000,  0.0000000,  1.0000000,  1.0000000,
 -1.0000000,  1.0000000, -1.0000000, -1.0000000,
 -1.0000000,  1.0000000, -1.0000000,  0.0000000,
 -1.0000000,  1.0000000, -1.0000000,  1.0000000,
 -1.0000000,  1.0000000,  0.0000000, -1.0000000,
 -1.0000000,  1.0000000,  0.0000000,  0.0000000,
 -1.0000000,  1.0000000,  0.0000000,  1.0000000,
 -1.0000000,  1.0000000,  1.0000000, -1.0000000,
 -1.0000000,  1.0000000,  1.0000000,  0.0000000,
 -1.0000000,  1.0000000,  1.0000000,  1.0000000,
  0.0000000, -1.0000000, -1.0000000, -1.0000000,
  0.0000000, -1.0000000, -1.0000000,  0.0000000,
  0.0000000, -1.0000000, -1.0000000,  1.0000000,
  0.0000000, -1.0000000,  0.0000000, -1.0000000,
  0.0000000, -1.0000000,  0.0000000,  0.0000000,
  0.0000000, -1.0000000,  0.0000000,  1.0000000,
  0.0000000, -1.0000000,  1.0000000, -1.0000000,
  0.0000000, -1.0000000,  1.0000000,  0.0000000,
  0.0000000, -1.0000000,  1.0000000,  1.0000000,
  0.0000000,  0.0000000, -1.0000000, -1.0000000,
  0.0000000,  0.0000000, -1.0000000,  0.0000000,
  0.0000000,  0.0000000, -1.0000000,  1.0000000,
  0.0000000,  0.0000000,  0.0000000, -1.0000000,
  0.0000000,  0.0000000,  0.0000000,  0.0000000,
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  0.0000000,  1.0000000, -1.0000000,
  0.0000000,  0.0000000,  1.0000000,  0.0000000,
  0.0000000,  0.0000000,  1.0000000,  1.0000000,
  0.0000000,  1.0000000, -1.0000000, -1.0000000,
  0.0000000,  1.0000000, -1.0000000,  0.0000000,
  0.0000000,  1.0000000, -1.0000000,  1.0000000,
  0.0000000,  1.0000000,  0.0000000, -1.0000000,
  0.0000000,  1.0000000,  0.0000000,  0.0000000,
  0.0000000,  1.0000000,  0.0000000,  1.0000000,
  0.0000000,  1.0000000,  1.0000000, -1.0000000,
  0.0000000,  1.0000000,  1.0000000,  0.0000000,
  0.0000000,  1.0000000,  1.0000000,  1.0000000,
  1.0000000, -1.0000000, -1.0000000, -1.0000000,
  1.0000000, -1.0000000, -1.0000000,  0.0000000,
  1.0000000, -1.0000000, -1.0000000,  1.0000000,
  1.0000000, -1.0000000,  0.0000000, -1.0000000,
  1.0000000, -1.0000000,  0.0000000,  0.0000000,
  1.0000000, -1.0000000,  0.0000000,  1.0000000,
  1.0000000, -1.0000000,  1.0000000, -1.0000000,
  1.0000000, -1.0000000,  1.0000000,  0.0000000,
  1.0000000, -1.0000000,  1.0000000,  1.0000000,
  1.0000000,  0.0000000, -1.0000000, -1.0000000,
  1.0000000,  0.0000000, -1.0000000,  0.0000000,
  1.0000000,  0.0000000, -1.0000000,  1.0000000,
  1.0000000,  0.0000000,  0.0000000, -1.0000000,
  1.0000000,  0.0000000,  0.0000000,  0.0000000,
  1.0000000,  0.0000000,  0.0000000,  1.0000000,
  1.0000000,  0.0000000,  1.0000000, -1.0000000,
  1.0000000,  0.0000000,  1.0000000,  0.0000000,
  1.0000000,  0.0000000,  1.0000000,  1.0000000,
  1.0000000,  1.0000000, -1.0000000, -1.0000000,
  1.0000000,  1.0000000, -1.0000000,  0.0000000,
  1.0000000,  1.0000000, -1.0000000,  1.0000000,
  1.0000000,  1.0000000,  0.0000000, -1.0000000,
  1.0000000,  1.0000000,  0.0000000,  0.0000000,
  1.0000000,  1.0000000,  0.0000000,  1.0000000,
  1.0000000,  1.0000000,  1.0000000, -1.0000000,
  1.0000000,  1.0000000,  1.0000000,  0.0000000,
  1.0000000,  1.0000000,  1.0000000,  1.0000000,
};

static const DECLARE_ALIGNED_16(float, codebook_vector2)[324] = {
  0.0000000,  0.0000000,  0.0000000,  0.0000000,
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  0.0000000,  0.0000000,  2.5198421,
  0.0000000,  0.0000000,  1.0000000,  0.0000000,
  0.0000000,  0.0000000,  1.0000000,  1.0000000,
  0.0000000,  0.0000000,  1.0000000,  2.5198421,
  0.0000000,  0.0000000,  2.5198421,  0.0000000,
  0.0000000,  0.0000000,  2.5198421,  1.0000000,
  0.0000000,  0.0000000,  2.5198421,  2.5198421,
  0.0000000,  1.0000000,  0.0000000,  0.0000000,
  0.0000000,  1.0000000,  0.0000000,  1.0000000,
  0.0000000,  1.0000000,  0.0000000,  2.5198421,
  0.0000000,  1.0000000,  1.0000000,  0.0000000,
  0.0000000,  1.0000000,  1.0000000,  1.0000000,
  0.0000000,  1.0000000,  1.0000000,  2.5198421,
  0.0000000,  1.0000000,  2.5198421,  0.0000000,
  0.0000000,  1.0000000,  2.5198421,  1.0000000,
  0.0000000,  1.0000000,  2.5198421,  2.5198421,
  0.0000000,  2.5198421,  0.0000000,  0.0000000,
  0.0000000,  2.5198421,  0.0000000,  1.0000000,
  0.0000000,  2.5198421,  0.0000000,  2.5198421,
  0.0000000,  2.5198421,  1.0000000,  0.0000000,
  0.0000000,  2.5198421,  1.0000000,  1.0000000,
  0.0000000,  2.5198421,  1.0000000,  2.5198421,
  0.0000000,  2.5198421,  2.5198421,  0.0000000,
  0.0000000,  2.5198421,  2.5198421,  1.0000000,
  0.0000000,  2.5198421,  2.5198421,  2.5198421,
  1.0000000,  0.0000000,  0.0000000,  0.0000000,
  1.0000000,  0.0000000,  0.0000000,  1.0000000,
  1.0000000,  0.0000000,  0.0000000,  2.5198421,
  1.0000000,  0.0000000,  1.0000000,  0.0000000,
  1.0000000,  0.0000000,  1.0000000,  1.0000000,
  1.0000000,  0.0000000,  1.0000000,  2.5198421,
  1.0000000,  0.0000000,  2.5198421,  0.0000000,
  1.0000000,  0.0000000,  2.5198421,  1.0000000,
  1.0000000,  0.0000000,  2.5198421,  2.5198421,
  1.0000000,  1.0000000,  0.0000000,  0.0000000,
  1.0000000,  1.0000000,  0.0000000,  1.0000000,
  1.0000000,  1.0000000,  0.0000000,  2.5198421,
  1.0000000,  1.0000000,  1.0000000,  0.0000000,
  1.0000000,  1.0000000,  1.0000000,  1.0000000,
  1.0000000,  1.0000000,  1.0000000,  2.5198421,
  1.0000000,  1.0000000,  2.5198421,  0.0000000,
  1.0000000,  1.0000000,  2.5198421,  1.0000000,
  1.0000000,  1.0000000,  2.5198421,  2.5198421,
  1.0000000,  2.5198421,  0.0000000,  0.0000000,
  1.0000000,  2.5198421,  0.0000000,  1.0000000,
  1.0000000,  2.5198421,  0.0000000,  2.5198421,
  1.0000000,  2.5198421,  1.0000000,  0.0000000,
  1.0000000,  2.5198421,  1.0000000,  1.0000000,
  1.0000000,  2.5198421,  1.0000000,  2.5198421,
  1.0000000,  2.5198421,  2.5198421,  0.0000000,
  1.0000000,  2.5198421,  2.5198421,  1.0000000,
  1.0000000,  2.5198421,  2.5198421,  2.5198421,
  2.5198421,  0.0000000,  0.0000000,  0.0000000,
  2.5198421,  0.0000000,  0.0000000,  1.0000000,
  2.5198421,  0.0000000,  0.0000000,  2.5198421,
  2.5198421,  0.0000000,  1.0000000,  0.0000000,
  2.5198421,  0.0000000,  1.0000000,  1.0000000,
  2.5198421,  0.0000000,  1.0000000,  2.5198421,
  2.5198421,  0.0000000,  2.5198421,  0.0000000,
  2.5198421,  0.0000000,  2.5198421,  1.0000000,
  2.5198421,  0.0000000,  2.5198421,  2.5198421,
  2.5198421,  1.0000000,  0.0000000,  0.0000000,
  2.5198421,  1.0000000,  0.0000000,  1.0000000,
  2.5198421,  1.0000000,  0.0000000,  2.5198421,
  2.5198421,  1.0000000,  1.0000000,  0.0000000,
  2.5198421,  1.0000000,  1.0000000,  1.0000000,
  2.5198421,  1.0000000,  1.0000000,  2.5198421,
  2.5198421,  1.0000000,  2.5198421,  0.0000000,
  2.5198421,  1.0000000,  2.5198421,  1.0000000,
  2.5198421,  1.0000000,  2.5198421,  2.5198421,
  2.5198421,  2.5198421,  0.0000000,  0.0000000,
  2.5198421,  2.5198421,  0.0000000,  1.0000000,
  2.5198421,  2.5198421,  0.0000000,  2.5198421,
  2.5198421,  2.5198421,  1.0000000,  0.0000000,
  2.5198421,  2.5198421,  1.0000000,  1.0000000,
  2.5198421,  2.5198421,  1.0000000,  2.5198421,
  2.5198421,  2.5198421,  2.5198421,  0.0000000,
  2.5198421,  2.5198421,  2.5198421,  1.0000000,
  2.5198421,  2.5198421,  2.5198421,  2.5198421,
};

static const DECLARE_ALIGNED_16(float, codebook_vector4)[162] = {
 -6.3496042, -6.3496042, -6.3496042, -4.3267487,
 -6.3496042, -2.5198421, -6.3496042, -1.0000000,
 -6.3496042,  0.0000000, -6.3496042,  1.0000000,
 -6.3496042,  2.5198421, -6.3496042,  4.3267487,
 -6.3496042,  6.3496042, -4.3267487, -6.3496042,
 -4.3267487, -4.3267487, -4.3267487, -2.5198421,
 -4.3267487, -1.0000000, -4.3267487,  0.0000000,
 -4.3267487,  1.0000000, -4.3267487,  2.5198421,
 -4.3267487,  4.3267487, -4.3267487,  6.3496042,
 -2.5198421, -6.3496042, -2.5198421, -4.3267487,
 -2.5198421, -2.5198421, -2.5198421, -1.0000000,
 -2.5198421,  0.0000000, -2.5198421,  1.0000000,
 -2.5198421,  2.5198421, -2.5198421,  4.3267487,
 -2.5198421,  6.3496042, -1.0000000, -6.3496042,
 -1.0000000, -4.3267487, -1.0000000, -2.5198421,
 -1.0000000, -1.0000000, -1.0000000,  0.0000000,
 -1.0000000,  1.0000000, -1.0000000,  2.5198421,
 -1.0000000,  4.3267487, -1.0000000,  6.3496042,
  0.0000000, -6.3496042,  0.0000000, -4.3267487,
  0.0000000, -2.5198421,  0.0000000, -1.0000000,
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  2.5198421,  0.0000000,  4.3267487,
  0.0000000,  6.3496042,  1.0000000, -6.3496042,
  1.0000000, -4.3267487,  1.0000000, -2.5198421,
  1.0000000, -1.0000000,  1.0000000,  0.0000000,
  1.0000000,  1.0000000,  1.0000000,  2.5198421,
  1.0000000,  4.3267487,  1.0000000,  6.3496042,
  2.5198421, -6.3496042,  2.5198421, -4.3267487,
  2.5198421, -2.5198421,  2.5198421, -1.0000000,
  2.5198421,  0.0000000,  2.5198421,  1.0000000,
  2.5198421,  2.5198421,  2.5198421,  4.3267487,
  2.5198421,  6.3496042,  4.3267487, -6.3496042,
  4.3267487, -4.3267487,  4.3267487, -2.5198421,
  4.3267487, -1.0000000,  4.3267487,  0.0000000,
  4.3267487,  1.0000000,  4.3267487,  2.5198421,
  4.3267487,  4.3267487,  4.3267487,  6.3496042,
  6.3496042, -6.3496042,  6.3496042, -4.3267487,
  6.3496042, -2.5198421,  6.3496042, -1.0000000,
  6.3496042,  0.0000000,  6.3496042,  1.0000000,
  6.3496042,  2.5198421,  6.3496042,  4.3267487,
  6.3496042,  6.3496042,
};

static const DECLARE_ALIGNED_16(float, codebook_vector6)[128] = {
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  2.5198421,  0.0000000,  4.3267487,
  0.0000000,  6.3496042,  0.0000000,  8.5498797,
  0.0000000, 10.9027236,  0.0000000, 13.3905183,
  1.0000000,  0.0000000,  1.0000000,  1.0000000,
  1.0000000,  2.5198421,  1.0000000,  4.3267487,
  1.0000000,  6.3496042,  1.0000000,  8.5498797,
  1.0000000, 10.9027236,  1.0000000, 13.3905183,
  2.5198421,  0.0000000,  2.5198421,  1.0000000,
  2.5198421,  2.5198421,  2.5198421,  4.3267487,
  2.5198421,  6.3496042,  2.5198421,  8.5498797,
  2.5198421, 10.9027236,  2.5198421, 13.3905183,
  4.3267487,  0.0000000,  4.3267487,  1.0000000,
  4.3267487,  2.5198421,  4.3267487,  4.3267487,
  4.3267487,  6.3496042,  4.3267487,  8.5498797,
  4.3267487, 10.9027236,  4.3267487, 13.3905183,
  6.3496042,  0.0000000,  6.3496042,  1.0000000,
  6.3496042,  2.5198421,  6.3496042,  4.3267487,
  6.3496042,  6.3496042,  6.3496042,  8.5498797,
  6.3496042, 10.9027236,  6.3496042, 13.3905183,
  8.5498797,  0.0000000,  8.5498797,  1.0000000,
  8.5498797,  2.5198421,  8.5498797,  4.3267487,
  8.5498797,  6.3496042,  8.5498797,  8.5498797,
  8.5498797, 10.9027236,  8.5498797, 13.3905183,
 10.9027236,  0.0000000, 10.9027236,  1.0000000,
 10.9027236,  2.5198421, 10.9027236,  4.3267487,
 10.9027236,  6.3496042, 10.9027236,  8.5498797,
 10.9027236, 10.9027236, 10.9027236, 13.3905183,
 13.3905183,  0.0000000, 13.3905183,  1.0000000,
 13.3905183,  2.5198421, 13.3905183,  4.3267487,
 13.3905183,  6.3496042, 13.3905183,  8.5498797,
 13.3905183, 10.9027236, 13.3905183, 13.3905183,
};

static const DECLARE_ALIGNED_16(float, codebook_vector8)[338] = {
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  2.5198421,  0.0000000,  4.3267487,
  0.0000000,  6.3496042,  0.0000000,  8.5498797,
  0.0000000, 10.9027236,  0.0000000, 13.3905183,
  0.0000000, 16.0000000,  0.0000000, 18.7207544,
  0.0000000, 21.5443469,  0.0000000, 24.4637810,
  0.0000000, 27.4731418,  1.0000000,  0.0000000,
  1.0000000,  1.0000000,  1.0000000,  2.5198421,
  1.0000000,  4.3267487,  1.0000000,  6.3496042,
  1.0000000,  8.5498797,  1.0000000, 10.9027236,
  1.0000000, 13.3905183,  1.0000000, 16.0000000,
  1.0000000, 18.7207544,  1.0000000, 21.5443469,
  1.0000000, 24.4637810,  1.0000000, 27.4731418,
  2.5198421,  0.0000000,  2.5198421,  1.0000000,
  2.5198421,  2.5198421,  2.5198421,  4.3267487,
  2.5198421,  6.3496042,  2.5198421,  8.5498797,
  2.5198421, 10.9027236,  2.5198421, 13.3905183,
  2.5198421, 16.0000000,  2.5198421, 18.7207544,
  2.5198421, 21.5443469,  2.5198421, 24.4637810,
  2.5198421, 27.4731418,  4.3267487,  0.0000000,
  4.3267487,  1.0000000,  4.3267487,  2.5198421,
  4.3267487,  4.3267487,  4.3267487,  6.3496042,
  4.3267487,  8.5498797,  4.3267487, 10.9027236,
  4.3267487, 13.3905183,  4.3267487, 16.0000000,
  4.3267487, 18.7207544,  4.3267487, 21.5443469,
  4.3267487, 24.4637810,  4.3267487, 27.4731418,
  6.3496042,  0.0000000,  6.3496042,  1.0000000,
  6.3496042,  2.5198421,  6.3496042,  4.3267487,
  6.3496042,  6.3496042,  6.3496042,  8.5498797,
  6.3496042, 10.9027236,  6.3496042, 13.3905183,
  6.3496042, 16.0000000,  6.3496042, 18.7207544,
  6.3496042, 21.5443469,  6.3496042, 24.4637810,
  6.3496042, 27.4731418,  8.5498797,  0.0000000,
  8.5498797,  1.0000000,  8.5498797,  2.5198421,
  8.5498797,  4.3267487,  8.5498797,  6.3496042,
  8.5498797,  8.5498797,  8.5498797, 10.9027236,
  8.5498797, 13.3905183,  8.5498797, 16.0000000,
  8.5498797, 18.7207544,  8.5498797, 21.5443469,
  8.5498797, 24.4637810,  8.5498797, 27.4731418,
 10.9027236,  0.0000000, 10.9027236,  1.0000000,
 10.9027236,  2.5198421, 10.9027236,  4.3267487,
 10.9027236,  6.3496042, 10.9027236,  8.5498797,
 10.9027236, 10.9027236, 10.9027236, 13.3905183,
 10.9027236, 16.0000000, 10.9027236, 18.7207544,
 10.9027236, 21.5443469, 10.9027236, 24.4637810,
 10.9027236, 27.4731418, 13.3905183,  0.0000000,
 13.3905183,  1.0000000, 13.3905183,  2.5198421,
 13.3905183,  4.3267487, 13.3905183,  6.3496042,
 13.3905183,  8.5498797, 13.3905183, 10.9027236,
 13.3905183, 13.3905183, 13.3905183, 16.0000000,
 13.3905183, 18.7207544, 13.3905183, 21.5443469,
 13.3905183, 24.4637810, 13.3905183, 27.4731418,
 16.0000000,  0.0000000, 16.0000000,  1.0000000,
 16.0000000,  2.5198421, 16.0000000,  4.3267487,
 16.0000000,  6.3496042, 16.0000000,  8.5498797,
 16.0000000, 10.9027236, 16.0000000, 13.3905183,
 16.0000000, 16.0000000, 16.0000000, 18.7207544,
 16.0000000, 21.5443469, 16.0000000, 24.4637810,
 16.0000000, 27.4731418, 18.7207544,  0.0000000,
 18.7207544,  1.0000000, 18.7207544,  2.5198421,
 18.7207544,  4.3267487, 18.7207544,  6.3496042,
 18.7207544,  8.5498797, 18.7207544, 10.9027236,
 18.7207544, 13.3905183, 18.7207544, 16.0000000,
 18.7207544, 18.7207544, 18.7207544, 21.5443469,
 18.7207544, 24.4637810, 18.7207544, 27.4731418,
 21.5443469,  0.0000000, 21.5443469,  1.0000000,
 21.5443469,  2.5198421, 21.5443469,  4.3267487,
 21.5443469,  6.3496042, 21.5443469,  8.5498797,
 21.5443469, 10.9027236, 21.5443469, 13.3905183,
 21.5443469, 16.0000000, 21.5443469, 18.7207544,
 21.5443469, 21.5443469, 21.5443469, 24.4637810,
 21.5443469, 27.4731418, 24.4637810,  0.0000000,
 24.4637810,  1.0000000, 24.4637810,  2.5198421,
 24.4637810,  4.3267487, 24.4637810,  6.3496042,
 24.4637810,  8.5498797, 24.4637810, 10.9027236,
 24.4637810, 13.3905183, 24.4637810, 16.0000000,
 24.4637810, 18.7207544, 24.4637810, 21.5443469,
 24.4637810, 24.4637810, 24.4637810, 27.4731418,
 27.4731418,  0.0000000, 27.4731418,  1.0000000,
 27.4731418,  2.5198421, 27.4731418,  4.3267487,
 27.4731418,  6.3496042, 27.4731418,  8.5498797,
 27.4731418, 10.9027236, 27.4731418, 13.3905183,
 27.4731418, 16.0000000, 27.4731418, 18.7207544,
 27.4731418, 21.5443469, 27.4731418, 24.4637810,
 27.4731418, 27.4731418,
};

static const DECLARE_ALIGNED_16(float, codebook_vector10)[578] = {
  0.0000000,  0.0000000,  0.0000000,  1.0000000,
  0.0000000,  2.5198421,  0.0000000,  4.3267487,
  0.0000000,  6.3496042,  0.0000000,  8.5498797,
  0.0000000, 10.9027236,  0.0000000, 13.3905183,
  0.0000000, 16.0000000,  0.0000000, 18.7207544,
  0.0000000, 21.5443469,  0.0000000, 24.4637810,
  0.0000000, 27.4731418,  0.0000000, 30.5673509,
  0.0000000, 33.7419917,  0.0000000, 36.9931811,
  0.0000000,      64.0f,  1.0000000,  0.0000000,
  1.0000000,  1.0000000,  1.0000000,  2.5198421,
  1.0000000,  4.3267487,  1.0000000,  6.3496042,
  1.0000000,  8.5498797,  1.0000000, 10.9027236,
  1.0000000, 13.3905183,  1.0000000, 16.0000000,
  1.0000000, 18.7207544,  1.0000000, 21.5443469,
  1.0000000, 24.4637810,  1.0000000, 27.4731418,
  1.0000000, 30.5673509,  1.0000000, 33.7419917,
  1.0000000, 36.9931811,  1.0000000,      64.0f,
  2.5198421,  0.0000000,  2.5198421,  1.0000000,
  2.5198421,  2.5198421,  2.5198421,  4.3267487,
  2.5198421,  6.3496042,  2.5198421,  8.5498797,
  2.5198421, 10.9027236,  2.5198421, 13.3905183,
  2.5198421, 16.0000000,  2.5198421, 18.7207544,
  2.5198421, 21.5443469,  2.5198421, 24.4637810,
  2.5198421, 27.4731418,  2.5198421, 30.5673509,
  2.5198421, 33.7419917,  2.5198421, 36.9931811,
  2.5198421,      64.0f,  4.3267487,  0.0000000,
  4.3267487,  1.0000000,  4.3267487,  2.5198421,
  4.3267487,  4.3267487,  4.3267487,  6.3496042,
  4.3267487,  8.5498797,  4.3267487, 10.9027236,
  4.3267487, 13.3905183,  4.3267487, 16.0000000,
  4.3267487, 18.7207544,  4.3267487, 21.5443469,
  4.3267487, 24.4637810,  4.3267487, 27.4731418,
  4.3267487, 30.5673509,  4.3267487, 33.7419917,
  4.3267487, 36.9931811,  4.3267487,      64.0f,
  6.3496042,  0.0000000,  6.3496042,  1.0000000,
  6.3496042,  2.5198421,  6.3496042,  4.3267487,
  6.3496042,  6.3496042,  6.3496042,  8.5498797,
  6.3496042, 10.9027236,  6.3496042, 13.3905183,
  6.3496042, 16.0000000,  6.3496042, 18.7207544,
  6.3496042, 21.5443469,  6.3496042, 24.4637810,
  6.3496042, 27.4731418,  6.3496042, 30.5673509,
  6.3496042, 33.7419917,  6.3496042, 36.9931811,
  6.3496042,      64.0f,  8.5498797,  0.0000000,
  8.5498797,  1.0000000,  8.5498797,  2.5198421,
  8.5498797,  4.3267487,  8.5498797,  6.3496042,
  8.5498797,  8.5498797,  8.5498797, 10.9027236,
  8.5498797, 13.3905183,  8.5498797, 16.0000000,
  8.5498797, 18.7207544,  8.5498797, 21.5443469,
  8.5498797, 24.4637810,  8.5498797, 27.4731418,
  8.5498797, 30.5673509,  8.5498797, 33.7419917,
  8.5498797, 36.9931811,  8.5498797,      64.0f,
 10.9027236,  0.0000000, 10.9027236,  1.0000000,
 10.9027236,  2.5198421, 10.9027236,  4.3267487,
 10.9027236,  6.3496042, 10.9027236,  8.5498797,
 10.9027236, 10.9027236, 10.9027236, 13.3905183,
 10.9027236, 16.0000000, 10.9027236, 18.7207544,
 10.9027236, 21.5443469, 10.9027236, 24.4637810,
 10.9027236, 27.4731418, 10.9027236, 30.5673509,
 10.9027236, 33.7419917, 10.9027236, 36.9931811,
 10.9027236,      64.0f, 13.3905183,  0.0000000,
 13.3905183,  1.0000000, 13.3905183,  2.5198421,
 13.3905183,  4.3267487, 13.3905183,  6.3496042,
 13.3905183,  8.5498797, 13.3905183, 10.9027236,
 13.3905183, 13.3905183, 13.3905183, 16.0000000,
 13.3905183, 18.7207544, 13.3905183, 21.5443469,
 13.3905183, 24.4637810, 13.3905183, 27.4731418,
 13.3905183, 30.5673509, 13.3905183, 33.7419917,
 13.3905183, 36.9931811, 13.3905183,      64.0f,
 16.0000000,  0.0000000, 16.0000000,  1.0000000,
 16.0000000,  2.5198421, 16.0000000,  4.3267487,
 16.0000000,  6.3496042, 16.0000000,  8.5498797,
 16.0000000, 10.9027236, 16.0000000, 13.3905183,
 16.0000000, 16.0000000, 16.0000000, 18.7207544,
 16.0000000, 21.5443469, 16.0000000, 24.4637810,
 16.0000000, 27.4731418, 16.0000000, 30.5673509,
 16.0000000, 33.7419917, 16.0000000, 36.9931811,
 16.0000000,      64.0f, 18.7207544,  0.0000000,
 18.7207544,  1.0000000, 18.7207544,  2.5198421,
 18.7207544,  4.3267487, 18.7207544,  6.3496042,
 18.7207544,  8.5498797, 18.7207544, 10.9027236,
 18.7207544, 13.3905183, 18.7207544, 16.0000000,
 18.7207544, 18.7207544, 18.7207544, 21.5443469,
 18.7207544, 24.4637810, 18.7207544, 27.4731418,
 18.7207544, 30.5673509, 18.7207544, 33.7419917,
 18.7207544, 36.9931811, 18.7207544,      64.0f,
 21.5443469,  0.0000000, 21.5443469,  1.0000000,
 21.5443469,  2.5198421, 21.5443469,  4.3267487,
 21.5443469,  6.3496042, 21.5443469,  8.5498797,
 21.5443469, 10.9027236, 21.5443469, 13.3905183,
 21.5443469, 16.0000000, 21.5443469, 18.7207544,
 21.5443469, 21.5443469, 21.5443469, 24.4637810,
 21.5443469, 27.4731418, 21.5443469, 30.5673509,
 21.5443469, 33.7419917, 21.5443469, 36.9931811,
 21.5443469,      64.0f, 24.4637810,  0.0000000,
 24.4637810,  1.0000000, 24.4637810,  2.5198421,
 24.4637810,  4.3267487, 24.4637810,  6.3496042,
 24.4637810,  8.5498797, 24.4637810, 10.9027236,
 24.4637810, 13.3905183, 24.4637810, 16.0000000,
 24.4637810, 18.7207544, 24.4637810, 21.5443469,
 24.4637810, 24.4637810, 24.4637810, 27.4731418,
 24.4637810, 30.5673509, 24.4637810, 33.7419917,
 24.4637810, 36.9931811, 24.4637810,      64.0f,
 27.4731418,  0.0000000, 27.4731418,  1.0000000,
 27.4731418,  2.5198421, 27.4731418,  4.3267487,
 27.4731418,  6.3496042, 27.4731418,  8.5498797,
 27.4731418, 10.9027236, 27.4731418, 13.3905183,
 27.4731418, 16.0000000, 27.4731418, 18.7207544,
 27.4731418, 21.5443469, 27.4731418, 24.4637810,
 27.4731418, 27.4731418, 27.4731418, 30.5673509,
 27.4731418, 33.7419917, 27.4731418, 36.9931811,
 27.4731418,      64.0f, 30.5673509,  0.0000000,
 30.5673509,  1.0000000, 30.5673509,  2.5198421,
 30.5673509,  4.3267487, 30.5673509,  6.3496042,
 30.5673509,  8.5498797, 30.5673509, 10.9027236,
 30.5673509, 13.3905183, 30.5673509, 16.0000000,
 30.5673509, 18.7207544, 30.5673509, 21.5443469,
 30.5673509, 24.4637810, 30.5673509, 27.4731418,
 30.5673509, 30.5673509, 30.5673509, 33.7419917,
 30.5673509, 36.9931811, 30.5673509,      64.0f,
 33.7419917,  0.0000000, 33.7419917,  1.0000000,
 33.7419917,  2.5198421, 33.7419917,  4.3267487,
 33.7419917,  6.3496042, 33.7419917,  8.5498797,
 33.7419917, 10.9027236, 33.7419917, 13.3905183,
 33.7419917, 16.0000000, 33.7419917, 18.7207544,
 33.7419917, 21.5443469, 33.7419917, 24.4637810,
 33.7419917, 27.4731418, 33.7419917, 30.5673509,
 33.7419917, 33.7419917, 33.7419917, 36.9931811,
 33.7419917,      64.0f, 36.9931811,  0.0000000,
 36.9931811,  1.0000000, 36.9931811,  2.5198421,
 36.9931811,  4.3267487, 36.9931811,  6.3496042,
 36.9931811,  8.5498797, 36.9931811, 10.9027236,
 36.9931811, 13.3905183, 36.9931811, 16.0000000,
 36.9931811, 18.7207544, 36.9931811, 21.5443469,
 36.9931811, 24.4637810, 36.9931811, 27.4731418,
 36.9931811, 30.5673509, 36.9931811, 33.7419917,
 36.9931811, 36.9931811, 36.9931811,      64.0f,
      64.0f,  0.0000000,      64.0f,  1.0000000,
      64.0f,  2.5198421,      64.0f,  4.3267487,
      64.0f,  6.3496042,      64.0f,  8.5498797,
      64.0f, 10.9027236,      64.0f, 13.3905183,
      64.0f, 16.0000000,      64.0f, 18.7207544,
      64.0f, 21.5443469,      64.0f, 24.4637810,
      64.0f, 27.4731418,      64.0f, 30.5673509,
      64.0f, 33.7419917,      64.0f, 36.9931811,
      64.0f,      64.0f,
};

const float * const ff_aac_codebook_vectors[] = {
    codebook_vector0, codebook_vector0, codebook_vector2,
    codebook_vector2, codebook_vector4, codebook_vector4,
    codebook_vector6, codebook_vector6, codebook_vector8,
    codebook_vector8, codebook_vector10,
};

static const float codebook_vector0_vals[] = {
   -1.0000000,  0.0000000,  1.0000000
};

/*
 * bits  0:1, 2:3, 4:5, 6:7  index into _vals array
 *       8:11                number of non-zero values
 *      12:15                bit mask of non-zero values
 */
static const uint16_t codebook_vector02_idx[] = {
    0x0000, 0x8140, 0x8180, 0x4110, 0xc250, 0xc290, 0x4120, 0xc260, 0xc2a0,
    0x2104, 0xa244, 0xa284, 0x6214, 0xe354, 0xe394, 0x6224, 0xe364, 0xe3a4,
    0x2108, 0xa248, 0xa288, 0x6218, 0xe358, 0xe398, 0x6228, 0xe368, 0xe3a8,
    0x1101, 0x9241, 0x9281, 0x5211, 0xd351, 0xd391, 0x5221, 0xd361, 0xd3a1,
    0x3205, 0xb345, 0xb385, 0x7315, 0xf455, 0xf495, 0x7325, 0xf465, 0xf4a5,
    0x3209, 0xb349, 0xb389, 0x7319, 0xf459, 0xf499, 0x7329, 0xf469, 0xf4a9,
    0x1102, 0x9242, 0x9282, 0x5212, 0xd352, 0xd392, 0x5222, 0xd362, 0xd3a2,
    0x3206, 0xb346, 0xb386, 0x7316, 0xf456, 0xf496, 0x7326, 0xf466, 0xf4a6,
    0x320a, 0xb34a, 0xb38a, 0x731a, 0xf45a, 0xf49a, 0x732a, 0xf46a, 0xf4aa,
};

static const float codebook_vector4_vals[] = {
   -6.3496042, -4.3267487,
   -2.5198421, -1.0000000,
    0.0000000,  1.0000000,
    2.5198421,  4.3267487,
    6.3496042,
};

/*
 * bits  0:3, 4:7  index into _vals array
 */
static const uint16_t codebook_vector4_idx[] = {
    0x0000, 0x0010, 0x0020, 0x0030, 0x0040, 0x0050, 0x0060, 0x0070, 0x0080,
    0x0001, 0x0011, 0x0021, 0x0031, 0x0041, 0x0051, 0x0061, 0x0071, 0x0081,
    0x0002, 0x0012, 0x0022, 0x0032, 0x0042, 0x0052, 0x0062, 0x0072, 0x0082,
    0x0003, 0x0013, 0x0023, 0x0033, 0x0043, 0x0053, 0x0063, 0x0073, 0x0083,
    0x0004, 0x0014, 0x0024, 0x0034, 0x0044, 0x0054, 0x0064, 0x0074, 0x0084,
    0x0005, 0x0015, 0x0025, 0x0035, 0x0045, 0x0055, 0x0065, 0x0075, 0x0085,
    0x0006, 0x0016, 0x0026, 0x0036, 0x0046, 0x0056, 0x0066, 0x0076, 0x0086,
    0x0007, 0x0017, 0x0027, 0x0037, 0x0047, 0x0057, 0x0067, 0x0077, 0x0087,
    0x0008, 0x0018, 0x0028, 0x0038, 0x0048, 0x0058, 0x0068, 0x0078, 0x0088,
};

/*
 * bits  0:3, 4:7  index into _vals array
 *       8:11      number of non-zero values
 *      12:15      1: only second value non-zero
 *                 0: other cases
 */
static const uint16_t codebook_vector6_idx[] = {
    0x0000, 0x0110, 0x0120, 0x0130, 0x0140, 0x0150, 0x0160, 0x0170,
    0x1101, 0x0211, 0x0221, 0x0231, 0x0241, 0x0251, 0x0261, 0x0271,
    0x1102, 0x0212, 0x0222, 0x0232, 0x0242, 0x0252, 0x0262, 0x0272,
    0x1103, 0x0213, 0x0223, 0x0233, 0x0243, 0x0253, 0x0263, 0x0273,
    0x1104, 0x0214, 0x0224, 0x0234, 0x0244, 0x0254, 0x0264, 0x0274,
    0x1105, 0x0215, 0x0225, 0x0235, 0x0245, 0x0255, 0x0265, 0x0275,
    0x1106, 0x0216, 0x0226, 0x0236, 0x0246, 0x0256, 0x0266, 0x0276,
    0x1107, 0x0217, 0x0227, 0x0237, 0x0247, 0x0257, 0x0267, 0x0277,
};

/*
 * bits  0:3, 4:7  index into _vals array
 *       8:11      number of non-zero values
 *      12:15      1: only second value non-zero
 *                 0: other cases
 */
static const uint16_t codebook_vector8_idx[] = {
  0x0000, 0x0110, 0x0120, 0x0130, 0x0140, 0x0150, 0x0160,
  0x0170, 0x0180, 0x0190, 0x01a0, 0x01b0, 0x01c0,
  0x1101, 0x0211, 0x0221, 0x0231, 0x0241, 0x0251, 0x0261,
  0x0271, 0x0281, 0x0291, 0x02a1, 0x02b1, 0x02c1,
  0x1102, 0x0212, 0x0222, 0x0232, 0x0242, 0x0252, 0x0262,
  0x0272, 0x0282, 0x0292, 0x02a2, 0x02b2, 0x02c2,
  0x1103, 0x0213, 0x0223, 0x0233, 0x0243, 0x0253, 0x0263,
  0x0273, 0x0283, 0x0293, 0x02a3, 0x02b3, 0x02c3,
  0x1104, 0x0214, 0x0224, 0x0234, 0x0244, 0x0254, 0x0264,
  0x0274, 0x0284, 0x0294, 0x02a4, 0x02b4, 0x02c4,
  0x1105, 0x0215, 0x0225, 0x0235, 0x0245, 0x0255, 0x0265,
  0x0275, 0x0285, 0x0295, 0x02a5, 0x02b5, 0x02c5,
  0x1106, 0x0216, 0x0226, 0x0236, 0x0246, 0x0256, 0x0266,
  0x0276, 0x0286, 0x0296, 0x02a6, 0x02b6, 0x02c6,
  0x1107, 0x0217, 0x0227, 0x0237, 0x0247, 0x0257, 0x0267,
  0x0277, 0x0287, 0x0297, 0x02a7, 0x02b7, 0x02c7,
  0x1108, 0x0218, 0x0228, 0x0238, 0x0248, 0x0258, 0x0268,
  0x0278, 0x0288, 0x0298, 0x02a8, 0x02b8, 0x02c8,
  0x1109, 0x0219, 0x0229, 0x0239, 0x0249, 0x0259, 0x0269,
  0x0279, 0x0289, 0x0299, 0x02a9, 0x02b9, 0x02c9,
  0x110a, 0x021a, 0x022a, 0x023a, 0x024a, 0x025a, 0x026a,
  0x027a, 0x028a, 0x029a, 0x02aa, 0x02ba, 0x02ca,
  0x110b, 0x021b, 0x022b, 0x023b, 0x024b, 0x025b, 0x026b,
  0x027b, 0x028b, 0x029b, 0x02ab, 0x02bb, 0x02cb,
  0x110c, 0x021c, 0x022c, 0x023c, 0x024c, 0x025c, 0x026c,
  0x027c, 0x028c, 0x029c, 0x02ac, 0x02bc, 0x02cc,
};

static const float codebook_vector10_vals[] = {
     0.0000000,  1.0000000,
     2.5198421,  4.3267487,
     6.3496042,  8.5498797,
    10.9027236, 13.3905183,
    16.0000000, 18.7207544,
    21.5443469, 24.4637810,
    27.4731418, 30.5673509,
    33.7419917, 36.9931811,
};

/*
 * bits  0:3, 4:7  index into _vals array
 *       8:9       bit mask of escape-coded entries
 *      12:15      number of non-zero values
 */
static const uint16_t codebook_vector10_idx[] = {
    0x0000, 0x1010, 0x1020, 0x1030, 0x1040, 0x1050, 0x1060, 0x1070,
    0x1080, 0x1090, 0x10a0, 0x10b0, 0x10c0, 0x10d0, 0x10e0, 0x10f0, 0x1200,
    0x1001, 0x2011, 0x2021, 0x2031, 0x2041, 0x2051, 0x2061, 0x2071,
    0x2081, 0x2091, 0x20a1, 0x20b1, 0x20c1, 0x20d1, 0x20e1, 0x20f1, 0x2201,
    0x1002, 0x2012, 0x2022, 0x2032, 0x2042, 0x2052, 0x2062, 0x2072,
    0x2082, 0x2092, 0x20a2, 0x20b2, 0x20c2, 0x20d2, 0x20e2, 0x20f2, 0x2202,
    0x1003, 0x2013, 0x2023, 0x2033, 0x2043, 0x2053, 0x2063, 0x2073,
    0x2083, 0x2093, 0x20a3, 0x20b3, 0x20c3, 0x20d3, 0x20e3, 0x20f3, 0x2203,
    0x1004, 0x2014, 0x2024, 0x2034, 0x2044, 0x2054, 0x2064, 0x2074,
    0x2084, 0x2094, 0x20a4, 0x20b4, 0x20c4, 0x20d4, 0x20e4, 0x20f4, 0x2204,
    0x1005, 0x2015, 0x2025, 0x2035, 0x2045, 0x2055, 0x2065, 0x2075,
    0x2085, 0x2095, 0x20a5, 0x20b5, 0x20c5, 0x20d5, 0x20e5, 0x20f5, 0x2205,
    0x1006, 0x2016, 0x2026, 0x2036, 0x2046, 0x2056, 0x2066, 0x2076,
    0x2086, 0x2096, 0x20a6, 0x20b6, 0x20c6, 0x20d6, 0x20e6, 0x20f6, 0x2206,
    0x1007, 0x2017, 0x2027, 0x2037, 0x2047, 0x2057, 0x2067, 0x2077,
    0x2087, 0x2097, 0x20a7, 0x20b7, 0x20c7, 0x20d7, 0x20e7, 0x20f7, 0x2207,
    0x1008, 0x2018, 0x2028, 0x2038, 0x2048, 0x2058, 0x2068, 0x2078,
    0x2088, 0x2098, 0x20a8, 0x20b8, 0x20c8, 0x20d8, 0x20e8, 0x20f8, 0x2208,
    0x1009, 0x2019, 0x2029, 0x2039, 0x2049, 0x2059, 0x2069, 0x2079,
    0x2089, 0x2099, 0x20a9, 0x20b9, 0x20c9, 0x20d9, 0x20e9, 0x20f9, 0x2209,
    0x100a, 0x201a, 0x202a, 0x203a, 0x204a, 0x205a, 0x206a, 0x207a,
    0x208a, 0x209a, 0x20aa, 0x20ba, 0x20ca, 0x20da, 0x20ea, 0x20fa, 0x220a,
    0x100b, 0x201b, 0x202b, 0x203b, 0x204b, 0x205b, 0x206b, 0x207b,
    0x208b, 0x209b, 0x20ab, 0x20bb, 0x20cb, 0x20db, 0x20eb, 0x20fb, 0x220b,
    0x100c, 0x201c, 0x202c, 0x203c, 0x204c, 0x205c, 0x206c, 0x207c,
    0x208c, 0x209c, 0x20ac, 0x20bc, 0x20cc, 0x20dc, 0x20ec, 0x20fc, 0x220c,
    0x100d, 0x201d, 0x202d, 0x203d, 0x204d, 0x205d, 0x206d, 0x207d,
    0x208d, 0x209d, 0x20ad, 0x20bd, 0x20cd, 0x20dd, 0x20ed, 0x20fd, 0x220d,
    0x100e, 0x201e, 0x202e, 0x203e, 0x204e, 0x205e, 0x206e, 0x207e,
    0x208e, 0x209e, 0x20ae, 0x20be, 0x20ce, 0x20de, 0x20ee, 0x20fe, 0x220e,
    0x100f, 0x201f, 0x202f, 0x203f, 0x204f, 0x205f, 0x206f, 0x207f,
    0x208f, 0x209f, 0x20af, 0x20bf, 0x20cf, 0x20df, 0x20ef, 0x20ff, 0x220f,
    0x1100, 0x2110, 0x2120, 0x2130, 0x2140, 0x2150, 0x2160, 0x2170,
    0x2180, 0x2190, 0x21a0, 0x21b0, 0x21c0, 0x21d0, 0x21e0, 0x21f0, 0x2300,
};

const float *const ff_aac_codebook_vector_vals[] = {
    codebook_vector0_vals,  codebook_vector0_vals,
    codebook_vector10_vals, codebook_vector10_vals,
    codebook_vector4_vals,  codebook_vector4_vals,
    codebook_vector10_vals, codebook_vector10_vals,
    codebook_vector10_vals, codebook_vector10_vals,
    codebook_vector10_vals,
};

const uint16_t *const ff_aac_codebook_vector_idx[] = {
    codebook_vector02_idx, codebook_vector02_idx,
    codebook_vector02_idx, codebook_vector02_idx,
    codebook_vector4_idx,  codebook_vector4_idx,
    codebook_vector6_idx,  codebook_vector6_idx,
    codebook_vector8_idx,  codebook_vector8_idx,
    codebook_vector10_idx,
};

/* @name swb_offsets
 * Sample offset into the window indicating the beginning of a scalefactor
 * window band
 *
 * scalefactor window band - term for scalefactor bands within a window,
 * given in Table 4.110 to Table 4.128.
 *
 * scalefactor band - a set of spectral coefficients which are scaled by one
 * scalefactor. In case of EIGHT_SHORT_SEQUENCE and grouping a scalefactor band
 * may contain several scalefactor window bands of corresponding frequency. For
 * all other window_sequences scalefactor bands and scalefactor window bands are
 * identical.
 * @{
 */

static const uint16_t swb_offset_1024_96[] = {
      0,   4,   8,  12,  16,  20,  24,  28,
     32,  36,  40,  44,  48,  52,  56,  64,
     72,  80,  88,  96, 108, 120, 132, 144,
    156, 172, 188, 212, 240, 276, 320, 384,
    448, 512, 576, 640, 704, 768, 832, 896,
    960, 1024
};

static const uint16_t swb_offset_128_96[] = {
    0, 4, 8, 12, 16, 20, 24, 32, 40, 48, 64, 92, 128
};

static const uint16_t swb_offset_1024_64[] = {
      0,   4,   8,  12,  16,  20,  24,  28,
     32,  36,  40,  44,  48,  52,  56,  64,
     72,  80,  88, 100, 112, 124, 140, 156,
    172, 192, 216, 240, 268, 304, 344, 384,
    424, 464, 504, 544, 584, 624, 664, 704,
    744, 784, 824, 864, 904, 944, 984, 1024
};

static const uint16_t swb_offset_1024_48[] = {
      0,   4,   8,  12,  16,  20,  24,  28,
     32,  36,  40,  48,  56,  64,  72,  80,
     88,  96, 108, 120, 132, 144, 160, 176,
    196, 216, 240, 264, 292, 320, 352, 384,
    416, 448, 480, 512, 544, 576, 608, 640,
    672, 704, 736, 768, 800, 832, 864, 896,
    928, 1024
};

static const uint16_t swb_offset_128_48[] = {
     0,   4,   8,  12,  16,  20,  28,  36,
    44,  56,  68,  80,  96, 112, 128
};

static const uint16_t swb_offset_1024_32[] = {
      0,   4,   8,  12,  16,  20,  24,  28,
     32,  36,  40,  48,  56,  64,  72,  80,
     88,  96, 108, 120, 132, 144, 160, 176,
    196, 216, 240, 264, 292, 320, 352, 384,
    416, 448, 480, 512, 544, 576, 608, 640,
    672, 704, 736, 768, 800, 832, 864, 896,
    928, 960, 992, 1024
};

static const uint16_t swb_offset_1024_24[] = {
      0,   4,   8,  12,  16,  20,  24,  28,
     32,  36,  40,  44,  52,  60,  68,  76,
     84,  92, 100, 108, 116, 124, 136, 148,
    160, 172, 188, 204, 220, 240, 260, 284,
    308, 336, 364, 396, 432, 468, 508, 552,
    600, 652, 704, 768, 832, 896, 960, 1024
};

static const uint16_t swb_offset_128_24[] = {
     0,   4,   8,  12,  16,  20,  24,  28,
    36,  44,  52,  64,  76,  92, 108, 128
};

static const uint16_t swb_offset_1024_16[] = {
      0,   8,  16,  24,  32,  40,  48,  56,
     64,  72,  80,  88, 100, 112, 124, 136,
    148, 160, 172, 184, 196, 212, 228, 244,
    260, 280, 300, 320, 344, 368, 396, 424,
    456, 492, 532, 572, 616, 664, 716, 772,
    832, 896, 960, 1024
};

static const uint16_t swb_offset_128_16[] = {
     0,   4,   8,  12,  16,  20,  24,  28,
    32,  40,  48,  60,  72,  88, 108, 128
};

static const uint16_t swb_offset_1024_8[] = {
      0,  12,  24,  36,  48,  60,  72,  84,
     96, 108, 120, 132, 144, 156, 172, 188,
    204, 220, 236, 252, 268, 288, 308, 328,
    348, 372, 396, 420, 448, 476, 508, 544,
    580, 620, 664, 712, 764, 820, 880, 944,
    1024
};

static const uint16_t swb_offset_128_8[] = {
     0,   4,   8,  12,  16,  20,  24,  28,
    36,  44,  52,  60,  72,  88, 108, 128
};

const uint16_t * const ff_swb_offset_1024[] = {
    swb_offset_1024_96, swb_offset_1024_96, swb_offset_1024_64,
    swb_offset_1024_48, swb_offset_1024_48, swb_offset_1024_32,
    swb_offset_1024_24, swb_offset_1024_24, swb_offset_1024_16,
    swb_offset_1024_16, swb_offset_1024_16, swb_offset_1024_8,
    swb_offset_1024_8
};

const uint16_t * const ff_swb_offset_128[] = {
    /* The last entry on the following row is swb_offset_128_64 but is a
       duplicate of swb_offset_128_96. */
    swb_offset_128_96, swb_offset_128_96, swb_offset_128_96,
    swb_offset_128_48, swb_offset_128_48, swb_offset_128_48,
    swb_offset_128_24, swb_offset_128_24, swb_offset_128_16,
    swb_offset_128_16, swb_offset_128_16, swb_offset_128_8,
    swb_offset_128_8
};

// @}

/* @name ff_tns_max_bands
 * The maximum number of scalefactor bands on which TNS can operate for the long
 * and short transforms respectively. The index to these tables is related to
 * the sample rate of the audio.
 * @{
 */
const uint8_t ff_tns_max_bands_1024[] = {
    31, 31, 34, 40, 42, 51, 46, 46, 42, 42, 42, 39, 39
};

const uint8_t ff_tns_max_bands_128[] = {
    9, 9, 10, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14
};
// @}


#if CONFIG_HARDCODED_TABLES

/**
 * Table of pow(2, (i - 200)/4.) used for different purposes depending on the
 * range of indices to the table:
 * [ 0, 255] scale factor decoding when using C dsp.float_to_int16
 * [60, 315] scale factor decoding when using SIMD dsp.float_to_int16
 * [45, 300] intensity stereo position decoding mapped in reverse order i.e. 0->300, 1->299, ..., 254->46, 255->45
 */
const float ff_aac_pow2sf_tab[428] = {
    8.88178420e-16, 1.05622810e-15, 1.25607397e-15, 1.49373210e-15,
    1.77635684e-15, 2.11245619e-15, 2.51214793e-15, 2.98746420e-15,
    3.55271368e-15, 4.22491238e-15, 5.02429587e-15, 5.97492839e-15,
    7.10542736e-15, 8.44982477e-15, 1.00485917e-14, 1.19498568e-14,
    1.42108547e-14, 1.68996495e-14, 2.00971835e-14, 2.38997136e-14,
    2.84217094e-14, 3.37992991e-14, 4.01943669e-14, 4.77994272e-14,
    5.68434189e-14, 6.75985982e-14, 8.03887339e-14, 9.55988543e-14,
    1.13686838e-13, 1.35197196e-13, 1.60777468e-13, 1.91197709e-13,
    2.27373675e-13, 2.70394393e-13, 3.21554936e-13, 3.82395417e-13,
    4.54747351e-13, 5.40788785e-13, 6.43109871e-13, 7.64790834e-13,
    9.09494702e-13, 1.08157757e-12, 1.28621974e-12, 1.52958167e-12,
    1.81898940e-12, 2.16315514e-12, 2.57243948e-12, 3.05916334e-12,
    3.63797881e-12, 4.32631028e-12, 5.14487897e-12, 6.11832668e-12,
    7.27595761e-12, 8.65262056e-12, 1.02897579e-11, 1.22366534e-11,
    1.45519152e-11, 1.73052411e-11, 2.05795159e-11, 2.44733067e-11,
    2.91038305e-11, 3.46104823e-11, 4.11590317e-11, 4.89466134e-11,
    5.82076609e-11, 6.92209645e-11, 8.23180635e-11, 9.78932268e-11,
    1.16415322e-10, 1.38441929e-10, 1.64636127e-10, 1.95786454e-10,
    2.32830644e-10, 2.76883858e-10, 3.29272254e-10, 3.91572907e-10,
    4.65661287e-10, 5.53767716e-10, 6.58544508e-10, 7.83145814e-10,
    9.31322575e-10, 1.10753543e-09, 1.31708902e-09, 1.56629163e-09,
    1.86264515e-09, 2.21507086e-09, 2.63417803e-09, 3.13258326e-09,
    3.72529030e-09, 4.43014173e-09, 5.26835606e-09, 6.26516652e-09,
    7.45058060e-09, 8.86028346e-09, 1.05367121e-08, 1.25303330e-08,
    1.49011612e-08, 1.77205669e-08, 2.10734243e-08, 2.50606661e-08,
    2.98023224e-08, 3.54411338e-08, 4.21468485e-08, 5.01213321e-08,
    5.96046448e-08, 7.08822677e-08, 8.42936970e-08, 1.00242664e-07,
    1.19209290e-07, 1.41764535e-07, 1.68587394e-07, 2.00485328e-07,
    2.38418579e-07, 2.83529071e-07, 3.37174788e-07, 4.00970657e-07,
    4.76837158e-07, 5.67058141e-07, 6.74349576e-07, 8.01941314e-07,
    9.53674316e-07, 1.13411628e-06, 1.34869915e-06, 1.60388263e-06,
    1.90734863e-06, 2.26823256e-06, 2.69739830e-06, 3.20776526e-06,
    3.81469727e-06, 4.53646513e-06, 5.39479661e-06, 6.41553051e-06,
    7.62939453e-06, 9.07293026e-06, 1.07895932e-05, 1.28310610e-05,
    1.52587891e-05, 1.81458605e-05, 2.15791864e-05, 2.56621220e-05,
    3.05175781e-05, 3.62917210e-05, 4.31583729e-05, 5.13242441e-05,
    6.10351562e-05, 7.25834421e-05, 8.63167458e-05, 1.02648488e-04,
    1.22070312e-04, 1.45166884e-04, 1.72633492e-04, 2.05296976e-04,
    2.44140625e-04, 2.90333768e-04, 3.45266983e-04, 4.10593953e-04,
    4.88281250e-04, 5.80667537e-04, 6.90533966e-04, 8.21187906e-04,
    9.76562500e-04, 1.16133507e-03, 1.38106793e-03, 1.64237581e-03,
    1.95312500e-03, 2.32267015e-03, 2.76213586e-03, 3.28475162e-03,
    3.90625000e-03, 4.64534029e-03, 5.52427173e-03, 6.56950324e-03,
    7.81250000e-03, 9.29068059e-03, 1.10485435e-02, 1.31390065e-02,
    1.56250000e-02, 1.85813612e-02, 2.20970869e-02, 2.62780130e-02,
    3.12500000e-02, 3.71627223e-02, 4.41941738e-02, 5.25560260e-02,
    6.25000000e-02, 7.43254447e-02, 8.83883476e-02, 1.05112052e-01,
    1.25000000e-01, 1.48650889e-01, 1.76776695e-01, 2.10224104e-01,
    2.50000000e-01, 2.97301779e-01, 3.53553391e-01, 4.20448208e-01,
    5.00000000e-01, 5.94603558e-01, 7.07106781e-01, 8.40896415e-01,
    1.00000000e+00, 1.18920712e+00, 1.41421356e+00, 1.68179283e+00,
    2.00000000e+00, 2.37841423e+00, 2.82842712e+00, 3.36358566e+00,
    4.00000000e+00, 4.75682846e+00, 5.65685425e+00, 6.72717132e+00,
    8.00000000e+00, 9.51365692e+00, 1.13137085e+01, 1.34543426e+01,
    1.60000000e+01, 1.90273138e+01, 2.26274170e+01, 2.69086853e+01,
    3.20000000e+01, 3.80546277e+01, 4.52548340e+01, 5.38173706e+01,
    6.40000000e+01, 7.61092554e+01, 9.05096680e+01, 1.07634741e+02,
    1.28000000e+02, 1.52218511e+02, 1.81019336e+02, 2.15269482e+02,
    2.56000000e+02, 3.04437021e+02, 3.62038672e+02, 4.30538965e+02,
    5.12000000e+02, 6.08874043e+02, 7.24077344e+02, 8.61077929e+02,
    1.02400000e+03, 1.21774809e+03, 1.44815469e+03, 1.72215586e+03,
    2.04800000e+03, 2.43549617e+03, 2.89630938e+03, 3.44431172e+03,
    4.09600000e+03, 4.87099234e+03, 5.79261875e+03, 6.88862343e+03,
    8.19200000e+03, 9.74198469e+03, 1.15852375e+04, 1.37772469e+04,
    1.63840000e+04, 1.94839694e+04, 2.31704750e+04, 2.75544937e+04,
    3.27680000e+04, 3.89679387e+04, 4.63409500e+04, 5.51089875e+04,
    6.55360000e+04, 7.79358775e+04, 9.26819000e+04, 1.10217975e+05,
    1.31072000e+05, 1.55871755e+05, 1.85363800e+05, 2.20435950e+05,
    2.62144000e+05, 3.11743510e+05, 3.70727600e+05, 4.40871900e+05,
    5.24288000e+05, 6.23487020e+05, 7.41455200e+05, 8.81743800e+05,
    1.04857600e+06, 1.24697404e+06, 1.48291040e+06, 1.76348760e+06,
    2.09715200e+06, 2.49394808e+06, 2.96582080e+06, 3.52697520e+06,
    4.19430400e+06, 4.98789616e+06, 5.93164160e+06, 7.05395040e+06,
    8.38860800e+06, 9.97579232e+06, 1.18632832e+07, 1.41079008e+07,
    1.67772160e+07, 1.99515846e+07, 2.37265664e+07, 2.82158016e+07,
    3.35544320e+07, 3.99031693e+07, 4.74531328e+07, 5.64316032e+07,
    6.71088640e+07, 7.98063385e+07, 9.49062656e+07, 1.12863206e+08,
    1.34217728e+08, 1.59612677e+08, 1.89812531e+08, 2.25726413e+08,
    2.68435456e+08, 3.19225354e+08, 3.79625062e+08, 4.51452825e+08,
    5.36870912e+08, 6.38450708e+08, 7.59250125e+08, 9.02905651e+08,
    1.07374182e+09, 1.27690142e+09, 1.51850025e+09, 1.80581130e+09,
    2.14748365e+09, 2.55380283e+09, 3.03700050e+09, 3.61162260e+09,
    4.29496730e+09, 5.10760567e+09, 6.07400100e+09, 7.22324521e+09,
    8.58993459e+09, 1.02152113e+10, 1.21480020e+10, 1.44464904e+10,
    1.71798692e+10, 2.04304227e+10, 2.42960040e+10, 2.88929808e+10,
    3.43597384e+10, 4.08608453e+10, 4.85920080e+10, 5.77859616e+10,
    6.87194767e+10, 8.17216907e+10, 9.71840160e+10, 1.15571923e+11,
    1.37438953e+11, 1.63443381e+11, 1.94368032e+11, 2.31143847e+11,
    2.74877907e+11, 3.26886763e+11, 3.88736064e+11, 4.62287693e+11,
    5.49755814e+11, 6.53773525e+11, 7.77472128e+11, 9.24575386e+11,
    1.09951163e+12, 1.30754705e+12, 1.55494426e+12, 1.84915077e+12,
    2.19902326e+12, 2.61509410e+12, 3.10988851e+12, 3.69830155e+12,
    4.39804651e+12, 5.23018820e+12, 6.21977702e+12, 7.39660309e+12,
    8.79609302e+12, 1.04603764e+13, 1.24395540e+13, 1.47932062e+13,
    1.75921860e+13, 2.09207528e+13, 2.48791081e+13, 2.95864124e+13,
    3.51843721e+13, 4.18415056e+13, 4.97582162e+13, 5.91728247e+13,
    7.03687442e+13, 8.36830112e+13, 9.95164324e+13, 1.18345649e+14,
    1.40737488e+14, 1.67366022e+14, 1.99032865e+14, 2.36691299e+14,
    2.81474977e+14, 3.34732045e+14, 3.98065730e+14, 4.73382598e+14,
    5.62949953e+14, 6.69464090e+14, 7.96131459e+14, 9.46765196e+14,
    1.12589991e+15, 1.33892818e+15, 1.59226292e+15, 1.89353039e+15,
    2.25179981e+15, 2.67785636e+15, 3.18452584e+15, 3.78706078e+15,
    4.50359963e+15, 5.35571272e+15, 6.36905167e+15, 7.57412156e+15,
    9.00719925e+15, 1.07114254e+16, 1.27381033e+16, 1.51482431e+16,
    1.80143985e+16, 2.14228509e+16, 2.54762067e+16, 3.02964863e+16,
    3.60287970e+16, 4.28457018e+16, 5.09524134e+16, 6.05929725e+16,
    7.20575940e+16, 8.56914035e+16, 1.01904827e+17, 1.21185945e+17,
};

#else

float ff_aac_pow2sf_tab[428];

#endif /* CONFIG_HARDCODED_TABLES */
