/*
 * Floating point AAN DCT
 * Copyright (c) 2003 Michael Niedermayer <michaelni@gmx.at>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * this implementation is based upon the IJG integer AAN DCT (see jfdctfst.c)
 */

/**
 * @file faandct.c
 * @brief 
 *     Floating point AAN DCT
 * @author Michael Niedermayer <michaelni@gmx.at>
 */

#include <math.h>

#include "dsputil.h"
#include "faandct.h"

#define FLOAT float

//numbers generated by simple c code (not as accurate as they could be)
/*
for(i=0; i<8; i++){
    printf("#define B%d %1.20llf\n", i, (long double)1.0/(cosl(i*acosl(-1.0)/(long double)16.0)*sqrtl(2)));
}
*/
#define B0 1.00000000000000000000
#define B1 0.72095982200694791383 // (cos(pi*1/16)sqrt(2))^-1
#define B2 0.76536686473017954350 // (cos(pi*2/16)sqrt(2))^-1
#define B3 0.85043009476725644878 // (cos(pi*3/16)sqrt(2))^-1
#define B4 1.00000000000000000000 // (cos(pi*4/16)sqrt(2))^-1
#define B5 1.27275858057283393842 // (cos(pi*5/16)sqrt(2))^-1
#define B6 1.84775906502257351242 // (cos(pi*6/16)sqrt(2))^-1
#define B7 3.62450978541155137218 // (cos(pi*7/16)sqrt(2))^-1


#define A1 0.70710678118654752438 // cos(pi*4/16)
#define A2 0.54119610014619698435 // cos(pi*6/16)sqrt(2)
#define A5 0.38268343236508977170 // cos(pi*6/16)
#define A4 1.30656296487637652774 // cos(pi*2/16)sqrt(2)

static FLOAT postscale[64]={
B0*B0, B0*B1, B0*B2, B0*B3, B0*B4, B0*B5, B0*B6, B0*B7,
B1*B0, B1*B1, B1*B2, B1*B3, B1*B4, B1*B5, B1*B6, B1*B7,
B2*B0, B2*B1, B2*B2, B2*B3, B2*B4, B2*B5, B2*B6, B2*B7,
B3*B0, B3*B1, B3*B2, B3*B3, B3*B4, B3*B5, B3*B6, B3*B7,
B4*B0, B4*B1, B4*B2, B4*B3, B4*B4, B4*B5, B4*B6, B4*B7,
B5*B0, B5*B1, B5*B2, B5*B3, B5*B4, B5*B5, B5*B6, B5*B7,
B6*B0, B6*B1, B6*B2, B6*B3, B6*B4, B6*B5, B6*B6, B6*B7,
B7*B0, B7*B1, B7*B2, B7*B3, B7*B4, B7*B5, B7*B6, B7*B7,
};

void ff_faandct(DCTELEM * data)
{
    FLOAT tmp0, tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7;
    FLOAT tmp10, tmp11, tmp12, tmp13;
    FLOAT z1, z2, z3, z4, z5, z11, z13;
    FLOAT temp[64];
    int i;

    emms();

    for (i=0; i<8*8; i+=8) {
        tmp0= data[0 + i] + data[7 + i];
        tmp7= data[0 + i] - data[7 + i];
        tmp1= data[1 + i] + data[6 + i];
        tmp6= data[1 + i] - data[6 + i];
        tmp2= data[2 + i] + data[5 + i];
        tmp5= data[2 + i] - data[5 + i];
        tmp3= data[3 + i] + data[4 + i];
        tmp4= data[3 + i] - data[4 + i];
        
        tmp10= tmp0 + tmp3;
        tmp13= tmp0 - tmp3;
        tmp11= tmp1 + tmp2;
        tmp12= tmp1 - tmp2;
        
        temp[0 + i]= tmp10 + tmp11;
        temp[4 + i]= tmp10 - tmp11;
        
        z1= (tmp12 + tmp13)*A1;
        temp[2 + i]= tmp13 + z1;
        temp[6 + i]= tmp13 - z1;
        
        tmp10= tmp4 + tmp5;
        tmp11= tmp5 + tmp6;
        tmp12= tmp6 + tmp7;

        z5= (tmp10 - tmp12) * A5;
        z2= tmp10*A2 + z5;
        z4= tmp12*A4 + z5;
        z3= tmp11*A1;

        z11= tmp7 + z3;
        z13= tmp7 - z3;

        temp[5 + i]= z13 + z2;
        temp[3 + i]= z13 - z2;
        temp[1 + i]= z11 + z4;
        temp[7 + i]= z11 - z4;
    }

    for (i=0; i<8; i++) {
        tmp0= temp[8*0 + i] + temp[8*7 + i];
        tmp7= temp[8*0 + i] - temp[8*7 + i];
        tmp1= temp[8*1 + i] + temp[8*6 + i];
        tmp6= temp[8*1 + i] - temp[8*6 + i];
        tmp2= temp[8*2 + i] + temp[8*5 + i];
        tmp5= temp[8*2 + i] - temp[8*5 + i];
        tmp3= temp[8*3 + i] + temp[8*4 + i];
        tmp4= temp[8*3 + i] - temp[8*4 + i];
        
        tmp10= tmp0 + tmp3;
        tmp13= tmp0 - tmp3;
        tmp11= tmp1 + tmp2;
        tmp12= tmp1 - tmp2;
        
        data[8*0 + i]= lrint(postscale[8*0 + i] * (tmp10 + tmp11));
        data[8*4 + i]= lrint(postscale[8*4 + i] * (tmp10 - tmp11));
        
        z1= (tmp12 + tmp13)* A1;
        data[8*2 + i]= lrint(postscale[8*2 + i] * (tmp13 + z1));
        data[8*6 + i]= lrint(postscale[8*6 + i] * (tmp13 - z1));
        
        tmp10= tmp4 + tmp5;
        tmp11= tmp5 + tmp6;
        tmp12= tmp6 + tmp7;

        z5= (tmp10 - tmp12) * A5;
        z2= tmp10*A2 + z5;
        z4= tmp12*A4 + z5;
        z3= tmp11*A1;

        z11= tmp7 + z3;
        z13= tmp7 - z3;

        data[8*5 + i]= lrint(postscale[8*5 + i] * (z13 + z2));
        data[8*3 + i]= lrint(postscale[8*3 + i] * (z13 - z2));
        data[8*1 + i]= lrint(postscale[8*1 + i] * (z11 + z4));
        data[8*7 + i]= lrint(postscale[8*7 + i] * (z11 - z4));
    }
}
