/*
 * LGPL
 */

/**
 * @file opts.c
 * options parser.
 * typical parsed command line:
 * msmpeg4:bitrate=720000:qmax=16
 *
 */

#include "avcodec.h"

extern const AVOption common_options[2];

const AVOption common_options[2] = {
    AVOPTION_CODEC_INT("common", "test", bit_rate, 0, 10, 0),
    AVOPTION_END()
};

static int parse_bool(const AVOption *c, char *s, int *var)
{
    int b = 1; /* by default -on- when present */
    if (s) {
	if (!strcasecmp(s, "off") || !strcasecmp(s, "false")
	    || !strcmp(s, "0"))
	    b = 0;
	else if (!strcasecmp(s, "on") || !strcasecmp(s, "true")
		 || !strcmp(s, "1"))
	    b = 1;
	else
	    return -1;
    }

    *var = b;
    return 0;
}

static int parse_double(const AVOption *c, char *s, double *var)
{
    double d;
    if (!s)
        return -1;
    d = atof(s);
    if (c->min != c->max) {
	if (d < c->min || d > c->max) {
	    fprintf(stderr, "Option: %s double value: %f out of range <%f, %f>\n",
		    c->name, d, c->min, c->max);
	    return -1;
	}
    }
    *var = d;
    return 0;
}

static int parse_int(const AVOption* c, char* s, int* var)
{
    int i;
    if (!s)
        return -1;
    i = atoi(s);
    if (c->min != c->max) {
	if (i < (int)c->min || i > (int)c->max) {
	    fprintf(stderr, "Option: %s integer value: %d out of range <%d, %d>\n",
		    c->name, i, (int)c->min, (int)c->max);
	    return -1;
	}
    }
    *var = i;
    return 0;
}

static int parse_string(const AVOption *c, char *s, AVCodecContext *avctx, char **var)
{
    if (!s)
	return -1;

    if (c->type == FF_OPT_TYPE_RCOVERRIDE) {
	int sf, ef, qs;
	float qf;
	if (sscanf(s, "%d,%d,%d,%f", &sf, &ef, &qs, &qf) == 4 && sf < ef) {
	    RcOverride *o;
	    avctx->rc_override = av_realloc(avctx->rc_override,
					    sizeof(RcOverride) * (avctx->rc_override_count + 1));
	    o = avctx->rc_override + avctx->rc_override_count++;
	    o->start_frame = sf;
	    o->end_frame = ef;
	    o->qscale = qs;
	    o->quality_factor = qf;

	    //printf("parsed Rc:  %d,%d,%d,%f  (%d)\n", sf,ef,qs,qf, avctx->rc_override_count);
	} else {
	    printf("incorrect/unparsable Rc: \"%s\"\n", s);
	}
    } else
	*var = av_strdup(s);
    return 0;
}

/**
 *
 * \param codec  codec for option parsing
 * \param opts   string with options for parsing
 * \param avctx  where to store parsed results
 */
int avcodec_parse(const AVCodec *codec, const char *opts, AVCodecContext *avctx)
{
    int r = 0;
    char* dopts = av_strdup(opts);
    if (dopts) {
        char *str = dopts;

	while (str && *str && r == 0) {
	    const AVOption *stack[FF_OPT_MAX_DEPTH];
	    int depth = 0;
	    const AVOption *c = codec->options;
	    char* e = strchr(str, ':');
	    char* p;
	    if (e)
		*e++ = 0;

	    p = strchr(str, '=');
	    if (p)
		*p++ = 0;

            // going through option structures
	    for (;;) {
		if (!c->name) {
		    if (c->sub) {
			stack[depth++] = c;
			c = c->sub;
			assert(depth > FF_OPT_MAX_DEPTH);
		    } else {
			if (depth == 0)
			    break; // finished
			c = stack[--depth];
                        c++;
		    }
		} else {
		    if (!strcmp(c->name, str)) {
			void* ptr = (char*)avctx + c->offset;

			switch (c->type & FF_OPT_TYPE_MASK) {
			case FF_OPT_TYPE_BOOL:
			    r = parse_bool(c, p, (int*)ptr);
			    break;
			case FF_OPT_TYPE_DOUBLE:
			    r = parse_double(c, p, (double*)ptr);
			    break;
			case FF_OPT_TYPE_INT:
			    r = parse_int(c, p, (int*)ptr);
			    break;
			case FF_OPT_TYPE_STRING:
			    r = parse_string(c, p, avctx, (char**)ptr);
			    break;
			default:
			    assert(0 == 1);
			}
		    }
		    c++;
		}
	    }
	    str = e;
	}
	av_free(dopts);
    }
    return r;
}
