$KCODE = 'e'
require "jcode"
require "kconv"
require 'suikyo/suikyo'

module Debug
  attr_accessor :debug_mode
  def debug_message (message, output = nil)
    if Debug::debug_mode?(output) || @debug_mode then
      puts "#{self.class}[#{self.id}]: #{message}"
    end
  end

  def Debug::message (message, output = nil)
    if Debug::debug_mode?(output) then
      puts message
    end
  end

  def Debug::debug_mode? (output = nil)
    return (ENV['RUBY_DEBUG'] || $DEBUG || output)
  end
end

class TextEdit
  include Debug

  attr_reader :cursor
  def initialize (text = "", cursor = 0)
    @debug_mode = false
    set (text, cursor)
  end

  def set (data, cursor = 0)
    @data   = data
    @cursor = cursor
  end

  def text
    return @data
  end

  def reset
    set ("", 0)
  end

  def insert (string)
    @data    = @data[0,@cursor] + string + @data[@cursor..-1]
    @cursor += string.length;
  end

  def delete
    if (@cursor < @data.length) then
      @data = @data[0,@cursor] + @data[(@cursor + 1)..-1]
    end
  end

  def backspace
    if (@cursor > 0) then
      @data = @data[0,(@cursor - 1)] + @data[@cursor..-1]
      @cursor -= 1
    end
  end

  def cursor_left
    if (@cursor > 0) then
      @cursor -= 1
    end
  end

  def cursor_right
    if (@cursor < @data.length) then
      @cursor += 1
    end
  end

  def cursor_beginning
    @cursor = 0
  end

  def cursor_end
    @cursor = @data.length
  end

  def segments
    # [text before the cursor, text on the cursor, text after the cursor]
    return [@data[0,@cursor], @data[@cursor,1], @data[(@cursor + 1)..-1]]
  end
end

class JTextEdit < TextEdit
  def initialize (text = "", cursor = 0)
    super
    @suikyo = Suikyo.new
    @suikyo.table.loadfile("romaji-kana")
  end

  def text
    if @cursor == @data.length then
      return @suikyo.convert (@data)
    else
      return super
    end
  end

  def segments
    if @cursor == @data.length then
      return [@suikyo.convert (@data), '', '']
    else
      return super
    end
  end
end

class Window 
  attr_reader :call_open, :call_close, :visible

  def initialize (parent)
    @parent = parent
    @visible = false
  end

  def open
    @call_open = true unless @visible
  end
  def open_end
    @call_open = false
    @visible = true
  end

  def close
    @call_close = true if @visible
  end
  def close_end
    @call_close = false
    @visible = false
  end
end

class Dialog < Window
  def text
    @parent.cand_list.join(" ")
  end
end

class Selection < Window
  attr_reader :call_right, :call_left, :call_line_beginning, :call_line_end,
    :call_up, :call_down

  def right
    @call_right = true
  end
  def right_end
    @call_right = false
  end

  def left
    @call_left = true
  end
  def left_end
    @call_left = false
  end

  def line_beginning
    @call_line_beginning = true
  end
  def line_beginning_end
    @call_line_beginning = false
  end

  def line_end
    @call_line_end = true
  end
  def line_end_end
    @call_line_end = false
  end

  def up
    @call_up = true
  end
  def up_end
    @call_up = false
  end

  def down
    @call_down = true
  end
  def down_end
    @call_down = false
  end
end

class KeyMap
  @@keysym = {
    :enter => 65293,
    :space => 32,
    :tab => 65289,
    :delete => 65535,
    :insert => 65379,
    :home => 65360,
    :end => 65367,
    :page_up => 65365,
    :page_down => 65366,
    :esc => 65307,
    :f1 => 65470,
    :f2 => 65471,
    :f3 => 65472,
    :f4 => 65473,
    :f5 => 65474,
    :f6 => 65475,
    :f7 => 65476,
    :f8 => 65477,
    :f9 => 65478,
    :f10 => 65479,
    :f11 => 65480,
    :f12 => 65481,
    :backspace => 65288,
    :muhenkan => 65314,
    :henkan => 65315,
    :hankaku_zenkaku => 65322,
    :katakana_hiragana => 65319,
    :up => 65362,
    :down => 65364,
    :right => 65363,
    :left => 65361,
    :ctrl_l => 65507,
    :ctrl_r => 65508,
    :alt_l => 65513,
    :alt_r => 65514,
    :shift_l => 65505,
    :shift_r => 65506,
  }

#   @@enter = [13, 65293, 0]
#   @@ctrl_a = [1,  97,  4];  @@alt_a = [97,  97,  8]
#   @@ctrl_b = [2,  98,  4];  @@alt_b = [98,  98,  8]
#   @@ctrl_c = [3,  99,  4];  @@alt_c = [99,  99,  8]
#   @@ctrl_d = [4,  100, 4];  @@alt_d = [100, 100, 8]
#   @@ctrl_e = [5,  101, 4];  @@alt_e = [101, 101, 8]
#   @@ctrl_f = [6,  102, 4];  @@alt_f = [102, 102, 8]
#   @@ctrl_g = [7,  103, 4];  @@alt_g = [103, 103, 8]
#   @@ctrl_h = [8,  104, 4];  @@alt_h = [104, 104, 8]
#   @@ctrl_i = [9,  105, 4];  @@alt_i = [105, 105, 8]
#   @@ctrl_j = [10, 106, 4];  @@alt_j = [106, 106, 8]
#   @@ctrl_k = [11, 107, 4];  @@alt_k = [107, 107, 8]
#   @@ctrl_l = [12, 108, 4];  @@alt_l = [108, 108, 8]
#   @@ctrl_m = [13, 109, 4];  @@alt_m = [109, 109, 8]
#   @@ctrl_n = [14, 110, 4];  @@alt_n = [110, 110, 8]
#   @@ctrl_o = [15, 111, 4];  @@alt_o = [111, 111, 8]
#   @@ctrl_p = [16, 112, 4];  @@alt_p = [112, 112, 8]
#   @@ctrl_q = [17, 113, 4];  @@alt_q = [113, 113, 8]
#   @@ctrl_r = [18, 114, 4];  @@alt_r = [114, 114, 8]
#   @@ctrl_s = [19, 115, 4];  @@alt_s = [115, 115, 8]
#   @@ctrl_t = [20, 116, 4];  @@alt_t = [116, 116, 8]
#   @@ctrl_u = [21, 117, 4];  @@alt_u = [117, 117, 8]
#   @@ctrl_v = [22, 118, 4];  @@alt_v = [118, 118, 8]
#   @@ctrl_w = [23, 119, 4];  @@alt_w = [119, 119, 8]
#   @@ctrl_x = [24, 120, 4];  @@alt_x = [120, 120, 8]
#   @@ctrl_y = [25, 121, 4];  @@alt_y = [121, 121, 8]
#   @@ctrl_z = [26, 122, 4];  @@alt_z = [122, 122, 8]

  @@modifier = { :shift => 1, :ctrl => 4, :alt => 8 }

  def initialize
    @keymap = Hash.new
  end

  def key (stroke)
    modifiers = 0
    if stroke.kind_of?(Array) then
      main_key = stroke[0]
      stroke[1..-1].each {|modifier|
	modifiers |=
	  (modifier.kind_of?(Symbol) ? @@modifier[modifier] : modifier)
      }
    else
      main_key = stroke
    end
    if main_key.kind_of?(Symbol) then
      main_key = @@keysym[main_key]
    end
    return [main_key, modifiers]
  end

  def ignore_shift (key)
    (main_key, modifiers) = key
    return [main_key, (modifiers & ~@@modifier[:shift])]
  end

  def add (stroke, command)
    @keymap[key(stroke)] = command
  end

  def del (stroke)
    @keymap.delete(key(stroke))
  end

  def command (stroke)
    return @keymap[key(stroke)] || @keymap[ignore_shift(key(stroke))]
  end
end


class ModeCore
  attr_accessor :trap
  attr_reader :label, :keymap
  def initialize (parent)
    @parent = parent
    @label  = ""
    @keymap = KeyMap.new
    initialize_keys
    @trap   = true
  end

  def on (prev_mode = nil)
  end

  def off (next_mode = nil)
  end

  def entries
    return []
  end

  def call (char, keysym, modifiers)
    command = @keymap.command([keysym, modifiers])
    if command then
      return send(command, keysym, modifiers)
    else
      return @trap
    end
  end
end

class ModeMaster
  attr_reader :current_name

  def initialize (parent, mode = :fund)
    @parent = parent
    @mode = Hash.new
    @current_name = nil
  end

  def set (name, mode)
    @mode[name] = mode
    unless @current then
      @current_name = name
    end
  end

  def current
    return @mode[@current_name]
  end

  def change (name)
    if @mode.key?(name) then
      @current_name = name
      current.on
      return true
    else
      return false
    end
  end

  def label (name = @current_name)
    if name then
      mode = @mode[name]
    else
      mode = current
    end

    if mode then
      return mode.label.toeuc
    else
      return "NOMODE"
    end
  end
end

class KanjiConvCore
  include Debug
  attr_reader :cand_list, :cand_index,
    :value_fixed, :call_fix, :mode

  def initialize
    @debug_mode = false

    @textEdit = TextEdit.new
    @mode = ModeMaster.new (self)
    clear
  end

  def reset
    clear
  end

  def clear
    @cand_list = []
    @cand_index = 0

    @call_fix = false
    @value_fixed = ""
  end

  ## inputEvent (keynum, keysym = nil, state = nil)
  ## ReturnValue 1:Pass 0:Trap
  def inputEvent (keynum, keysym = nil, state = nil)
    return 1
  end

  def fix (fixed_arg)
    # fixed_arg is a string or an index number of cand_list.
    @call_fix = true
    if fixed_arg.kind_of?(Integer) then
      word = @cand_list[fixed_arg]
    else
      word = fixed_arg
    end
    @value_fixed = word
  end
  def fix_end
    @call_fix = false
    @value_fixed = ""
  end

  def modeline
    return @mode.label
  end

  def set_cand_list (list, index = nil)
    if (list.is_a?(Array) && list.length > 0) then
      @cand_list  = list.map {|cand| cand.toeuc}
      @cand_index = index if index
      return true
    else
      reset_cand_list
      return false
    end
  end
  def set_cand_index (index)
    debug_message("<<set_cand_index>> #{index}")
    @cand_index = index
  end
  def reset_cand_list
    @cand_list  = []
    @cand_index = 0
  end

  def segment_length
    segments = (@mode.current.entries - [''])
    return segments.length
  end
  def segment_word (n)
    segments = (@mode.current.entries - [''])
    return segments[n]
  end
  def segment_status (n)
    segments = @mode.current.entries
    offset = 0
    if segments[0] == "" then
      offset += 1
    end
    return ((n + offset) % 3 == 1) ? :highlight : :normal
  end
end
