/*****************************************************************************/
/*  charset-conv.c - contains functions for performing conversions between   */
/*     character sets.                                                       */
/*  Copyright (C) 1998-2008 Brian Masney <masneyb@gftp.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "gftp.h"
#include <langinfo.h>
static const char cvsid[] = "$Id: protocols.c 952 2008-01-24 23:31:26Z masneyb $";

static /*@null@*/ char *
_gftp_get_next_charset (char **curpos)
{
  char *ret, *endpos;
  size_t len, retlen;

  if (**curpos == '\0')
    return (NULL);

  for (; **curpos == ' ' || **curpos == '\t'; (*curpos)++);

  if ((endpos = strchr (*curpos, ',')) == NULL)
    len = strlen (*curpos) - 1; /* the trailing ',' should be omitted */
  else
    len = endpos - *curpos;

  for (retlen = len - 1;
       (*curpos)[retlen - 1] == ' ' || (*curpos)[retlen - 1] == '\t';
       retlen--);

  retlen++; /* Needed due to the len - 1 above... */
  ret = g_malloc0 (retlen + 1);
  memcpy (ret, *curpos, retlen);

  for (*curpos += len; **curpos == ','; (*curpos)++);

  return (ret);
}


static void
_do_show_iconv_error (const char *str, char *charset, int from_utf8,
                      GError * error)
{
  const char *fromset, *toset;

  if (from_utf8)
    {
      fromset = "UTF-8";
      toset = charset;
    }
  else
    {
      fromset = charset;
      toset = "UTF-8";
    }

  printf (_("Error converting string '%s' from character set %s to character set %s: %s\n"),
          str, fromset, toset, error->message);
}


char *
gftp_string_to_utf8 (gftp_request *request, const char *str, size_t *dest_len)
{
    gchar *remote_charset = NULL, *default_charset = NULL, *ret;
    GError *error = NULL;

    gftp_lookup_global_option ("default_charset", &default_charset);
    if(!default_charset)
        default_charset = "UTF-8";

    if(request)
        remote_charset = request->remote_charset;
    if(!remote_charset)
        remote_charset = default_charset;

    ret = g_convert_with_fallback(str, -1, "UTF-8", remote_charset, "?", NULL, dest_len, &error);

    return ret;
}


char *
gftp_string_from_utf8 (gftp_request *request, int force_local, const char *str,
                       size_t *dest_len)
{
    gchar *remote_charset = NULL, *default_charset = NULL, *ret;
    GError *error = NULL;

    gftp_lookup_global_option ("default_charset", &default_charset);
    if(!default_charset)
        default_charset = "UTF-8";

    if(request)
        remote_charset = request->remote_charset;
    if(!remote_charset)
        remote_charset = default_charset;

    ret = g_convert_with_fallback(str, -1, remote_charset, "UTF-8", "?", NULL, dest_len, &error);

    return ret;
}


char *
gftp_filename_to_utf8 (gftp_request *request, const char *str,
                       size_t *dest_len)
{
    char *codeset = nl_langinfo(CODESET);
    gchar *ret;
    GError *error = NULL;

    ret = g_filename_to_utf8 (str, -1, NULL, dest_len, &error);
    if(!ret) {
        error = NULL;
        ret = g_convert_with_fallback(str, -1, "UTF-8", codeset, "?", NULL, dest_len, &error);
    }

    if(!ret) {
        error = NULL;
        ret = g_strdup(str);
    }

    return ret;
}


char *
gftp_filename_from_utf8 (gftp_request *request, const char *str,
                         size_t *dest_len)
{
    char *codeset = nl_langinfo(CODESET);
    gchar *ret;
    GError *error = NULL;

    ret = g_filename_from_utf8 (str, -1, NULL, dest_len, &error);
    if(!ret) {
        error = NULL;
        ret = g_convert_with_fallback(str, -1, codeset, "UTF-8", "?", NULL, dest_len, &error);
    }

    return ret;
}


char *
gftp_remote_filename_to_utf8 (gftp_request *request, const char *str,
                       size_t *dest_len)
{
    char *remote_charset = NULL, *default_charset = NULL;
    gchar *ret;
    GError *error = NULL;

    if(request == NULL)
        return (NULL);

    /* get remote_charset */
    remote_charset = request->remote_charset;
    gftp_lookup_global_option ("default_charset", &default_charset);
    if(!remote_charset)
       remote_charset = (default_charset && *default_charset != '\0')
           ? default_charset : "UTF8";

    ret = g_convert(str, -1, "UTF8", remote_charset, NULL, dest_len, &error);
    return ret;
}


char *
gftp_remote_filename_from_utf8 (gftp_request *request, const char *str,
                         size_t *dest_len)
{
    char *remote_charset = NULL, *default_charset = NULL;
    gchar *ret;
    GError *error = NULL;

    if(request == NULL)
        return (NULL);

    if(!g_utf8_validate (str, -1, NULL))
        return (NULL);

    /* get remote_charset */
    remote_charset = request->remote_charset;
    gftp_lookup_global_option ("default_charset", &default_charset);
    if(!remote_charset)
        remote_charset = (default_charset && *default_charset != '\0')
            ? default_charset : "UTF8";

    ret = g_convert(str, -1, remote_charset, "UTF8", NULL, dest_len, &error);
    return ret;
}
